<?php

namespace Modules\Exchange\Http\Controllers;

use App\Account;
use App\Product;
use App\TaxRate;
use App\Business;
use Carbon\Carbon;
use App\Transaction;
use App\BusinessLocation;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\TransactionPayment;
use App\Utils\BusinessUtil;
use App\TransactionSellLine;
use Illuminate\Http\Request;
use App\Utils\TransactionUtil;
use App\User;
use App\CashRegister;
use App\CashRegisterTransaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;
use Modules\Exchange\Entities\TransactionExchange;
use Modules\Exchange\Entities\TransactionExchangeLine;
use Illuminate\Routing\Controller;
use App\Utils\CashRegisterUtil;

class ExchangeController extends Controller
{
    protected $businessUtil;
    protected $transactionUtil;
    protected $productUtil;
    protected $cashRegisterUtil;

    public function __construct(BusinessUtil $businessUtil, TransactionUtil $transactionUtil, ProductUtil $pUtil, CashRegisterUtil $cashRegisterUtil)
    {
        $this->businessUtil = $businessUtil;
        $this->transactionUtil = $transactionUtil;
        $this->productUtil = $pUtil;
        $this->cashRegisterUtil = $cashRegisterUtil;
    }

    /**
     * Display exchange index page
     */
    public function index()
    {
        if (!auth()->user()->can('exchange.access')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business_locations = BusinessLocation::forDropdown($business_id, false, false);
        $users = TransactionExchange::getUsersForDropdown($business_id);

        return view('exchange::index', compact('business_locations', 'users'));
    }

    /**
     * Show exchange creation form
     */
    public function create()
    {
        if (!auth()->user()->can('exchange.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business_details = $this->businessUtil->getDetails($business_id);
        $business_locations = BusinessLocation::forDropdown($business_id, false, false);
        $default_location = null;

        if (count($business_locations) == 1) {
            foreach ($business_locations as $id => $name) {
                $default_location = BusinessLocation::findOrFail($id);
            }
        }

        $payment_types = $this->transactionUtil->payment_types(null, true, $business_id);
        $accounts = [];
        if ($this->businessUtil->isModuleEnabled('account')) {
            $accounts = Account::forDropdown($business_id, true, false, true);
        }

        return view('exchange::create', compact(
            'business_details',
            'business_locations',
            'default_location',
            'payment_types',
            'accounts'
        ));
    }

    /**
     * Search for original transaction by invoice number
     */
    public function searchTransaction(Request $request)
    {
        if (!auth()->user()->can('exchange.access')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $invoice_no = $request->get('invoice_no');

        $transaction = Transaction::with([
            'sell_lines' => function ($query) {
                $query->where('quantity_returned', '<', DB::raw('quantity'));
            },
            'sell_lines.product',
            'sell_lines.variations',
            'sell_lines.variations.product_variation',
            'sell_lines.product.unit',
            'contact'
        ])
            ->where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->where('invoice_no', $invoice_no)
            ->first();

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => __('exchange::lang.transaction_not_found')
            ]);
        }

        $exchangeable_lines = $transaction->sell_lines->filter(function ($line) {
            return ($line->quantity - $line->quantity_returned) > 0;
        });

        if ($exchangeable_lines->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => __('exchange::lang.no_items_available_for_exchange')
            ]);
        }

        return response()->json([
            'success' => true,
            'transaction' => $transaction,
            'exchangeable_lines' => $exchangeable_lines->values()
        ]);
    }

    /**
     * Process the exchange - FIXED VERSION WITH PROPER STOCK AND CASH REGISTER HANDLING
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('exchange.create')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'original_transaction_id' => 'required|exists:transactions,id',
            'location_id' => 'required|exists:business_locations,id',
            'exchange_lines' => 'required|array|min:1',
            'exchange_lines.*.original_sell_line_id' => 'required|exists:transaction_sell_lines,id',
            'exchange_lines.*.original_quantity' => 'required|numeric|min:0.01',
            'payment_method' => 'sometimes|string',
        ]);

        try {
            DB::beginTransaction();

            $business_id = request()->session()->get('user.business_id');
            $user_id = request()->session()->get('user.id');
            $original_transaction = Transaction::findOrFail($request->original_transaction_id);
            $exchange_ref_no = TransactionExchange::generateExchangeRefNo($business_id);
            $now = new \DateTime();
            $timestamp_db = $now->format('Y-m-d H:i:s');

            \Log::info('Exchange process started', [
                'timestamp' => $timestamp_db,
                'exchange_ref' => $exchange_ref_no
            ]);

            // Calculate amounts and prepare data
            $original_amount = 0;
            $new_amount = 0;
            $exchange_lines_data = [];
            $new_sell_lines = [];
            $stock_adjustments = []; // Track stock changes for reversal

            foreach ($request->exchange_lines as $line_data) {
                $original_sell_line = TransactionSellLine::findOrFail($line_data['original_sell_line_id']);

                // Validate quantity
                $available_qty = $original_sell_line->quantity - $original_sell_line->quantity_returned;
                if ($line_data['original_quantity'] > $available_qty) {
                    throw new \Exception("Insufficient quantity available for exchange");
                }

                $line_original_amount = round($line_data['original_quantity'] * $original_sell_line->unit_price_inc_tax, 4);
                $line_new_amount = 0;

                // FIXED: Track stock adjustment for returned item
                $stock_adjustments[] = [
                    'type' => 'return',
                    'location_id' => $request->location_id,
                    'product_id' => $original_sell_line->product_id,
                    'variation_id' => $original_sell_line->variation_id,
                    'quantity' => $line_data['original_quantity']
                ];

                // If new item is selected for exchange
                if (!empty($line_data['new_product_id']) && !empty($line_data['new_variation_id'])) {
                    $new_quantity = $line_data['new_quantity'] ?? $line_data['original_quantity'];
                    $new_unit_price = $line_data['new_unit_price'];
                    $line_new_amount = round($new_quantity * $new_unit_price, 4);

                    // FIXED: Track stock adjustment for new item
                    $stock_adjustments[] = [
                        'type' => 'sell',
                        'location_id' => $request->location_id,
                        'product_id' => $line_data['new_product_id'],
                        'variation_id' => $line_data['new_variation_id'],
                        'quantity' => $new_quantity
                    ];

                    $product = Product::find($line_data['new_product_id']);
                    $unit_id = $product ? $product->unit_id : null;

                    $new_sell_lines[] = [
                        'product_id' => $line_data['new_product_id'],
                        'variation_id' => $line_data['new_variation_id'],
                        'quantity' => $new_quantity,
                        'unit_price' => $new_unit_price,
                        'unit_price_inc_tax' => $new_unit_price,
                        'unit_price_before_discount' => $new_unit_price,
                        'line_discount_type' => 'fixed',
                        'line_discount_amount' => 0.0000,
                        'sell_line_note' => 'Exchange item',
                        'sub_unit_id' => $unit_id,
                        'exchange_parent_line_id' => $original_sell_line->id,
                        'is_exchange_return' => 0,
                        'item_tax' => 0.0000,
                        'secondary_unit_quantity' => 0.0000,
                        'quantity_returned' => 0.0000,
                        'mfg_waste_percent' => 0.0000,
                        'so_quantity_invoiced' => 0.0000,
                    ];
                }

                $original_amount += $line_original_amount;
                $new_amount += $line_new_amount;
                $price_difference = round($line_new_amount - $line_original_amount, 4);

                $exchange_lines_data[] = [
                    'original_sell_line_id' => $original_sell_line->id,
                    'new_sell_line_id' => null,
                    'exchange_type' => !empty($line_data['new_product_id']) ? 'exchange_with_new' : 'return_only',
                    'original_quantity' => round($line_data['original_quantity'], 4),
                    'original_unit_price' => round($original_sell_line->unit_price_inc_tax, 4),
                    'new_quantity' => round($line_data['new_quantity'] ?? 0, 4),
                    'new_unit_price' => round($line_data['new_unit_price'] ?? 0, 4),
                    'price_difference' => $price_difference,
                ];

                // Update original sell line quantity returned
                $original_sell_line->quantity_returned += $line_data['original_quantity'];
                $original_sell_line->save();
            }

            $exchange_difference = $new_amount - $original_amount;

            // FIXED: Apply stock adjustments
            foreach ($stock_adjustments as $adjustment) {
                if ($adjustment['type'] == 'return') {
                    // Add stock back for returned items
                    $this->productUtil->updateProductQuantity(
                        $adjustment['location_id'],
                        $adjustment['product_id'],
                        $adjustment['variation_id'],
                        $adjustment['quantity'], // Positive for return
                        0,
                        null,
                        false
                    );
                } elseif ($adjustment['type'] == 'sell') {
                    // Reduce stock for new items
                    $this->productUtil->updateProductQuantity(
                        $adjustment['location_id'],
                        $adjustment['product_id'],
                        $adjustment['variation_id'],
                        -$adjustment['quantity'], // Negative for sale
                        0,
                        null,
                        false
                    );
                }
            }

            // Create exchange transaction if there are new items
            $exchange_transaction = null;
            if (!empty($new_sell_lines)) {
                \Log::info('Creating exchange transaction');

                $exchange_transaction_data = [
                    'business_id' => $business_id,
                    'location_id' => $request->location_id,
                    'type' => 'sell',
                    'status' => 'final',
                    'contact_id' => $original_transaction->contact_id,
                    'transaction_date' => $timestamp_db,
                    'invoice_no' => $this->transactionUtil->getInvoiceNumber($business_id, 'final', $request->location_id),
                    'is_exchange' => 1,
                    'exchange_parent_id' => $original_transaction->id,
                    'created_by' => $user_id,
                    'total_before_tax' => $new_amount,
                    'tax_amount' => 0,
                    'final_total' => $new_amount,
                ];

                $exchange_transaction = Transaction::create($exchange_transaction_data);

                // Create new sell lines
                foreach ($new_sell_lines as $index => $new_sell_line_data) {
                    $new_sell_line_data['transaction_id'] = $exchange_transaction->id;
                    $new_sell_line = TransactionSellLine::create($new_sell_line_data);

                    if (isset($exchange_lines_data[$index])) {
                        $exchange_lines_data[$index]['new_sell_line_id'] = $new_sell_line->id;
                    }
                }
            }

            // Create exchange record
            $exchange_data = [
                'business_id' => $business_id,
                'location_id' => $request->location_id,
                'original_transaction_id' => $original_transaction->id,
                'exchange_transaction_id' => $exchange_transaction ? $exchange_transaction->id : null,
                'exchange_ref_no' => $exchange_ref_no,
                'exchange_date' => $timestamp_db,
                'original_amount' => $original_amount,
                'new_amount' => $new_amount,
                'exchange_difference' => $exchange_difference,
                'payment_received' => $exchange_difference > 0 ? $exchange_difference : 0,
                'refund_given' => $exchange_difference < 0 ? abs($exchange_difference) : 0,
                'total_exchange_amount' => $exchange_difference,
                'status' => 'completed',
                'created_by' => $user_id,
                'notes' => $request->notes ?? '',
                'stock_adjustments' => json_encode($stock_adjustments), // Store for reversal
            ];

            $exchange = TransactionExchange::create($exchange_data);

            // Create exchange lines
            foreach ($exchange_lines_data as $line_data) {
                $line_data['exchange_id'] = $exchange->id;
                TransactionExchangeLine::create($line_data);
            }

            // FIXED: Handle payments with proper cash register integration
            if ($exchange_transaction) {
                $this->handleExchangePayments(
                    $exchange_transaction,
                    $exchange_difference,
                    $original_amount,
                    $new_amount,
                    $request,
                    $business_id,
                    $user_id
                );
            }

            DB::commit();

            $invoice_url = null;
            if ($exchange_transaction) {
                $invoice_url = action([\App\Http\Controllers\SellPosController::class, 'printInvoice'], [$exchange_transaction->id]);
            }

            \Log::info('Exchange completed successfully', [
                'exchange_id' => $exchange->id,
                'exchange_ref' => $exchange_ref_no
            ]);

            return response()->json([
                'success' => true,
                'message' => __('exchange::lang.exchange_completed_successfully'),
                'exchange_id' => $exchange->id,
                'invoice_url' => $invoice_url
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            \Log::emergency("Exchange Error - File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * FIXED: Handle exchange payments with proper cash register integration
     * This method handles all payment scenarios for exchanges and ensures proper cash register entries
     */
    private function handleExchangePayments($exchange_transaction, $exchange_difference, $original_amount, $new_amount, $request, $business_id, $user_id)
    {
        try {
            $payment_method = $request->payment_method ?? 'cash';
            $current_date = date('Y-m-d H:i:s');

            \Log::info('Processing exchange payments', [
                'exchange_difference' => $exchange_difference,
                'original_amount' => $original_amount,
                'new_amount' => $new_amount,
                'payment_method' => $payment_method
            ]);

            if ($exchange_difference > 0) {
                // SCENARIO 1: Customer pays extra money (new items cost more than returned items)
                // Example: Return ৳200 item + Buy ৳400 items = Customer pays ৳200 extra

                \Log::info('Customer pays extra: ' . $exchange_difference);

                // 1. Record the full sale amount (new items)
                $sale_payment = new \App\TransactionPayment();
                $sale_payment->transaction_id = $exchange_transaction->id;
                $sale_payment->business_id = $business_id;
                $sale_payment->amount = $new_amount; // Full amount of new items
                $sale_payment->method = $payment_method;
                $sale_payment->paid_on = $current_date;
                $sale_payment->created_by = $user_id;
                $sale_payment->is_return = 0;
                $sale_payment->paid_through_link = 0;
                $sale_payment->is_advance = 0;
                $sale_payment->payment_for = $exchange_transaction->contact_id ?? 1;
                $sale_payment->note = 'Exchange sale payment';

                // Add optional payment details
                $this->addPaymentDetails($sale_payment, $request, $payment_method);

                // Generate payment reference
                $ref_count = $this->transactionUtil->setAndGetReferenceCount('sell_payment', $business_id);
                $sale_payment->payment_ref_no = $this->transactionUtil->generateReferenceNumber('sell_payment', $ref_count, $business_id);
                $sale_payment->save();

                // 2. Record the refund for returned items (negative amount)
                $refund_payment = new \App\TransactionPayment();
                $refund_payment->transaction_id = $exchange_transaction->id;
                $refund_payment->business_id = $business_id;
                $refund_payment->amount = -$original_amount; // Negative for refund
                $refund_payment->method = $payment_method;
                $refund_payment->paid_on = $current_date;
                $refund_payment->created_by = $user_id;
                $refund_payment->is_return = 1; // Mark as refund
                $refund_payment->paid_through_link = 0;
                $refund_payment->is_advance = 0;
                $refund_payment->payment_for = $exchange_transaction->contact_id ?? 1;
                $refund_payment->note = 'Exchange refund for returned items';

                $ref_count_refund = $this->transactionUtil->setAndGetReferenceCount('sell_payment', $business_id);
                $refund_payment->payment_ref_no = $this->transactionUtil->generateReferenceNumber('sell_payment', $ref_count_refund, $business_id);
                $refund_payment->save();

                // 3. Cash register entries
                if ($payment_method === 'cash') {
                    // Record sale (credit - money coming in)
                    $this->addToCashRegister($exchange_transaction, $new_amount, 'credit', $payment_method, 'sale');

                    // Record refund (debit - money going out)
                    $this->addToCashRegister($exchange_transaction, $original_amount, 'debit', $payment_method, 'refund');
                }
            } elseif ($exchange_difference < 0) {
                // SCENARIO 2: Customer gets refund (returned items cost more than new items)
                // Example: Return ৳200 item + Buy ৳125 item = Customer gets ৳75 refund

                $refund_amount = abs($exchange_difference);
                \Log::info('Customer gets refund: ' . $refund_amount);

                // 1. Record the sale for new items
                $sale_payment = new \App\TransactionPayment();
                $sale_payment->transaction_id = $exchange_transaction->id;
                $sale_payment->business_id = $business_id;
                $sale_payment->amount = $new_amount; // Amount of new items
                $sale_payment->method = $payment_method;
                $sale_payment->paid_on = $current_date;
                $sale_payment->created_by = $user_id;
                $sale_payment->is_return = 0;
                $sale_payment->paid_through_link = 0;
                $sale_payment->is_advance = 0;
                $sale_payment->payment_for = $exchange_transaction->contact_id ?? 1;
                $sale_payment->note = 'Exchange sale payment';

                $ref_count_sale = $this->transactionUtil->setAndGetReferenceCount('sell_payment', $business_id);
                $sale_payment->payment_ref_no = $this->transactionUtil->generateReferenceNumber('sell_payment', $ref_count_sale, $business_id);
                $sale_payment->save();

                // 2. Record the refund for returned items
                $refund_payment = new \App\TransactionPayment();
                $refund_payment->transaction_id = $exchange_transaction->id;
                $refund_payment->business_id = $business_id;
                $refund_payment->amount = -$original_amount; // Negative for refund
                $refund_payment->method = $payment_method;
                $refund_payment->paid_on = $current_date;
                $refund_payment->created_by = $user_id;
                $refund_payment->is_return = 1;
                $refund_payment->paid_through_link = 0;
                $refund_payment->is_advance = 0;
                $refund_payment->payment_for = $exchange_transaction->contact_id ?? 1;
                $refund_payment->note = 'Exchange refund for returned items';

                $ref_count_refund = $this->transactionUtil->setAndGetReferenceCount('sell_payment', $business_id);
                $refund_payment->payment_ref_no = $this->transactionUtil->generateReferenceNumber('sell_payment', $ref_count_refund, $business_id);
                $refund_payment->save();

                // 3. Cash register entries
                if ($payment_method === 'cash') {
                    // Record sale (credit - money coming in)
                    $this->addToCashRegister($exchange_transaction, $new_amount, 'credit', $payment_method, 'sale');

                    // Record refund (debit - money going out)
                    $this->addToCashRegister($exchange_transaction, $original_amount, 'debit', $payment_method, 'refund');
                }
            } else {
                // SCENARIO 3: Even exchange (same value)
                // Example: Return ৳200 item + Buy ৳200 item = No money exchange

                \Log::info('Even exchange - no cash difference');

                // 1. Record the sale for new items
                $sale_payment = new \App\TransactionPayment();
                $sale_payment->transaction_id = $exchange_transaction->id;
                $sale_payment->business_id = $business_id;
                $sale_payment->amount = $new_amount;
                $sale_payment->method = $payment_method;
                $sale_payment->paid_on = $current_date;
                $sale_payment->created_by = $user_id;
                $sale_payment->is_return = 0;
                $sale_payment->paid_through_link = 0;
                $sale_payment->is_advance = 0;
                $sale_payment->payment_for = $exchange_transaction->contact_id ?? 1;
                $sale_payment->note = 'Exchange sale payment';

                $ref_count_sale = $this->transactionUtil->setAndGetReferenceCount('sell_payment', $business_id);
                $sale_payment->payment_ref_no = $this->transactionUtil->generateReferenceNumber('sell_payment', $ref_count_sale, $business_id);
                $sale_payment->save();

                // 2. Record the refund for returned items
                $refund_payment = new \App\TransactionPayment();
                $refund_payment->transaction_id = $exchange_transaction->id;
                $refund_payment->business_id = $business_id;
                $refund_payment->amount = -$original_amount;
                $refund_payment->method = $payment_method;
                $refund_payment->paid_on = $current_date;
                $refund_payment->created_by = $user_id;
                $refund_payment->is_return = 1;
                $refund_payment->paid_through_link = 0;
                $refund_payment->is_advance = 0;
                $refund_payment->payment_for = $exchange_transaction->contact_id ?? 1;
                $refund_payment->note = 'Exchange refund for returned items';

                $ref_count_refund = $this->transactionUtil->setAndGetReferenceCount('sell_payment', $business_id);
                $refund_payment->payment_ref_no = $this->transactionUtil->generateReferenceNumber('sell_payment', $ref_count_refund, $business_id);
                $refund_payment->save();

                // 3. Cash register entries (even though no net cash change)
                if ($payment_method === 'cash') {
                    // Record sale (credit)
                    $this->addToCashRegister($exchange_transaction, $new_amount, 'credit', $payment_method, 'sale');

                    // Record refund (debit)
                    $this->addToCashRegister($exchange_transaction, $original_amount, 'debit', $payment_method, 'refund');
                }
            }

            // Update payment status
            $exchange_transaction->payment_status = 'paid';
            $exchange_transaction->save();

            \Log::info('Exchange payments processed successfully', [
                'transaction_id' => $exchange_transaction->id,
                'payment_method' => $payment_method
            ]);
        } catch (\Exception $e) {
            \Log::error('Exchange payment error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    /**
     * Add payment details to payment record
     */
    private function addPaymentDetails($payment, $request, $payment_method)
    {
        if ($request->payment_account) {
            $payment->account_id = $request->payment_account;
        }
        if ($request->card_number) {
            $payment->card_number = $request->card_number;
        }
        if ($request->card_holder_name) {
            $payment->card_holder_name = $request->card_holder_name;
        }
        if ($payment_method === 'card') {
            $payment->card_type = $request->card_type ?? 'credit';
        }
        if ($request->cheque_number) {
            $payment->cheque_number = $request->cheque_number;
        }
        if ($request->bank_name) {
            $payment->bank_account_number = $request->bank_name;
        }
        if ($request->transaction_no) {
            $payment->transaction_no = $request->transaction_no;
        }
    }

    /**
     * FIXED: Add proper cash register entries with correct operation types
     */
    private function addToCashRegister($transaction, $amount, $type, $payment_method, $operation_type = 'exchange')
    {
        try {
            // Get the current open cash register
            $cash_register = \App\CashRegister::where('user_id', auth()->user()->id)
                ->where('status', 'open')
                ->first();

            if ($cash_register) {
                $cash_register_transaction = new \App\CashRegisterTransaction();
                $cash_register_transaction->cash_register_id = $cash_register->id;
                $cash_register_transaction->transaction_id = $transaction->id;
                $cash_register_transaction->type = $type; // 'credit' or 'debit'
                $cash_register_transaction->amount = $amount;
                $cash_register_transaction->pay_method = $payment_method;

                // Set transaction_type based on operation for cash register reporting
                if ($operation_type == 'sale') {
                    $cash_register_transaction->transaction_type = 'sell'; // This will show in sales
                } elseif ($operation_type == 'refund') {
                    $cash_register_transaction->transaction_type = 'refund'; // This will show in refunds
                } else {
                    $cash_register_transaction->transaction_type = 'exchange';
                }

                $cash_register_transaction->created_at = now();
                $cash_register_transaction->updated_at = now();
                $cash_register_transaction->save();

                \Log::info('Added to cash register', [
                    'register_id' => $cash_register->id,
                    'transaction_id' => $transaction->id,
                    'amount' => $amount,
                    'type' => $type,
                    'method' => $payment_method,
                    'operation' => $operation_type,
                    'transaction_type' => $cash_register_transaction->transaction_type
                ]);
            } else {
                \Log::warning('No open cash register found for user: ' . auth()->user()->id);
            }
        } catch (\Exception $e) {
            \Log::error('Cash register error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
        }
    }

    /**
     * FIXED: Cancel exchange with proper stock reversal
     */
    public function cancel($id)
    {
        if (!auth()->user()->can('exchange.cancel')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $business_id = request()->session()->get('user.business_id');
            $exchange = TransactionExchange::with(['exchangeLines'])->where('business_id', $business_id)->findOrFail($id);

            if ($exchange->status == 'cancelled') {
                return response()->json([
                    'success' => false,
                    'message' => __('exchange::lang.exchange_already_cancelled')
                ]);
            }

            // FIXED: Reverse stock adjustments
            if ($exchange->stock_adjustments) {
                $stock_adjustments = json_decode($exchange->stock_adjustments, true);
                foreach ($stock_adjustments as $adjustment) {
                    if ($adjustment['type'] == 'return') {
                        // Reverse: Remove stock that was added back
                        $this->productUtil->updateProductQuantity(
                            $adjustment['location_id'],
                            $adjustment['product_id'],
                            $adjustment['variation_id'],
                            -$adjustment['quantity'], // Reverse the return
                            0,
                            null,
                            false
                        );
                    } elseif ($adjustment['type'] == 'sell') {
                        // Reverse: Add stock back that was sold
                        $this->productUtil->updateProductQuantity(
                            $adjustment['location_id'],
                            $adjustment['product_id'],
                            $adjustment['variation_id'],
                            $adjustment['quantity'], // Reverse the sale
                            0,
                            null,
                            false
                        );
                    }
                }
            }

            // Restore original sell line quantities
            foreach ($exchange->exchangeLines as $exchange_line) {
                $original_sell_line = TransactionSellLine::find($exchange_line->original_sell_line_id);
                if ($original_sell_line) {
                    $original_sell_line->quantity_returned -= $exchange_line->original_quantity;
                    $original_sell_line->save();
                }
            }

            // Remove cash register entries
            if ($exchange->exchange_transaction_id) {
                \App\CashRegisterTransaction::where('transaction_id', $exchange->exchange_transaction_id)->delete();
            }

            // Update exchange status
            $exchange->status = 'cancelled';
            $exchange->cancelled_at = now();
            $exchange->cancelled_by = auth()->user()->id;
            $exchange->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => __('exchange::lang.exchange_cancelled_successfully')
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            \Log::emergency("Exchange cancel error: " . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * FIXED: Delete exchange with proper cleanup
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('exchange.delete')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $business_id = request()->session()->get('user.business_id');
            $exchange = TransactionExchange::where('business_id', $business_id)->findOrFail($id);

            // Only allow deletion of cancelled exchanges
            if ($exchange->status !== 'cancelled') {
                return response()->json([
                    'success' => false,
                    'message' => 'Can only delete cancelled exchanges. Please cancel the exchange first.'
                ]);
            }

            // Delete exchange lines
            TransactionExchangeLine::where('exchange_id', $exchange->id)->delete();

            // Clean up exchange transaction if exists
            if ($exchange->exchange_transaction_id) {
                $exchange_transaction = Transaction::find($exchange->exchange_transaction_id);
                if ($exchange_transaction) {
                    // Remove cash register entries
                    \App\CashRegisterTransaction::where('transaction_id', $exchange->exchange_transaction_id)->delete();

                    // Delete payments
                    \App\TransactionPayment::where('transaction_id', $exchange->exchange_transaction_id)->delete();

                    // Delete sell lines
                    TransactionSellLine::where('transaction_id', $exchange->exchange_transaction_id)->delete();

                    // Delete transaction
                    $exchange_transaction->delete();
                }
            }

            // Delete the exchange record
            $exchange->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Exchange deleted successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            \Log::emergency("Exchange delete error: " . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get exchange list via DataTables
     */
    public function getExchanges()
    {
        if (!auth()->user()->can('exchange.access')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $exchanges = TransactionExchange::leftJoin('business_locations as bl', 'transaction_exchanges.location_id', '=', 'bl.id')
            ->leftJoin('transactions as ot', 'transaction_exchanges.original_transaction_id', '=', 'ot.id')
            ->leftJoin('transactions as et', 'transaction_exchanges.exchange_transaction_id', '=', 'et.id')
            ->leftJoin('contacts as c', 'ot.contact_id', '=', 'c.id')
            ->leftJoin('users as u', 'transaction_exchanges.created_by', '=', 'u.id')
            ->where('transaction_exchanges.business_id', $business_id);

        // Apply filters
        if (request()->has('location_id') && request()->location_id != '') {
            $exchanges->where('transaction_exchanges.location_id', request()->location_id);
        }

        if (request()->has('customer_id') && request()->customer_id != '') {
            $exchanges->where('ot.contact_id', request()->customer_id);
        }

        if (request()->has('status') && request()->status != '') {
            $exchanges->where('transaction_exchanges.status', request()->status);
        }

        if (request()->has('date_range') && request()->date_range != '') {
            $date_range = explode(' - ', request()->date_range);
            if (count($date_range) == 2) {
                $exchanges->whereBetween('transaction_exchanges.exchange_date', [
                    \Carbon\Carbon::createFromFormat('m/d/Y', $date_range[0])->format('Y-m-d'),
                    \Carbon\Carbon::createFromFormat('m/d/Y', $date_range[1])->format('Y-m-d')
                ]);
            }
        }

        if (request()->has('created_by') && request()->created_by != '') {
            $exchanges->where('transaction_exchanges.created_by', request()->created_by);
        }

        $exchanges = $exchanges->select([
            'transaction_exchanges.id',
            'transaction_exchanges.exchange_ref_no',
            'transaction_exchanges.exchange_date',
            'transaction_exchanges.total_exchange_amount',
            'transaction_exchanges.status',
            'transaction_exchanges.exchange_transaction_id',
            'transaction_exchanges.original_transaction_id',
            'transaction_exchanges.cancelled_at',
            'bl.name as location_name',
            'ot.invoice_no as original_invoice',
            'et.invoice_no as exchange_invoice',
            'c.name as customer_name',
            DB::raw("CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, '')) as created_by_name"),
        ]);

        return DataTables::of($exchanges)
            ->addColumn('action', function ($row) {
                $is_cancelled = (!is_null($row->cancelled_at) || $row->status == 'cancelled');

                $html = '<div class="btn-group">';
                $html .= '<button type="button" class="btn btn-xs ' . ($is_cancelled ? 'btn-default' : 'btn-info') . ' dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">';
                $html .= __('messages.actions') . ' <span class="caret"></span>';
                $html .= '</button>';
                $html .= '<ul class="dropdown-menu dropdown-menu-right">';

                // View exchange
                $html .= '<li><a href="#" data-href="' . action([\Modules\Exchange\Http\Controllers\ExchangeController::class, 'show'], [$row->id]) . '" class="btn-modal">';
                $html .= '<i class="fas fa-eye"></i> ' . __('exchange::lang.view_exchange') . '</a></li>';

                // Print exchange receipt
                $html .= '<li><a href="' . route('exchange.print-receipt', [$row->id]) . '" target="_blank">';
                $html .= '<i class="fas fa-receipt"></i> ' . __('exchange::lang.print_exchange_receipt') . '</a></li>';

                if ($is_cancelled) {
                    // For cancelled exchanges - show delete option
                    if (auth()->user()->can('exchange.delete')) {
                        $html .= '<li><a href="#" class="delete-exchange" data-exchange-id="' . $row->id . '" data-href="' . route('exchange.destroy', [$row->id]) . '">';
                        $html .= '<i class="fas fa-trash text-danger"></i> ' . __('exchange::lang.delete_exchange') . '</a></li>';
                    }
                } else {
                    // For active exchanges - show cancel option
                    if (auth()->user()->can('exchange.delete')) {
                        $html .= '<li><a href="#" class="cancel-exchange" data-exchange-id="' . $row->id . '" data-href="' . action([\Modules\Exchange\Http\Controllers\ExchangeController::class, 'cancel'], [$row->id]) . '">';
                        $html .= '<i class="fas fa-ban text-danger"></i> ' . __('exchange::lang.cancel_exchange') . '</a></li>';
                    }
                }

                $html .= '</ul></div>';
                return $html;
            })
            ->editColumn('status', function ($row) {
                if (!is_null($row->cancelled_at) || $row->status == 'cancelled') {
                    return '<span class="label label-danger">' . __('exchange::lang.cancelled') . '</span>';
                } else {
                    return '<span class="label label-success">' . ucfirst($row->status) . '</span>';
                }
            })
            ->editColumn('total_exchange_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->total_exchange_amount . '</span>';
            })
            ->editColumn('exchange_date', function ($row) {
                return $this->transactionUtil->format_date($row->exchange_date, true);
            })
            ->setRowClass(function ($row) {
                return (!is_null($row->cancelled_at) || $row->status == 'cancelled') ? 'cancelled-exchange' : '';
            })
            ->rawColumns(['action', 'total_exchange_amount', 'status'])
            ->make(true);
    }

    /**
     * Show exchange details
     */
    public function show($id)
    {
        if (!auth()->user()->can('exchange.access')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $exchange = TransactionExchange::with([
            'location',
            'originalTransaction',
            'originalTransaction.contact',
            'exchangeTransaction',
            'creator',
            'exchangeLines.originalSellLine.product',
            'exchangeLines.originalSellLine.variations',
            'exchangeLines.newSellLine.product',
            'exchangeLines.newSellLine.variations'
        ])->where('business_id', $business_id)->findOrFail($id);

        return view('exchange::show', compact('exchange'));
    }

    /**
     * Print exchange receipt
     */
    public function printReceipt($id)
    {
        if (!auth()->user()->can('exchange.access')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business_details = $this->businessUtil->getDetails($business_id);

        $exchange = TransactionExchange::with([
            'location',
            'originalTransaction',
            'originalTransaction.contact',
            'exchangeTransaction',
            'exchangeLines.originalSellLine.product',
            'exchangeLines.originalSellLine.variations',
            'exchangeLines.newSellLine.product',
            'exchangeLines.newSellLine.variations'
        ])->where('business_id', $business_id)->findOrFail($id);

        return view('exchange::receipt', compact('exchange', 'business_details'));
    }

    /**
     * Print exchange receipt in print-only format (no headers/sidebars)
     */
    public function printReceiptOnly($id)
    {
        if (!auth()->user()->can('exchange.access')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business_details = $this->businessUtil->getDetails($business_id);

        $exchange = TransactionExchange::with([
            'location',
            'originalTransaction',
            'originalTransaction.contact',
            'exchangeTransaction',
            'exchangeLines.originalSellLine.product',
            'exchangeLines.originalSellLine.variations',
            'exchangeLines.newSellLine.product',
            'exchangeLines.newSellLine.variations',
            'creator'
        ])->where('business_id', $business_id)->findOrFail($id);

        // Return print-only view (no app layout)
        return view('exchange::receipt-print-only', compact('exchange', 'business_details'));
    }

    /**
     * INTEGRATED: Get register details for cash register reporting
     * This method is integrated into ExchangeController to handle exchange-specific cash register calculations
     */
    public function getExchangeRegisterDetails($register_id = null)
    {
        if (!auth()->user()->can('view_cash_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $query = CashRegister::leftjoin(
            'cash_register_transactions as ct',
            'ct.cash_register_id',
            '=',
            'cash_registers.id'
        )
            ->join(
                'users as u',
                'u.id',
                '=',
                'cash_registers.user_id'
            )
            ->leftJoin(
                'business_locations as bl',
                'bl.id',
                '=',
                'cash_registers.location_id'
            );

        if (empty($register_id)) {
            $user_id = auth()->user()->id;
            $query->where('user_id', $user_id)
                ->where('cash_registers.status', 'open');
        } else {
            $query->where('cash_registers.id', $register_id);
        }

        $register_details = $query->select(
            'cash_registers.created_at as open_time',
            'cash_registers.closed_at as closed_at',
            'cash_registers.user_id',
            'cash_registers.closing_note',
            'cash_registers.location_id',
            'cash_registers.denominations',

            // Initial cash
            DB::raw("SUM(IF(ct.transaction_type='initial', ct.amount, 0)) as cash_in_hand"),

            // FIXED: Total sales - includes regular sales and exchange sales
            DB::raw("SUM(IF(ct.type='credit' AND ct.transaction_type = 'sell', ct.amount, 0)) as total_sale"),

            // FIXED: Total refunds - includes regular refunds and exchange refunds  
            DB::raw("SUM(IF(ct.type='debit' AND ct.transaction_type = 'refund', ct.amount, 0)) as total_refund"),

            // Expenses
            DB::raw("SUM(IF(ct.transaction_type='expense', ct.amount, 0)) as total_expense"),

            // FIXED: Cash totals - net amount considering credits and debits
            DB::raw("SUM(IF(ct.pay_method='cash', IF(ct.type='credit', ct.amount, -ct.amount), 0)) as total_cash"),
            DB::raw("SUM(IF(ct.pay_method='cash' AND ct.transaction_type='expense', ct.amount, 0)) as total_cash_expense"),

            // FIXED: Cheque totals
            DB::raw("SUM(IF(ct.pay_method='cheque', IF(ct.type='credit', ct.amount, -ct.amount), 0)) as total_cheque"),
            DB::raw("SUM(IF(ct.pay_method='cheque' AND ct.transaction_type='expense', ct.amount, 0)) as total_cheque_expense"),

            // FIXED: Card totals
            DB::raw("SUM(IF(ct.pay_method='card', IF(ct.type='credit', ct.amount, -ct.amount), 0)) as total_card"),
            DB::raw("SUM(IF(ct.pay_method='card' AND ct.transaction_type='expense', ct.amount, 0)) as total_card_expense"),

            // FIXED: Bank transfer totals
            DB::raw("SUM(IF(ct.pay_method='bank_transfer', IF(ct.type='credit', ct.amount, -ct.amount), 0)) as total_bank_transfer"),
            DB::raw("SUM(IF(ct.pay_method='bank_transfer' AND ct.transaction_type='expense', ct.amount, 0)) as total_bank_transfer_expense"),

            // FIXED: Other payment method totals
            DB::raw("SUM(IF(ct.pay_method='other', IF(ct.type='credit', ct.amount, -ct.amount), 0)) as total_other"),
            DB::raw("SUM(IF(ct.pay_method='other' AND ct.transaction_type='expense', ct.amount, 0)) as total_other_expense"),

            // Advance payments
            DB::raw("SUM(IF(ct.pay_method='advance', IF(ct.type='credit', ct.amount, -ct.amount), 0)) as total_advance"),
            DB::raw("SUM(IF(ct.pay_method='advance' AND ct.transaction_type='expense', ct.amount, 0)) as total_advance_expense"),

            // FIXED: Specific refund totals by payment method
            DB::raw("SUM(IF(ct.type='debit' AND ct.pay_method='cash', ct.amount, 0)) as total_cash_refund"),
            DB::raw("SUM(IF(ct.type='debit' AND ct.pay_method='cheque', ct.amount, 0)) as total_cheque_refund"),
            DB::raw("SUM(IF(ct.type='debit' AND ct.pay_method='card', ct.amount, 0)) as total_card_refund"),
            DB::raw("SUM(IF(ct.type='debit' AND ct.pay_method='bank_transfer', ct.amount, 0)) as total_bank_transfer_refund"),
            DB::raw("SUM(IF(ct.type='debit' AND ct.pay_method='other', ct.amount, 0)) as total_other_refund"),
            DB::raw("SUM(IF(ct.type='debit' AND ct.pay_method='advance', ct.amount, 0)) as total_advance_refund"),

            // Count totals
            DB::raw("SUM(IF(ct.pay_method='cheque', 1, 0)) as total_cheques"),
            DB::raw("SUM(IF(ct.pay_method='card', 1, 0)) as total_card_slips"),

            // User and location info
            DB::raw("CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, '')) as user_name"),
            'u.email',
            'bl.name as location_name'
        )->first();

        return $register_details;
    }
}
