<?php

namespace Modules\AdvancedReports\Http\Controllers;

use App\System;
use Carbon\Exceptions\Exception;
use Composer\Semver\Comparator;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Log;

class InstallController extends Controller
{
    public function __construct()
    {
        $this->module_name = 'advancedreports';
        $this->appVersion = config('advancedreports.module_version', '1.0.0');
        $this->module_display_name = 'Advanced Reports';
    }

    /**
     * Install
     *
     * @return Response
     */
    public function index()
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        $this->installSettings();

        //Check if installed or not.
        $is_installed = System::getProperty($this->module_name . '_version');
        if (empty($is_installed)) {
            try {
                DB::beginTransaction();
                DB::statement('SET default_storage_engine=INNODB;');

                // Debug: Check before migration
                \Log::info('AdvancedReports: About to run migration');

                // Run migrations
                Artisan::call('module:migrate', ['module' => 'AdvancedReports', '--force' => true]);

                // Create default report configurations
                $this->createDefaultReportConfigs();

                // Debug: Check after migration
                \Log::info('AdvancedReports: Migration completed');

                System::addProperty($this->module_name . '_version', $this->appVersion);
                
                // Check if transaction is still active before committing
                if (DB::transactionLevel() > 0) {
                    DB::commit();
                }

                // Clear all caches after successful installation
                $this->clearAllCaches();

                \Log::info('AdvancedReports: Installation successful');
            } catch (\Exception $e) {
                DB::rollBack();
                \Log::error('AdvancedReports Installation error: ' . $e->getMessage());
                throw $e;
            }
        }

        $output = [
            'success' => 1,
            'msg' => 'Advanced Reports module installed successfully. All caches have been cleared automatically.',
        ];

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    /**
     * Initialize all install functions
     */
    private function installSettings()
    {
        config(['app.debug' => true]);
        
        // Clear all caches for fresh installation
        $this->clearAllCaches();
    }

    /**
     * Clear all Laravel caches (config, route, view, cache)
     */
    private function clearAllCaches()
    {
        try {
            // Clear configuration cache
            Artisan::call('config:clear');
            \Log::info('AdvancedReports: Configuration cache cleared');

            // Clear route cache
            Artisan::call('route:clear');
            \Log::info('AdvancedReports: Route cache cleared');

            // Clear view cache
            Artisan::call('view:clear');
            \Log::info('AdvancedReports: View cache cleared');

            // Clear application cache
            Artisan::call('cache:clear');
            \Log::info('AdvancedReports: Application cache cleared');

            // Clear compiled services and packages
            Artisan::call('clear-compiled');
            \Log::info('AdvancedReports: Compiled services cleared');

            // Optimize autoloader (optional but recommended)
            if (app()->environment('production')) {
                Artisan::call('optimize');
                \Log::info('AdvancedReports: Application optimized for production');
            }

            \Log::info('AdvancedReports: All caches cleared successfully');
        } catch (\Exception $e) {
            \Log::error('AdvancedReports: Error clearing caches: ' . $e->getMessage());
            // Don't throw exception here as cache clearing shouldn't stop installation
        }
    }

    /**
     * Create default report configurations
     */
    private function createDefaultReportConfigs()
    {
        try {
            // Create report_configurations table data if needed
            DB::table('advanced_report_configurations')->insert([
                [
                    'report_type' => 'stock',
                    'report_name' => 'Stock Report',
                    'columns' => json_encode([
                        'sku',
                        'product_name',
                        'category',
                        'location',
                        'current_stock',
                        'stock_value_purchase',
                        'stock_value_sale',
                        'potential_profit'
                    ]),
                    'filters' => json_encode(['location', 'category', 'date_range']),
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now()
                ],
                [
                    'report_type' => 'sales',
                    'report_name' => 'Sales Report',
                    'columns' => json_encode([
                        'invoice_no',
                        'date',
                        'customer',
                        'location',
                        'payment_status',
                        'total_before_tax',
                        'tax_amount',
                        'total_amount'
                    ]),
                    'filters' => json_encode(['date_range', 'location', 'customer', 'payment_status']),
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now()
                ],
                [
                    'report_type' => 'product',
                    'report_name' => 'Product Sales Report',
                    'columns' => json_encode([
                        'product_name',
                        'sku',
                        'category',
                        'total_sold',
                        'total_amount',
                        'profit_margin',
                        'location'
                    ]),
                    'filters' => json_encode(['date_range', 'location', 'category']),
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now()
                ]
            ]);

            \Log::info('AdvancedReports: Default configurations created');
        } catch (\Exception $e) {
            \Log::error('AdvancedReports: Error creating default configs: ' . $e->getMessage());
        }
    }

    // Updating
    public function update()
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '512M');

            $advanced_reports_version = System::getProperty($this->module_name . '_version');

            if (Comparator::greaterThan($this->appVersion, $advanced_reports_version)) {
                ini_set('max_execution_time', 0);
                ini_set('memory_limit', '512M');
                
                // Clear all caches before update
                $this->clearAllCaches();

                DB::statement('SET default_storage_engine=INNODB;');
                Artisan::call('module:migrate', ['module' => 'AdvancedReports', '--force' => true]);

                System::setProperty($this->module_name . '_version', $this->appVersion);

                // Clear all caches after update
                $this->clearAllCaches();

                \Log::info('AdvancedReports: Update completed to version ' . $this->appVersion);
            } else {
                abort(404);
            }

            DB::commit();

            $output = [
                'success' => 1,
                'msg' => 'Advanced Reports module updated successfully to version ' . $this->appVersion . '! All caches have been cleared automatically.',
            ];

            return redirect()
                ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
                ->with('status', $output);
        } catch (Exception $e) {
            DB::rollBack();
            \Log::error('AdvancedReports Update error: ' . $e->getMessage());
            exit($e->getMessage());
        }
    }

    /**
     * Uninstall
     *
     * @return Response
     */
    public function uninstall()
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            // Drop module tables (optional - you might want to keep data)
            // DB::statement('DROP TABLE IF EXISTS advanced_report_configurations');

            System::removeProperty($this->module_name . '_version');

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('advancedreports::lang.uninstall_success'),
            ];

            \Log::info('AdvancedReports: Uninstalled successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('AdvancedReports Uninstall error: ' . $e->getMessage());
            $output = [
                'success' => false,
                'msg' => $e->getMessage(),
            ];
        }

        return redirect()->back()->with(['status' => $output]);
    }
}
