@extends('advancedreports::layouts.app')
@section('title', __('advancedreports::lang.daily_report'))

@section('content')

<!-- Add CSRF token meta tag -->
<meta name="csrf-token" content="{{ csrf_token() }}">

<!-- Content Header (Page header) -->
<section class="content-header">
    <h1>@lang('advancedreports::lang.daily_report')
        <small>@lang('advancedreports::lang.comprehensive_daily_overview')</small>
    </h1>
</section>

<!-- Main content -->
<section class="content">
    <div class="row">
        <div class="col-md-12">
            @component('components.filters', [
            'title' => __('report.filters'),
            'class' => 'box-primary'
            ])
            <div class="col-md-4">
                <div class="form-group">
                    {!! Form::label('location_id', __('purchase.business_location') . ':') !!}
                    {!! Form::select('location_id', $business_locations, null, ['class' => 'form-control select2',
                    'style' => 'width:100%', 'id' => 'location_filter']); !!}
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-group">
                    {!! Form::label('end_date', __('advancedreports::lang.report_date') . ':') !!}
                    <div class="input-group">
                        {!! Form::text('end_date', \Carbon\Carbon::now()->format(session('business.date_format') ?:
                        'd/m/Y'),
                        ['class' => 'form-control', 'id' => 'end_date_filter', 'placeholder' => 'DD/MM/YYYY']); !!}
                        <div class="input-group-addon">
                            <i class="fa fa-calendar"></i>
                        </div>
                    </div>
                    <small class="text-muted">Format: DD/MM/YYYY</small>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-group">
                    <br>
                    <button type="button" class="btn btn-primary" id="filter_btn">
                        <i class="fa fa-refresh"></i> @lang('advancedreports::lang.refresh')
                    </button>
                    {{-- <button type="button" class="btn btn-success" id="export_btn">
                        <i class="fa fa-download"></i> @lang('lang_v1.export')
                    </button> --}}
                    {{-- <button type="button" class="btn btn-info" id="test_data_btn">
                        <i class="fa fa-bug"></i> Test Data
                    </button> --}}
                </div>
            </div>
            @endcomponent
        </div>
    </div>


    <!-- Replace your Main Summary Cards section with this (FontAwesome 5/6 compatible) -->
    <div class="row" id="summary_cards">
        <div class="col-xl-2 col-lg-4 col-md-6 col-sm-6 col-xs-12">
            <div class="modern-widget widget-sales">
                <i class="fas fa-shopping-cart modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Today's Sales</div>
                    <div class="modern-widget-number" id="today_sales">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-lg-4 col-md-6 col-sm-6 col-xs-12">
            <div class="modern-widget widget-purchases">
                <i class="fas fa-truck modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Today's Purchases</div>
                    <div class="modern-widget-number" id="today_purchases">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-lg-4 col-md-6 col-sm-6 col-xs-12">
            <div class="modern-widget widget-expenses">
                <i class="fas fa-credit-card modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Today's Expenses</div>
                    <div class="modern-widget-number" id="today_expenses">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-lg-4 col-md-6 col-sm-6 col-xs-12">
            <div class="modern-widget widget-profit">
                <i class="fas fa-chart-line modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Today's Profit</div>
                    <div class="modern-widget-number" id="today_profit">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-lg-4 col-md-6 col-sm-6 col-xs-12">
            <div class="modern-widget widget-cash">
                <i class="fas fa-money-bill-wave modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Cash in Hand</div>
                    <div class="modern-widget-number" id="cash_in_hand">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-lg-4 col-md-6 col-sm-6 col-xs-12">
            <div class="modern-widget widget-bank">
                <i class="fas fa-university modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Bank Balance</div>
                    <div class="modern-widget-number" id="bank_balance">0</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Replace your Financial Position Cards section with this (FontAwesome 5/6 compatible) -->
    <div class="row">
        <div class="col-xl-3 col-lg-3 col-md-3 col-sm-6 col-xs-12">
            <div class="modern-widget widget-customer-due">
                <i class="fas fa-users modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Customer Due</div>
                    <div class="modern-widget-number" id="customer_due">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-lg-3 col-md-3 col-sm-6 col-xs-12">
            <div class="modern-widget widget-supplier-due">
                <i class="fas fa-industry modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Supplier Due</div>
                    <div class="modern-widget-number" id="supplier_due">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-lg-3 col-md-3 col-sm-6 col-xs-12">
            <div class="modern-widget widget-collections">
                <i class="fas fa-hand-holding-usd modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Today's Collections</div>
                    <div class="modern-widget-number" id="today_collections">0</div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-lg-3 col-md-3 col-sm-6 col-xs-12">
            <div class="modern-widget widget-net-worth">
                <i class="fas fa-gem modern-widget-icon"></i>
                <div class="modern-widget-content">
                    <div class="modern-widget-text">Net Worth</div>
                    <div class="modern-widget-number" id="net_worth">0</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Activity Summary -->
    <div class="row">
        <div class="col-md-6">
            @component('components.widget', ['class' => 'box-success', 'title' =>
            __('advancedreports::lang.todays_activity')])
            <div class="table-responsive">
                <table class="table table-condensed" id="activity_summary">
                    <tbody>
                        <tr>
                            <td>@lang('advancedreports::lang.total_transactions'):</td>
                            <td class="text-right" id="transactions_count">0</td>
                        </tr>
                        <tr>
                            <td>@lang('advancedreports::lang.new_customers'):</td>
                            <td class="text-right" id="new_customers">0</td>
                        </tr>
                        <tr>
                            <td>@lang('advancedreports::lang.avg_transaction_value'):</td>
                            <td class="text-right" id="avg_transaction">
                                <span class="display_currency" data-currency_symbol="true">0</span>
                            </td>
                        </tr>
                        <tr class="info">
                            <td><strong>@lang('advancedreports::lang.gross_profit_margin'):</strong></td>
                            <td class="text-right" id="profit_margin"><strong>0%</strong></td>
                        </tr>
                    </tbody>
                </table>
            </div>
            @endcomponent
        </div>

        <div class="col-md-6">
            @component('components.widget', ['class' => 'box-info', 'title' =>
            __('advancedreports::lang.financial_snapshot')])
            <div class="table-responsive">
                <table class="table table-condensed" id="financial_snapshot">
                    <tbody>
                        <tr>
                            <td>@lang('advancedreports::lang.opening_balance'):</td>
                            <td class="text-right" id="opening_balance">
                                <span class="display_currency" data-currency_symbol="true">0</span>
                            </td>
                        </tr>
                        <tr>
                            <td>@lang('advancedreports::lang.closing_balance'):</td>
                            <td class="text-right" id="closing_balance">
                                <span class="display_currency" data-currency_symbol="true">0</span>
                            </td>
                        </tr>
                        <tr>
                            <td>@lang('advancedreports::lang.cash_flow'):</td>
                            <td class="text-right" id="cash_flow">
                                <span class="display_currency" data-currency_symbol="true">0</span>
                            </td>
                        </tr>
                        <tr class="info">
                            <td><strong>@lang('advancedreports::lang.liquidity_ratio'):</strong></td>
                            <td class="text-right" id="liquidity_ratio"><strong>0:1</strong></td>
                        </tr>
                    </tbody>
                </table>
            </div>
            @endcomponent
        </div>
    </div>

    <!-- Sales & Purchase Analysis -->
    <div class="row">
        <div class="col-md-6">
            @component('components.widget', ['class' => 'box-primary', 'title' => 'Sales Analysis'])
            <div id="sales_analysis_content">
                <div class="text-center">
                    <i class="fa fa-spinner fa-spin fa-2x text-muted"></i>
                    <p class="text-muted">@lang('advancedreports::lang.loading')</p>
                </div>
            </div>
            @endcomponent
        </div>

        <div class="col-md-6">
            @component('components.widget', ['class' => 'box-warning', 'title' => 'Purchase Analysis'])
            <div id="purchase_analysis_content">
                <div class="text-center">
                    <i class="fa fa-spinner fa-spin fa-2x text-muted"></i>
                    <p class="text-muted">@lang('advancedreports::lang.loading')</p>
                </div>
            </div>
            @endcomponent
        </div>
    </div>

    <!-- Top Products & Payment Methods -->
    <div class="row">
        <div class="col-md-6">
            @component('components.widget', ['class' => 'box-success', 'title' => 'Top Selling Products'])
            <div id="top_products_content">
                <div class="text-center">
                    <i class="fa fa-spinner fa-spin fa-2x text-muted"></i>
                    <p class="text-muted">@lang('advancedreports::lang.loading')</p>
                </div>
            </div>
            @endcomponent
        </div>

        <div class="col-md-6">
            @component('components.widget', ['class' => 'box-info', 'title' => 'Payment Methods Analysis'])
            <div id="payment_methods_content">
                <div class="text-center">
                    <i class="fa fa-spinner fa-spin fa-2x text-muted"></i>
                    <p class="text-muted">@lang('advancedreports::lang.loading')</p>
                </div>
            </div>
            @endcomponent
        </div>
    </div>

    <!-- Expense Breakdown -->
    <div class="row">
        <div class="col-md-12">
            @component('components.widget', ['class' => 'box-danger', 'title' => 'Expense Breakdown'])
            <div id="expense_breakdown_content">
                <div class="text-center">
                    <i class="fa fa-spinner fa-spin fa-2x text-muted"></i>
                    <p class="text-muted">@lang('advancedreports::lang.loading')</p>
                </div>
            </div>
            @endcomponent
        </div>
    </div>

    <!-- Monthly Cash Flow Breakdown -->
    <div class="row">
        <div class="col-md-12">
            @component('components.widget', ['class' => 'box-primary', 'title' => 'Monthly Cash Flow Breakdown'])
            <div id="monthly_cash_breakdown_content">
                <div class="text-center">
                    <i class="fa fa-spinner fa-spin fa-2x text-muted"></i>
                    <p class="text-muted">@lang('advancedreports::lang.loading')</p>
                </div>
            </div>
            @endcomponent
        </div>
    </div>

</section>
<!-- /.content -->

@stop

@section('css')
<style>
    /* Modern Widget Styling */
    .modern-widget {
        height: 95px !important;
        display: flex !important;
        align-items: center !important;
        padding: 18px !important;
        border-radius: 12px !important;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.12) !important;
        transition: all 0.3s ease !important;
        margin-bottom: 20px;
        position: relative;
        overflow: hidden;
        color: white;
        border: none !important;
    }

    .modern-widget::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: inherit;
        opacity: 0.95;
        z-index: 1;
    }

    .modern-widget:hover {
        transform: translateY(-4px) scale(1.02) !important;
        box-shadow: 0 8px 30px rgba(0, 0, 0, 0.2) !important;
    }

    /* Enhanced Icon Styling for FontAwesome 5/6 */
    .modern-widget-icon {
        font-size: 45px !important;
        opacity: 1 !important;
        margin-right: 18px !important;
        width: 55px !important;
        text-align: center !important;
        flex-shrink: 0 !important;
        color: white !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        z-index: 2 !important;
        position: relative !important;
        text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3) !important;
        font-family: 'Font Awesome 5 Free', 'Font Awesome 5 Pro' !important;
        font-weight: 900 !important;
        /* Required for FontAwesome 5 solid icons */
        font-style: normal !important;
        font-variant: normal !important;
        text-rendering: auto !important;
        -webkit-font-smoothing: antialiased !important;
    }

    /* Ensure FontAwesome icons are visible */
    .fas,
    .far,
    .fab {
        font-family: 'Font Awesome 5 Free', 'Font Awesome 5 Pro' !important;
        font-weight: 900 !important;
    }

    .far {
        font-weight: 400 !important;
    }

    /* Force icon visibility */
    .modern-widget-icon {
        visibility: visible !important;
        opacity: 1 !important;
        min-width: 55px !important;
    }

    /* Fallback for missing icons - Unicode symbols */
    .widget-sales .modern-widget-icon:empty:before {
        content: "🛒";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-purchases .modern-widget-icon:empty:before {
        content: "🚚";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-expenses .modern-widget-icon:empty:before {
        content: "💳";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-profit .modern-widget-icon:empty:before {
        content: "📈";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-cash .modern-widget-icon:empty:before {
        content: "💰";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-bank .modern-widget-icon:empty:before {
        content: "🏦";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-customer-due .modern-widget-icon:empty:before {
        content: "👥";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-supplier-due .modern-widget-icon:empty:before {
        content: "🏭";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-collections .modern-widget-icon:empty:before {
        content: "💵";
        font-family: inherit;
        font-weight: normal;
    }

    .widget-net-worth .modern-widget-icon:empty:before {
        content: "💎";
        font-family: inherit;
        font-weight: normal;
    }

    /* Mobile responsive updates */
    @media (max-width: 768px) {
        .modern-widget-icon {
            font-size: 36px !important;
            width: 45px !important;
            min-width: 45px !important;
            margin-right: 12px !important;
        }
    }

    @media (max-width: 992px) {
        .modern-widget-icon {
            font-size: 40px !important;
            width: 50px !important;
            min-width: 50px !important;
        }
    }

    /* Content Area Enhanced */
    .modern-widget-content {
        flex-grow: 1 !important;
        display: flex !important;
        flex-direction: column !important;
        justify-content: center !important;
        min-height: 65px !important;
        color: white !important;
        z-index: 2 !important;
        position: relative !important;
    }

    /* Text Styling Enhanced */
    .modern-widget-text {
        font-size: 13px !important;
        opacity: 0.98 !important;
        font-weight: 600 !important;
        text-transform: uppercase !important;
        letter-spacing: 1px !important;
        margin-bottom: 6px !important;
        line-height: 1.2 !important;
        color: white !important;
        text-shadow: 0 1px 3px rgba(0, 0, 0, 0.2) !important;
    }

    /* Number Styling Enhanced */
    .modern-widget-number {
        font-size: 26px !important;
        font-weight: 700 !important;
        line-height: 1 !important;
        color: white !important;
        text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2) !important;
        margin: 0 !important;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif !important;
    }

    /* Enhanced Widget Color Schemes with Stronger Gradients */
    .widget-sales {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-purchases {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-expenses {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-profit {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-loss {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-cash {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-bank {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-customer-due {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-supplier-due {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-collections {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    .widget-net-worth {
        background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%) !important;
        box-shadow: 0 4px 20px rgba(30, 60, 114, 0.4) !important;
    }

    /* Enhanced hover effects for these specific widgets */
    .widget-sales:hover,
    .widget-collections:hover,
    .widget-net-worth:hover {
        transform: translateY(-6px) scale(1.04) !important;
        box-shadow: 0 15px 45px rgba(0, 0, 0, 0.3) !important;
    }

    /* Enhanced hover effects */
    .modern-widget:hover {
        transform: translateY(-5px) scale(1.03) !important;
        box-shadow: 0 12px 40px rgba(0, 0, 0, 0.25) !important;
        transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94) !important;
    }

    /* Ensure text visibility with stronger contrast */
    .modern-widget * {
        color: white !important;
        text-shadow: 0 2px 4px rgba(0, 0, 0, 0.4) !important;
    }

    .modern-widget-text {
        text-shadow: 0 1px 3px rgba(0, 0, 0, 0.5) !important;
        font-weight: 700 !important;
    }

    .modern-widget-number {
        text-shadow: 0 2px 6px rgba(0, 0, 0, 0.5) !important;
        font-weight: 800 !important;
    }

    /* Add subtle animation on page load */
    @keyframes slideInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }

        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .modern-widget {
        animation: slideInUp 0.6s ease-out;
    }

    /* Stagger animation for multiple widgets */
    .modern-widget:nth-child(1) {
        animation-delay: 0.1s;
    }

    .modern-widget:nth-child(2) {
        animation-delay: 0.2s;
    }

    .modern-widget:nth-child(3) {
        animation-delay: 0.3s;
    }

    .modern-widget:nth-child(4) {
        animation-delay: 0.4s;
    }

    .modern-widget:nth-child(5) {
        animation-delay: 0.5s;
    }

    .modern-widget:nth-child(6) {
        animation-delay: 0.6s;
    }

    /* Force all text to be white */
    .modern-widget * {
        color: white !important;
    }

    /* Enhanced Mobile Responsive */
    @media (max-width: 768px) {
        .modern-widget {
            height: 85px !important;
            padding: 14px !important;
            border-radius: 10px !important;
        }

        .modern-widget-icon {
            font-size: 36px !important;
            width: 45px !important;
            margin-right: 12px !important;
        }

        .modern-widget-text {
            font-size: 11px !important;
            letter-spacing: 0.5px !important;
        }

        .modern-widget-number {
            font-size: 22px !important;
        }

        .modern-widget-content {
            min-height: 55px !important;
        }
    }

    @media (max-width: 992px) {
        .modern-widget {
            height: 90px !important;
            padding: 16px !important;
        }

        .modern-widget-icon {
            font-size: 40px !important;
            width: 50px !important;
        }

        .modern-widget-text {
            font-size: 12px !important;
        }

        .modern-widget-number {
            font-size: 24px !important;
        }
    }

    /* Ensure Bootstrap columns are equal height */
    .row {
        display: flex;
        flex-wrap: wrap;
    }

    .col-xl-2,
    .col-lg-4,
    .col-md-6,
    .col-sm-6,
    .col-xs-12,
    .col-xl-3,
    .col-lg-6 {
        display: flex;
        flex-direction: column;
    }

    /* Info Box Styling for Analysis Sections */
    .info-box {
        display: block;
        min-height: 70px;
        background: #fff;
        width: 100%;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.12), 0 1px 2px rgba(0, 0, 0, 0.24);
        border-radius: 8px;
        margin-bottom: 15px;
        transition: all 0.3s ease;
    }

    .info-box:hover {
        box-shadow: 0 3px 6px rgba(0, 0, 0, 0.16), 0 3px 6px rgba(0, 0, 0, 0.23);
        transform: translateY(-2px);
    }

    .info-box-icon {
        border-top-left-radius: 8px;
        border-top-right-radius: 0;
        border-bottom-right-radius: 0;
        border-bottom-left-radius: 8px;
        display: block;
        float: left;
        height: 70px;
        width: 70px;
        text-align: center;
        font-size: 28px;
        line-height: 70px;
        background: rgba(0, 0, 0, 0.2);
        color: rgba(255, 255, 255, 0.9);
    }

    .info-box-content {
        padding: 8px 10px;
        margin-left: 70px;
    }

    .info-box-text {
        text-transform: uppercase;
        font-weight: 600;
        font-size: 11px;
        color: rgba(255, 255, 255, 0.9);
        display: block;
        margin-bottom: 2px;
    }

    .info-box-number {
        display: block;
        font-weight: 700;
        font-size: 16px;
        color: white;
        text-shadow: 0 1px 2px rgba(0, 0, 0, 0.3);
    }

    /* Color-specific info boxes */
    .info-box.bg-aqua {
        background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);
        color: white;
    }

    .info-box.bg-green {
        background: linear-gradient(135deg, #28a745 0%, #218838 100%);
        color: white;
    }

    .info-box.bg-yellow {
        background: linear-gradient(135deg, #ffc107 0%, #e0a800 100%);
        color: white;
    }

    .info-box.bg-red {
        background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
        color: white;
    }

    /* Loading States */
    .loading-overlay {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(255, 255, 255, 0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        border-radius: 8px;
    }

    .loading-spinner {
        display: inline-block;
        width: 40px;
        height: 40px;
        border: 3px solid #f3f3f3;
        border-top: 3px solid #3498db;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        0% {
            transform: rotate(0deg);
        }

        100% {
            transform: rotate(360deg);
        }
    }

    /* Table Enhancements */
    .table-condensed {
        font-size: 13px;
    }

    .table-condensed td {
        padding: 6px 8px;
        border-top: 1px solid #ddd;
    }

    .table-condensed tr.info {
        background-color: #d9edf7;
        font-weight: 600;
    }

    .table-striped>tbody>tr:nth-of-type(odd) {
        background-color: #f9f9f9;
    }

    .table-striped>tbody>tr:hover {
        background-color: #f5f5f5;
    }

    /* Box Widget Enhancements */
    .box {
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        margin-bottom: 20px;
        transition: all 0.3s ease;
    }

    .box:hover {
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
        transform: translateY(-2px);
    }

    .box-header {
        border-bottom: 1px solid #f0f0f0;
        padding: 15px;
        font-weight: 600;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border-top-left-radius: 8px;
        border-top-right-radius: 8px;
    }

    .box-body {
        padding: 20px;
    }

    .box.box-primary .box-header {
        background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
        color: white;
    }

    .box.box-success .box-header {
        background: linear-gradient(135deg, #28a745 0%, #1e7e34 100%);
        color: white;
    }

    .box.box-info .box-header {
        background: linear-gradient(135deg, #17a2b8 0%, #117a8b 100%);
        color: white;
    }

    .box.box-warning .box-header {
        background: linear-gradient(135deg, #ffc107 0%, #d39e00 100%);
        color: white;
    }

    .box.box-danger .box-header {
        background: linear-gradient(135deg, #dc3545 0%, #bd2130 100%);
        color: white;
    }

    /* Responsive Tables */
    .table-responsive {
        border: 1px solid #ddd;
        border-radius: 6px;
        overflow-x: auto;
    }

    .table-responsive table {
        margin-bottom: 0;
    }

    /* Print Styles */
    @media print {
        .modern-widget {
            background: #f8f9fa !important;
            color: #333 !important;
            border: 2px solid #dee2e6 !important;
            box-shadow: none !important;
            break-inside: avoid;
        }

        .modern-widget * {
            color: #333 !important;
        }

        .box {
            break-inside: avoid;
            box-shadow: none !important;
            border: 1px solid #ccc !important;
        }

        .btn {
            display: none !important;
        }
    }

    /* Animation for number updates */
    @keyframes numberUpdate {
        0% {
            transform: scale(1);
        }

        50% {
            transform: scale(1.1);
        }

        100% {
            transform: scale(1);
        }
    }

    .modern-widget-number.updating {
        animation: numberUpdate 0.5s ease;
    }

    /* Status indicators */
    .status-positive {
        color: #28a745 !important;
    }

    .status-negative {
        color: #dc3545 !important;
    }

    .status-neutral {
        color: #6c757d !important;
    }

    /* Loading states for boxes */
    .box-loading {
        position: relative;
        overflow: hidden;
    }

    .box-loading::after {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
        animation: loading-shimmer 1.5s infinite;
    }

    @keyframes loading-shimmer {
        0% {
            left: -100%;
        }

        100% {
            left: 100%;
        }
    }

    /* Tooltip styling */
    .tooltip {
        font-size: 12px;
    }

    .tooltip-inner {
        background-color: #333;
        color: white;
        border-radius: 4px;
        padding: 8px 12px;
        max-width: 250px;
    }

    /* Error states */
    .error-message {
        color: #dc3545;
        font-style: italic;
        text-align: center;
        padding: 20px;
        background-color: #f8d7da;
        border: 1px solid #f5c6cb;
        border-radius: 4px;
        margin: 10px 0;
    }

    /* Success states */
    .success-message {
        color: #155724;
        text-align: center;
        padding: 15px;
        background-color: #d4edda;
        border: 1px solid #c3e6cb;
        border-radius: 4px;
        margin: 10px 0;
    }

    /* Monthly Cash Breakdown Specific Styles */
    .info-box.bg-blue {
        background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
        color: white;
    }

    .badge-primary {
        background-color: #007bff;
        color: white;
        padding: 4px 8px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: bold;
    }

    .table-condensed th {
        background-color: #f8f9fa !important;
        border-bottom: 2px solid #dee2e6;
        font-weight: 600;
        text-transform: uppercase;
        font-size: 12px;
        letter-spacing: 0.5px;
    }

    .table-condensed tbody tr:hover {
        background-color: #f1f3f4;
        transition: background-color 0.2s ease;
    }

    .table-condensed .text-success {
        color: #28a745 !important;
        font-weight: 600;
    }

    .table-condensed .text-danger {
        color: #dc3545 !important;
        font-weight: 600;
    }

    /* Info boxes for monthly breakdown */
    .info-box {
        margin-bottom: 15px;
    }

    .info-box-icon {
        width: 70px;
        height: 70px;
        line-height: 70px;
        text-align: center;
        font-size: 28px;
    }

    .info-box-content {
        padding: 8px 10px;
        margin-left: 70px;
    }

    .info-box-text {
        text-transform: uppercase;
        font-weight: 600;
        font-size: 11px;
        display: block;
        margin-bottom: 2px;
        color: rgba(255, 255, 255, 0.9);
    }

    .info-box-number {
        display: block;
        font-weight: 700;
        font-size: 16px;
        color: white;
    }
</style>
@endsection
@section('javascript')

<script type="text/javascript">
    // Currency formatting function
    function formatCurrency(num) {
        if (typeof __currency_trans_from_en === 'function') {
            return __currency_trans_from_en(num, true);
        }
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD',
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        }).format(num || 0);
    }

    function formatNumber(num) {
        return new Intl.NumberFormat('en-US', {
            minimumFractionDigits: 0,
            maximumFractionDigits: 2
        }).format(num || 0);
    }

    $(document).ready(function() {
        console.log('Initializing daily report (safe version)...');
        
        // ===== SIMPLE DATE PICKER INITIALIZATION (NO LOOPS) =====
        $('#end_date_filter').datepicker({
            autoclose: true,
            format: 'dd/mm/yyyy',
            todayHighlight: true,
            todayBtn: 'linked',
            clearBtn: false,
            orientation: 'bottom auto',
            weekStart: 1,
            startDate: '-5y',
            endDate: '+1m'
        });

        // ===== SET DEFAULT DATE TO TODAY (SIMPLE) =====
        var today = moment().format('DD/MM/YYYY');
        $('#end_date_filter').val(today);
        console.log('Default date set to:', today);

        // ===== SIMPLE DATE CHANGE HANDLER =====
        $('#end_date_filter').on('changeDate', function(e) {
            if (e.date) {
                var formattedDate = moment(e.date).format('DD/MM/YYYY');
                console.log('Date changed to:', formattedDate);
                $(this).val(formattedDate);
                
                // Debounced reload to prevent multiple calls
                clearTimeout(window.dateChangeTimeout);
                window.dateChangeTimeout = setTimeout(function() {
                    loadAllData();
                }, 300);
            }
        });

        // ===== MANUAL DATE INPUT (SIMPLIFIED) =====
        $('#end_date_filter').on('blur', function() {
            var inputValue = $(this).val().trim();
            
            if (inputValue && inputValue.length >= 8) {
                try {
                    var parsedDate = moment(inputValue, ['DD/MM/YYYY', 'DD-MM-YYYY', 'D/M/YYYY'], true);
                    
                    if (parsedDate.isValid()) {
                        var correctedFormat = parsedDate.format('DD/MM/YYYY');
                        $(this).val(correctedFormat);
                        console.log('Date corrected to:', correctedFormat);
                        
                        clearTimeout(window.dateChangeTimeout);
                        window.dateChangeTimeout = setTimeout(function() {
                            loadAllData();
                        }, 300);
                    } else {
                        console.warn('Invalid date, resetting to today');
                        $(this).val(moment().format('DD/MM/YYYY'));
                    }
                } catch (e) {
                    console.error('Date parsing error:', e);
                    $(this).val(moment().format('DD/MM/YYYY'));
                }
            }
        });

        // ===== FILTER BUTTON =====
        $('#filter_btn').click(function() {
            var $btn = $(this);
            var originalText = $btn.html();
            $btn.html('<i class="fa fa-spinner fa-spin"></i> Loading...').prop('disabled', true);
            
            loadAllData().finally(function() {
                $btn.html(originalText).prop('disabled', false);
            });
        });

        // ===== LOCATION FILTER =====
        $('#location_filter').change(function() {
            console.log('Location changed to:', $(this).val());
            clearTimeout(window.locationChangeTimeout);
            window.locationChangeTimeout = setTimeout(function() {
                loadAllData();
            }, 300);
        });

        // ===== EXPORT BUTTON =====
        $('#export_btn').click(function(e) {
            e.preventDefault();
            
            var location_id = $('#location_filter').val() || '';
            var end_date = $('#end_date_filter').val() || '';
            
            var $btn = $(this);
            var originalText = $btn.html();
            
            $btn.html('<i class="fa fa-spinner fa-spin"></i> Exporting...').prop('disabled', true);
            
            var url = "{!! action('\\Modules\\AdvancedReports\\Http\\Controllers\\DailyReportController@export') !!}";
            url += '?location_id=' + encodeURIComponent(location_id) + '&end_date=' + encodeURIComponent(end_date);
            
            window.open(url, '_blank');
            
            setTimeout(function() {
                $btn.html(originalText).prop('disabled', false);
            }, 2000);
        });

        // ===== LOAD ALL DATA (SAFE VERSION) =====
        function loadAllData() {
            console.log('Loading all data (safe)...');
            
            var location_id = $('#location_filter').val() || '';
            var end_date = $('#end_date_filter').val() || '';
            
            // Convert DD/MM/YYYY to YYYY-MM-DD
            if (end_date && end_date.match(/^\d{1,2}\/\d{1,2}\/\d{4}$/)) {
                var parts = end_date.split('/');
                var day = parts[0].padStart(2, '0');
                var month = parts[1].padStart(2, '0');
                var year = parts[2];
                end_date = year + '-' + month + '-' + day;
            }
            
            // Load summary and detailed data
            return Promise.all([
                loadSummary(end_date),
                loadDetailedBreakdown(end_date)
            ]).catch(function(error) {
                console.error('Error loading data:', error);
                showToast('error', 'Error loading report data');
            });
        }

        // ===== LOAD SUMMARY (SAFE VERSION) =====
        function loadSummary(converted_date) {
            var location_id = $('#location_filter').val() || '';
            var end_date = converted_date || $('#end_date_filter').val() || '';
            
            return $.ajax({
                url: "{!! action('\\Modules\\AdvancedReports\\Http\\Controllers\\DailyReportController@getSummary') !!}",
                method: 'GET',
                data: {
                    location_id: location_id,
                    end_date: end_date,
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                dataType: 'json',
                timeout: 15000,
                success: function(data) {
                    console.log('Summary data received');
                    
                    // Update widgets safely
                    updateWidgetValue('#today_sales', data.today_sales || 0);
                    updateWidgetValue('#today_purchases', data.today_purchases || 0);
                    updateWidgetValue('#today_expenses', data.today_expenses || 0);
                    updateWidgetValue('#today_profit', data.today_profit || 0);
                    updateWidgetValue('#cash_in_hand', data.cash_in_hand || 0);
                    updateWidgetValue('#bank_balance', data.bank_balance || 0);
                    updateWidgetValue('#customer_due', data.customer_due || 0);
                    updateWidgetValue('#supplier_due', data.supplier_due || 0);
                    updateWidgetValue('#today_collections', data.today_collections || 0);
                    updateWidgetValue('#net_worth', data.net_worth || 0);

                    // Update activity summary
                    $('#transactions_count').text(formatNumber(data.transactions_count || 0));
                    $('#new_customers').text(formatNumber(data.new_customers || 0));
                    $('#profit_margin').text((data.profit_margin || 0).toFixed(2) + '%');
                    $('#liquidity_ratio').text((data.liquidity_ratio || 0).toFixed(2) + ':1');

                    // Update financial snapshot
                    $('#opening_balance .display_currency').text((data.opening_balance || 0).toFixed(2));
                    $('#closing_balance .display_currency').text((data.closing_balance || 0).toFixed(2));
                    $('#cash_flow .display_currency').text((data.cash_flow || 0).toFixed(2));
                    $('#avg_transaction .display_currency').text((data.avg_transaction || 0).toFixed(2));
                },
                error: function(xhr, status, error) {
                    console.error('Summary loading error:', error);
                    showToast('error', 'Error loading summary data');
                }
            });
        }

        // ===== LOAD DETAILED BREAKDOWN (SAFE VERSION) =====
        function loadDetailedBreakdown(converted_date) {
            var location_id = $('#location_filter').val() || '';
            var end_date = $('#end_date_filter').val() || '';
            
            return $.ajax({
                url: "{!! action('\\Modules\\AdvancedReports\\Http\\Controllers\\DailyReportController@getDailyReportData') !!}",
                method: 'GET',
                data: {
                    location_id: location_id,
                    end_date: end_date,
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                dataType: 'json',
                timeout: 15000,
                success: function(data) {
                    console.log('Detailed data received');
                    
                    // Render sections safely
                    if (data.sales_breakdown) {
                        renderSalesAnalysis(data.sales_breakdown);
                    }
                    if (data.purchase_breakdown) {
                        renderPurchaseAnalysis(data.purchase_breakdown);
                    }
                    if (data.top_products) {
                        renderTopProducts(data.top_products);
                    }
                    if (data.payment_methods) {
                        renderPaymentMethods(data.payment_methods);
                    }
                    if (data.expense_breakdown) {
                        renderExpenseBreakdown(data.expense_breakdown);
                    }
                    if (data.monthly_cash_breakdown) {
                        renderMonthlyCashBreakdown(data.monthly_cash_breakdown);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Detailed data error:', error);
                    showToast('error', 'Error loading detailed data');
                }
            });
        }

        // ===== HELPER FUNCTIONS =====
        function updateWidgetValue(selector, value) {
            try {
                $(selector).text(formatCurrency(value));
            } catch (e) {
                console.error('Error updating widget:', selector, e);
            }
        }

        function showToast(type, message) {
            if (typeof toastr !== 'undefined') {
                toastr[type](message);
            } else {
                console.log(type.toUpperCase() + ':', message);
            }
        }

        // ===== RENDER FUNCTIONS (SIMPLIFIED) =====
        function renderSalesAnalysis(salesData) {
            if (!salesData) return;
            
            var html = `
                <div class="row">
                    <div class="col-md-6">
                        <div class="info-box bg-aqua">
                            <span class="info-box-icon"><i class="fa fa-shopping-cart"></i></span>
                            <div class="info-box-content">
                                <span class="info-box-text">Total Sales</span>
                                <span class="info-box-number">${formatNumber(salesData.count || 0)}</span>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="info-box bg-green">
                            <span class="info-box-icon"><i class="fa fa-money"></i></span>
                            <div class="info-box-content">
                                <span class="info-box-text">Total Amount</span>
                                <span class="info-box-number">${formatCurrency(salesData.total || 0)}</span>
                            </div>
                        </div>
                    </div>
                </div>
                <table class="table table-condensed">
                    <tr><td>Subtotal:</td><td class="text-right">${formatCurrency(salesData.subtotal || 0)}</td></tr>
                    <tr><td>Tax:</td><td class="text-right">${formatCurrency(salesData.tax || 0)}</td></tr>
                    <tr><td>Discount:</td><td class="text-right">${formatCurrency(salesData.discount || 0)}</td></tr>
                </table>
            `;
            $('#sales_analysis_content').html(html);
        }

        function renderPurchaseAnalysis(purchaseData) {
            if (!purchaseData) return;
            
            var html = `
                <div class="info-box bg-yellow">
                    <span class="info-box-icon"><i class="fa fa-truck"></i></span>
                    <div class="info-box-content">
                        <span class="info-box-text">Total Purchases</span>
                        <span class="info-box-number">${formatCurrency(purchaseData.total || 0)}</span>
                    </div>
                </div>
            `;
            $('#purchase_analysis_content').html(html);
        }

        function renderTopProducts(products) {
            if (!products || products.length === 0) {
                $('#top_products_content').html('<p class="text-muted">No products sold</p>');
                return;
            }

            var html = '<table class="table table-condensed"><thead><tr><th>Product</th><th>Qty</th><th>Amount</th></tr></thead><tbody>';
            products.forEach(function(product) {
                html += `<tr><td>${product.name}</td><td>${formatNumber(product.quantity_sold)}</td><td>${formatCurrency(product.total_amount)}</td></tr>`;
            });
            html += '</tbody></table>';
            $('#top_products_content').html(html);
        }

        function renderPaymentMethods(paymentMethods) {
            if (!paymentMethods || paymentMethods.length === 0) {
                $('#payment_methods_content').html('<p class="text-muted">No payments</p>');
                return;
            }

            var html = '<table class="table table-condensed"><thead><tr><th>Method</th><th>Amount</th></tr></thead><tbody>';
            paymentMethods.forEach(function(method) {
                html += `<tr><td>${method.method}</td><td>${formatCurrency(method.total_amount)}</td></tr>`;
            });
            html += '</tbody></table>';
            $('#payment_methods_content').html(html);
        }

        function renderExpenseBreakdown(expenses) {
            if (!expenses || expenses.length === 0) {
                $('#expense_breakdown_content').html('<p class="text-muted">No expenses</p>');
                return;
            }

            var html = '<table class="table table-condensed"><thead><tr><th>Category</th><th>Amount</th></tr></thead><tbody>';
            expenses.forEach(function(expense) {
                html += `<tr><td>${expense.category || 'Uncategorized'}</td><td>${formatCurrency(expense.total)}</td></tr>`;
            });
            html += '</tbody></table>';
            $('#expense_breakdown_content').html(html);
        }

        function renderMonthlyCashBreakdown(monthlyData) {
            if (!monthlyData || monthlyData.length === 0) {
                $('#monthly_cash_breakdown_content').html('<p class="text-muted">No monthly data</p>');
                return;
            }

            var html = '<table class="table table-condensed"><thead><tr><th>Month</th><th>Transactions</th><th>Net Flow</th></tr></thead><tbody>';
            monthlyData.forEach(function(month) {
                html += `<tr><td>${month.month_name}</td><td>${month.transaction_count}</td><td>${formatCurrency(month.net_cash_flow)}</td></tr>`;
            });
            html += '</tbody></table>';
            $('#monthly_cash_breakdown_content').html(html);
        }

        // ===== INITIAL LOAD (SAFE) =====
        console.log('Starting initial data load...');
        setTimeout(function() {
            loadAllData();
        }, 500); // Small delay to ensure DOM is ready

        console.log('Daily report initialized successfully');
    });
</script>


@endsection