@extends('advancedreports::layouts.app')

@section('title', __('advancedreports::lang.sales_detail_report'))

@section('content')
<!-- Content Header (Page header) -->
<section class="content-header">
    <h1>{{ __('advancedreports::lang.sales_detail_report') }}</h1>
    <ol class="breadcrumb">
        <li><a href="{{ route('home') }}"><i class="fa fa-dashboard"></i> {{ __('home.home') }}</a></li>
        <li><a href="{{ route('advancedreports.index') }}">{{ __('advancedreports::lang.advanced_reports') }}</a></li>
        <li class="active">{{ __('advancedreports::lang.sales_detail_report') }}</li>
    </ol>
</section>

<!-- Main content -->
<section class="content">
    <div class="sales-detail-container">
        <!-- Report Header -->
        <div class="report-header">
            <div class="business-name" id="business-name">{{ session('business.name', 'BUSINESS NAME') }}</div>
            <div class="report-title">{{ __('advancedreports::lang.sales_detail_report') }}</div>
        </div>

        <!-- Filters Section -->
        <div class="filters-section">
            <form id="sales_detail_filter_form">
                <div class="filters-grid">
                    <!-- Updated Date Range Filter -->
                    <div class="filter-group">
                        <label for="sales_detail_date_filter">{{ __('report.date_range') }}:</label>
                        <div class="input-group">
                            <span class="input-group-addon">
                                <i class="fa fa-calendar"></i>
                            </span>
                            {!! Form::text('date_range', null, [
                            'placeholder' => __('lang_v1.select_a_date_range'),
                            'class' => 'form-control',
                            'id' => 'sales_detail_date_filter',
                            'readonly'
                            ]) !!}
                        </div>
                    </div>

                    <!-- Quick Date Filter Buttons -->
                    <div class="filter-group">
                        <label>&nbsp;</label>
                        <div class="quick-date-buttons">
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="today">
                                {{ __('advancedreports::lang.today') }}
                            </button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="yesterday">
                                {{ __('advancedreports::lang.yesterday') }}
                            </button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="this_week">
                                {{ __('advancedreports::lang.this_week') }}
                            </button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="this_month">
                                {{ __('advancedreports::lang.this_month') }}
                            </button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="last_month">
                                {{ __('advancedreports::lang.last_month') }}
                            </button>
                            <button type="button" class="btn btn-sm btn-default" id="clear_date_filter">
                                {{ __('advancedreports::lang.clear') }}
                            </button>
                        </div>
                    </div>

                    <!-- Existing filters remain the same -->
                    <div class="filter-group">
                        <label for="location_id">{{ __('business.business_location') }}:</label>
                        @php
                        $formatted_locations = is_array($business_locations) && isset($business_locations['locations'])
                        ? $business_locations['locations']
                        : $business_locations;

                        if (is_object($formatted_locations)) {
                        $formatted_locations = $formatted_locations->toArray();
                        }
                        @endphp
                        {!! Form::select('location_id', $formatted_locations, null, [
                        'class' => 'form-control select2',
                        'placeholder' => __('advancedreports::lang.all_locations'),
                        'id' => 'location_id'
                        ]) !!}
                    </div>

                    <div class="filter-group">
                        <label for="user_id">{{ __('advancedreports::lang.staff') }}:</label>
                        @php
                        $formatted_users = is_object($users) ? $users->toArray() : $users;
                        @endphp
                        {!! Form::select('user_id', $formatted_users, null, [
                        'class' => 'form-control select2',
                        'placeholder' => __('advancedreports::lang.all_staff'),
                        'id' => 'user_id'
                        ]) !!}
                    </div>

                    <div class="filter-group">
                        <label for="week_number">{{ __('advancedreports::lang.week') }}:</label>
                        <select id="week_number" name="week_number" class="form-control">
                            <option value="">{{ __('advancedreports::lang.all_weeks') }}</option>
                            <option value="1">{{ __('advancedreports::lang.week_1') }}</option>
                            <option value="2">{{ __('advancedreports::lang.week_2') }}</option>
                            <option value="3">{{ __('advancedreports::lang.week_3') }}</option>
                            <option value="4">{{ __('advancedreports::lang.week_4') }}</option>
                            <option value="5">{{ __('advancedreports::lang.week_5') }}</option>
                        </select>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-12">
                        <div class="form-group">
                            <button type="button" class="btn btn-primary" id="apply-filters">
                                <i class="fa fa-filter"></i> {{ __('advancedreports::lang.apply_filters') }}
                            </button>
                            <button type="button" class="btn btn-default" id="clear-filters">
                                <i class="fa fa-refresh"></i> {{ __('advancedreports::lang.clear') }}
                            </button>
                            <button type="button" class="btn btn-success" id="export-report">
                                <i class="fa fa-download"></i> {{ __('lang_v1.export') }}
                            </button>
                            <button type="button" class="btn btn-info" id="refresh-report">
                                <i class="fa fa-refresh"></i> {{ __('lang_v1.refresh') }}
                            </button>
                            <div class="checkbox" style="display: inline-block; margin-left: 10px;">
                                <label>
                                    <input type="checkbox" id="auto_refresh"> {{
                                    __('advancedreports::lang.auto_refresh_5_min') }}
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Weekly Totals -->
        <div class="weekly-totals">
            <h3>{{ __('advancedreports::lang.weekly_totals') }}</h3>
            <div class="totals-grid">
                <div class="total-item">
                    <div class="total-label">{{ __('advancedreports::lang.total_sales') }}</div>
                    <div class="total-value" id="total-sales">$0.00</div>
                </div>
                <div class="total-item">
                    <div class="total-label">{{ __('advancedreports::lang.total_purchase') }}</div>
                    <div class="total-value" id="total-purchase">$0.00</div>
                </div>
                <div class="total-item">
                    <div class="total-label">{{ __('advancedreports::lang.total_profit') }}</div>
                    <div class="total-value" id="total-profit">$0.00</div>
                </div>
                <div class="total-item">
                    <div class="total-label">{{ __('advancedreports::lang.profit_margin') }}</div>
                    <div class="total-value" id="profit-margin">0.00%</div>
                </div>
            </div>
        </div>

        <!-- Sales Detail Table -->
        <div class="table-container">
            <table class="sales-table" id="sales-table">
                <thead>
                    <tr>
                        <th class="col-product">{{ __('business.product') }}</th>
                        <th class="col-invoice">{{ __('advancedreports::lang.invoice_no') }}</th>
                        <th class="col-date">{{ __('advancedreports::lang.sales_date') }}</th>
                        <th class="col-week">{{ __('advancedreports::lang.week_number') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.sales_unit') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.qty_sold') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.selling_price') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.purchase_price') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.total_sales_amt') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.total_purchase_amt') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.profit_earned') }}</th>
                        <th class="col-numeric">{{ __('advancedreports::lang.margin_percent') }}</th>
                        <th class="col-user">{{ __('advancedreports::lang.staff') }}</th>
                        <th class="col-day">{{ __('advancedreports::lang.day_number') }}</th>
                    </tr>
                </thead>
                <tbody id="sales-table-body">
                    <tr>
                        <td colspan="14" class="loading">
                            <i class="fa fa-spinner fa-spin"></i>
                            <div>{{ __('advancedreports::lang.loading_sales_data') }}</div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</section>
@endsection

@section('css')
<style>
    /* Sales Detail Report Styles */
    .sales-detail-container {
        background: white;
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        overflow: hidden;
        margin: 20px 0;
    }

    /* Header Styles */
    .report-header {
        background: linear-gradient(135deg, #ff6b35, #f7931e);
        color: white;
        padding: 20px;
        text-align: center;
    }

    .business-name {
        font-size: 24px;
        font-weight: bold;
        margin-bottom: 5px;
    }

    .report-title {
        font-size: 18px;
        font-weight: bold;
    }

    /* Filters Section */
    .filters-section {
        padding: 20px;
        background: #f8f9fa;
        border-bottom: 1px solid #dee2e6;
    }

    .filters-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 15px;
        margin-bottom: 15px;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
    }

    .filter-group label {
        font-weight: 500;
        margin-bottom: 5px;
        color: #495057;
    }

    .filter-buttons {
        display: flex;
        gap: 10px;
        justify-content: center;
        flex-wrap: wrap;
    }

    /* Weekly Totals */
    .weekly-totals {
        background: #e7f3ff;
        padding: 15px 20px;
        border-bottom: 1px solid #dee2e6;
        display: flex;
        justify-content: space-between;
        align-items: center;
        flex-wrap: wrap;
        gap: 15px;
    }

    .weekly-totals h3 {
        color: #495057;
        font-size: 16px;
        margin: 0;
    }

    .totals-grid {
        display: flex;
        gap: 30px;
        flex-wrap: wrap;
    }

    .total-item {
        text-align: center;
    }

    .total-label {
        font-size: 12px;
        color: #6c757d;
        margin-bottom: 2px;
    }

    .total-value {
        font-size: 16px;
        font-weight: bold;
        color: #495057;
    }

    /* Table Styles */
    .table-container {
        overflow-x: auto;
        max-height: 600px;
        overflow-y: auto;
    }

    .sales-table {
        width: 100%;
        border-collapse: collapse;
        font-size: 13px;
    }

    .sales-table th {
        background: #495057;
        color: white;
        padding: 12px 8px;
        text-align: left;
        font-weight: 600;
        position: sticky;
        top: 0;
        z-index: 10;
        border-bottom: 2px solid #343a40;
    }

    .sales-table td {
        padding: 10px 8px;
        border-bottom: 1px solid #dee2e6;
        vertical-align: middle;
    }

    .sales-table tbody tr:nth-child(even) {
        background: #f8f9fa;
    }

    .sales-table tbody tr:hover {
        background: #e3f2fd;
    }

    /* Column specific styles */
    .col-product {
        min-width: 200px;
        font-weight: 500;
    }

    .col-invoice {
        min-width: 80px;
        text-align: center;
    }

    .col-date {
        min-width: 120px;
        text-align: center;
    }

    .col-week {
        min-width: 70px;
        text-align: center;
        font-weight: bold;
    }

    .col-numeric {
        min-width: 80px;
        text-align: right;
        font-family: monospace;
    }

    .col-user {
        min-width: 100px;
    }

    .col-day {
        min-width: 60px;
        text-align: center;
    }

    /* Week-specific styling */
    .week-1 {
        background-color: #fff3cd !important;
    }

    .week-2 {
        background-color: #d4edda !important;
    }

    .week-3 {
        background-color: #d1ecf1 !important;
    }

    .week-4 {
        background-color: #f8d7da !important;
    }

    .week-5 {
        background-color: #e2e3e5 !important;
    }

    /* Profit margin styling */
    .profit-positive {
        color: #28a745;
        font-weight: bold;
    }

    .profit-negative {
        color: #dc3545;
        font-weight: bold;
    }

    /* Loading state */
    .loading {
        text-align: center;
        padding: 40px;
        color: #6c757d;
    }

    .loading i {
        font-size: 24px;
        margin-bottom: 10px;
    }

    /* Responsive */
    @media (max-width: 768px) {
        .filters-grid {
            grid-template-columns: 1fr;
        }

        .totals-grid {
            flex-direction: column;
            gap: 10px;
        }

        .sales-table {
            font-size: 12px;
        }

        .sales-table th,
        .sales-table td {
            padding: 8px 4px;
        }
    }

    /* Additional CSS for quick date buttons - add to your existing styles */

    .quick-date-buttons {
        display: flex;
        flex-wrap: wrap;
        gap: 5px;
    }

    .quick-date-buttons .btn {
        margin-bottom: 5px;
    }

    .quick-date.active {
        background-color: #337ab7 !important;
        border-color: #2e6da4 !important;
        color: white !important;
    }

    .filter-group {
        margin-bottom: 15px;
    }

    /* Responsive adjustments for quick date buttons */
    @media (max-width: 768px) {
        .quick-date-buttons {
            flex-direction: column;
        }

        .quick-date-buttons .btn {
            width: 100%;
            margin-bottom: 5px;
        }
    }
</style>
@endsection

@section('javascript')
<script>
    $(document).ready(function() {
    // Initialize date range picker
    var dateRangeSettings = {
        startDate: moment().startOf('month'),
        endDate: moment().endOf('month'),
        ranges: {
            'Today': [moment(), moment()],
            'Yesterday': [moment().subtract(1, 'days'), moment().subtract(1, 'days')],
            'Last 7 Days': [moment().subtract(6, 'days'), moment()],
            'Last 30 Days': [moment().subtract(29, 'days'), moment()],
            'This Month': [moment().startOf('month'), moment().endOf('month')],
            'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')]
        },
        locale: {
            format: moment_date_format || 'DD/MM/YYYY'
        }
    };

    $('#sales_detail_date_filter').daterangepicker(
        dateRangeSettings,
        function(start, end) {
            $('#sales_detail_date_filter').val(
                start.format(moment_date_format) + ' ~ ' + end.format(moment_date_format)
            );
            loadSalesDetailData();
            loadWeeklySummary();
        }
    );

    $('#sales_detail_date_filter').on('cancel.daterangepicker', function(ev, picker) {
        $('#sales_detail_date_filter').val('');
        loadSalesDetailData();
        loadWeeklySummary();
    });

    // Helper function for currency formatting
    function formatCurrency(amount) {
        if (typeof __currency_trans_from_en === 'function') {
            return __currency_trans_from_en(amount, true);
        }
        return '$' + parseFloat(amount || 0).toFixed(2);
    }

    // Initialize select2
    $('.select2').select2();

    // Quick date filters
    $('.quick-date').click(function() {
        var range = $(this).data('range');
        var start, end;
        
        switch(range) {
            case 'today':
                start = end = moment();
                break;
            case 'yesterday':
                start = end = moment().subtract(1, 'day');
                break;
            case 'this_week':
                start = moment().startOf('week');
                end = moment().endOf('week');
                break;
            case 'last_week':
                start = moment().subtract(1, 'week').startOf('week');
                end = moment().subtract(1, 'week').endOf('week');
                break;
            case 'this_month':
                start = moment().startOf('month');
                end = moment().endOf('month');
                break;
            case 'last_month':
                start = moment().subtract(1, 'month').startOf('month');
                end = moment().subtract(1, 'month').endOf('month');
                break;
            case 'this_quarter':
                start = moment().startOf('quarter');
                end = moment().endOf('quarter');
                break;
            case 'this_year':
                start = moment().startOf('year');
                end = moment().endOf('year');
                break;
        }
        
        $('#sales_detail_date_filter').data('daterangepicker').setStartDate(start);
        $('#sales_detail_date_filter').data('daterangepicker').setEndDate(end);
        $('#sales_detail_date_filter').val(start.format(moment_date_format) + ' ~ ' + end.format(moment_date_format));
        
        // Highlight active quick date button
        $('.quick-date').removeClass('active btn-primary').addClass('btn-default');
        $(this).removeClass('btn-default').addClass('btn-primary active');
        
        loadSalesDetailData();
        loadWeeklySummary();
    });

    // Clear date filter
    $('#clear_date_filter').click(function() {
        $('#sales_detail_date_filter').val('');
        $('.quick-date').removeClass('active btn-primary').addClass('btn-default');
        loadSalesDetailData();
        loadWeeklySummary();
    });

    // Load initial data
    loadSalesDetailData();
    loadWeeklySummary();

    // Apply filters
    $('#apply-filters').click(function() {
        var $btn = $(this);
        $btn.html('<i class="fa fa-spinner fa-spin"></i> Loading...').prop('disabled', true);
        
        loadSalesDetailData();
        loadWeeklySummary();
        
        setTimeout(function() {
            $btn.html('<i class="fa fa-filter"></i> Apply Filters').prop('disabled', false);
        }, 1000);
    });

    // Clear filters
    $('#clear-filters').click(function() {
        var $btn = $(this);
        $btn.html('<i class="fa fa-spinner fa-spin"></i> Loading...').prop('disabled', true);
        
        $('#sales_detail_filter_form')[0].reset();
        $('.select2').val(null).trigger('change');
        $('#sales_detail_date_filter').val('');
        $('.quick-date').removeClass('active btn-primary').addClass('btn-default');
        
        loadSalesDetailData();
        loadWeeklySummary();
        
        setTimeout(function() {
            $btn.html('<i class="fa fa-refresh"></i> Clear').prop('disabled', false);
        }, 1000);
    });

    // Refresh report
    $('#refresh-report').click(function() {
        var $btn = $(this);
        $btn.html('<i class="fa fa-spinner fa-spin"></i> Loading...').prop('disabled', true);
        
        loadSalesDetailData();
        loadWeeklySummary();
        
        if (typeof toastr !== 'undefined') {
            toastr.success('Report refreshed successfully');
        }
        
        setTimeout(function() {
            $btn.html('<i class="fa fa-refresh"></i> Refresh').prop('disabled', false);
        }, 1000);
    });

    // Export functionality
    $('#export-report').click(function() {
        var $btn = $(this);
        $btn.html('<i class="fa fa-spinner fa-spin"></i> Loading...').prop('disabled', true);
        
        var params = getFilterData();
        var url = '{{ route("advancedreports.sales-detail.export") }}';
        var queryString = $.param(params);
        
        setTimeout(function() {
            window.open(url + '?' + queryString, '_blank');
            $btn.html('<i class="fa fa-download"></i> Export').prop('disabled', false);
        }, 2000);
    });

    // Auto-refresh every 5 minutes
    setInterval(function() {
        if ($('#auto_refresh').is(':checked')) {
            loadSalesDetailData();
            loadWeeklySummary();
        }
    }, 300000);

    // Load sales detail data
    function loadSalesDetailData() {
        $.ajax({
            url: '{{ route("advancedreports.sales-detail.data") }}',
            type: 'GET',
            data: getFilterData(),
            beforeSend: function() {
                $('#sales-table-body').html(`
                    <tr>
                        <td colspan="14" class="loading">
                            <i class="fa fa-spinner fa-spin"></i>
                            <div>Loading sales data...</div>
                        </td>
                    </tr>
                `);
            },
            success: function(response) {
                if (response.success && response.data && response.data.length > 0) {
                    renderSalesTable(response.data);
                } else {
                    $('#sales-table-body').html(`
                        <tr>
                            <td colspan="14" style="text-align: center; padding: 40px; color: #6c757d;">
                                No sales data found for the selected filters
                            </td>
                        </tr>
                    `);
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading sales data:', error);
                $('#sales-table-body').html(`
                    <tr>
                        <td colspan="14" style="text-align: center; padding: 40px; color: #dc3545;">
                            Error loading sales data
                        </td>
                    </tr>
                `);
            }
        });
    }

    // Load weekly summary
    function loadWeeklySummary() {
        $.ajax({
            url: '{{ route("advancedreports.sales-detail.summary") }}',
            type: 'GET',
            data: getFilterData(),
            success: function(response) {
                $('#total-sales').text(formatCurrency(response.total_sales));
                $('#total-purchase').text(formatCurrency(response.total_purchase));
                $('#total-profit').text(formatCurrency(response.total_profit));
                $('#profit-margin').text(response.profit_margin + '%');
            },
            error: function() {
                $('#total-sales, #total-purchase, #total-profit').text('$0.00');
                $('#profit-margin').text('0.00%');
            }
        });
    }

    // Render sales table
    function renderSalesTable(data) {
        const rows = data.map(item => {
            const profitClass = item.profit_earned >= 0 ? 'profit-positive' : 'profit-negative';
            const weekClass = `week-${item.week_number}`;
            
            return `
                <tr class="${weekClass}">
                    <td class="col-product">${item.product_name}</td>
                    <td class="col-invoice">${item.invoice_no}</td>
                    <td class="col-date">${item.sales_date_formatted}</td>
                    <td class="col-week">${item.week_number}</td>
                    <td class="col-numeric">${item.sales_unit}</td>
                    <td class="col-numeric">${item.qty_sold}</td>
                    <td class="col-numeric">${formatCurrency(item.selling_price)}</td>
                    <td class="col-numeric">${formatCurrency(item.purchase_price)}</td>
                    <td class="col-numeric">${formatCurrency(item.total_sales_amt)}</td>
                    <td class="col-numeric">${formatCurrency(item.total_purchase_amt)}</td>
                    <td class="col-numeric ${profitClass}">${formatCurrency(item.profit_earned)}</td>
                    <td class="col-numeric ${profitClass}">${item.margin_percent}%</td>
                    <td class="col-user">${item.user_name}</td>
                    <td class="col-day">${item.day_number}</td>
                </tr>
            `;
        }).join('');

        $('#sales-table-body').html(rows);
    }

    // Get filter data
    function getFilterData() {
        var start = '';
        var end = '';
        
        if ($('#sales_detail_date_filter').val()) {
            start = $('#sales_detail_date_filter').data('daterangepicker').startDate.format('YYYY-MM-DD');
            end = $('#sales_detail_date_filter').data('daterangepicker').endDate.format('YYYY-MM-DD');
        }

        return {
            start_date: start,
            end_date: end,
            location_id: $('#location_id').val(),
            user_id: $('#user_id').val(),
            week_number: $('#week_number').val()
        };
    }

    // Show success message
    if (typeof toastr !== 'undefined') {
        toastr.info('Sales detail report loaded');
    }
});
</script>
@endsection