<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (! Schema::hasTable('mosque_member_fees')) {
            return;
        }

        Schema::table('mosque_member_fees', function (Blueprint $table) {
            if (! Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                $table->enum('lifecycle_status', ['due', 'paid', 'cancelled'])->default('due')->index();
            }
            if (! Schema::hasColumn('mosque_member_fees', 'cancel_reason')) {
                $table->text('cancel_reason')->nullable();
            }
            if (! Schema::hasColumn('mosque_member_fees', 'cancelled_by')) {
                $table->unsignedBigInteger('cancelled_by')->nullable()->index();
            }
            if (! Schema::hasColumn('mosque_member_fees', 'cancelled_at')) {
                $table->dateTime('cancelled_at')->nullable()->index();
            }
            if (! Schema::hasColumn('mosque_member_fees', 'period_type')) {
                $table->enum('period_type', ['monthly', 'yearly'])->nullable()->index();
            }
        });

        // Backfill lifecycle_status from legacy status (pending/partial/paid) where possible.
        if (Schema::hasColumn('mosque_member_fees', 'status') && Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
            DB::table('mosque_member_fees')
                ->whereNull('cancelled_at')
                ->update([
                    'lifecycle_status' => DB::raw("CASE WHEN status = 'paid' THEN 'paid' ELSE 'due' END"),
                ]);
        }

        // Backfill period_type from plan type when possible.
        if (
            Schema::hasTable('mosque_membership_plans')
            && Schema::hasColumn('mosque_member_fees', 'period_type')
            && Schema::hasColumn('mosque_member_fees', 'plan_id')
        ) {
            DB::statement(
                "UPDATE mosque_member_fees f
                 JOIN mosque_membership_plans p
                   ON p.id = f.plan_id AND p.business_id = f.business_id
                 SET f.period_type = COALESCE(f.period_type, p.type)
                 WHERE f.period_type IS NULL"
            );
        }

        // Fallback inference for any remaining rows.
        if (Schema::hasColumn('mosque_member_fees', 'period_type') && Schema::hasColumn('mosque_member_fees', 'period_ym')) {
            // Yearly: period like 'YYYY'
            DB::table('mosque_member_fees')
                ->whereNull('period_type')
                ->whereRaw("period_ym REGEXP '^[0-9]{4}$'")
                ->update(['period_type' => 'yearly']);

            // Monthly: any remaining (including legacy YYYY-MM and YYYY-01 yearly placeholders without plan)
            DB::table('mosque_member_fees')
                ->whereNull('period_type')
                ->update(['period_type' => 'monthly']);
        }
    }

    public function down(): void
    {
        // Non-destructive: do not drop columns.
    }
};

