<?php

namespace Modules\Mosque\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueAsset;
use Modules\Mosque\Entities\MosqueAssetMaintenance;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueProperty;
use Modules\Mosque\Entities\MosqueRental;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class AssetsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.assets.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        return view('mosque::assets.index');
    }

    // Assets
    public function assetsData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueAsset::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'tag_no', 'acquired_on', 'value', 'status']);

        return DataTables::of($query)
            ->editColumn('value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->value.'">'.$row->value.'</span>';
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'assetsEdit'], [$row->id], false).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_asset_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'assetsDestroy'], [$row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'asset\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}var jq=window.jQuery||window.$;if(jq){jq(\'#mosque_delete_confirm_url\').val(url);jq(\'#mosque_delete_confirm_method\').val(\'DELETE\');jq(\'#mosque_delete_confirm_entity\').val(ent);jq(\'#mosque_delete_confirm_input\').val(\'\');jq(\'#mosque_delete_confirm_btn\').prop(\'disabled\',true);jq(\'#mosque_delete_confirm_modal\').modal(\'show\');return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_asset"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action', 'value'])
            ->make(true);
    }

    public function assetsCreate()
    {
        $this->ensurePermission();
        return view('mosque::assets.assets.create');
    }

    public function assetsStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'tag_no' => 'nullable|string|max:100',
            'acquired_on' => 'nullable|date',
            'value' => 'nullable|numeric|min:0',
            'status' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            MosqueAsset::query()->create(array_merge(
                $request->only(['name', 'tag_no', 'acquired_on', 'value', 'status', 'notes']),
                ['business_id' => $businessId]
            ));
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function assetsEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $asset = MosqueAsset::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::assets.assets.edit', compact('asset'));
    }

    public function assetsUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'tag_no' => 'nullable|string|max:100',
            'acquired_on' => 'nullable|date',
            'value' => 'nullable|numeric|min:0',
            'status' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $asset = MosqueAsset::query()->where('business_id', $businessId)->findOrFail($id);
            $asset->update($request->only(['name', 'tag_no', 'acquired_on', 'value', 'status', 'notes']));
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function assetsDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $asset = MosqueAsset::query()->where('business_id', $businessId)->find($id);
            if (! $asset) {
                return ['success' => true, 'msg' => __('lang_v1.success')];
            }
            $asset->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'asset', (int) $asset->id, [
                'name' => (string) $asset->name,
                'tag_no' => (string) ($asset->tag_no ?? ''),
                'status' => (string) ($asset->status ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'asset', (int) $asset->id, [
                'name' => (string) $asset->name,
                'tag_no' => (string) ($asset->tag_no ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Maintenance
    public function maintenanceData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_asset_maintenance as am')
            ->join('mosque_assets as a', function ($join) use ($businessId) {
                $join->on('a.id', '=', 'am.asset_id')
                    ->where('a.business_id', '=', $businessId);
            })
            ->where('am.business_id', $businessId)
            ->whereNull('am.deleted_at')
            ->select([
                'am.id',
                'a.name as asset_name',
                'am.date',
                'am.cost',
                'am.remark',
            ])
            ->orderByDesc('am.date');

        if (! empty($request->input('asset_id'))) {
            $query->where('am.asset_id', $request->input('asset_id'));
        }

        return DataTables::of($query)
            ->editColumn('cost', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->cost.'">'.$row->cost.'</span>';
            })
            ->addColumn('action', function ($row) {
                $deleteUrl = action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'maintenanceDestroy'], [$row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'maintenance\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}var jq=window.jQuery||window.$;if(jq){jq(\'#mosque_delete_confirm_url\').val(url);jq(\'#mosque_delete_confirm_method\').val(\'DELETE\');jq(\'#mosque_delete_confirm_entity\').val(ent);jq(\'#mosque_delete_confirm_input\').val(\'\');jq(\'#mosque_delete_confirm_btn\').prop(\'disabled\',true);jq(\'#mosque_delete_confirm_modal\').modal(\'show\');return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_maintenance"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $delete;
            })
            ->rawColumns(['action', 'cost'])
            ->make(true);
    }

    public function maintenanceCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $assets = MosqueAsset::query()->where('business_id', $businessId)->orderBy('name')->pluck('name', 'id');
        return view('mosque::assets.maintenance.create', compact('assets'));
    }

    public function maintenanceStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'asset_id' => 'required|integer',
            'date' => 'required|date',
            'cost' => 'required|numeric|min:0',
            'remark' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $maintenance = MosqueAssetMaintenance::query()->create(array_merge(
                $request->only(['asset_id', 'date', 'cost', 'remark']),
                ['business_id' => $businessId]
            ));

            $cost = (float) ($maintenance->cost ?? 0);
            if ($cost > 0) {
                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'expense', 'name' => 'Maintenance'],
                    ['active' => true, 'sort_order' => 3]
                );

                MosqueFinanceEntry::query()->create([
                    'business_id' => $businessId,
                    'location_id' => null,
                    'type' => 'expense',
                    'category_id' => $category->id,
                    'amount' => $cost,
                    'entry_date' => $maintenance->date,
                    'ref_module' => 'asset_maintenance',
                    'ref_id' => $maintenance->id,
                    'fund_tag' => null,
                    'note' => 'Asset maintenance',
                    'created_by' => auth()->id(),
                ]);
            }

            MosqueAuditUtil::log($businessId, 'create', 'asset_maintenance', (int) $maintenance->id, [
                'asset_id' => (int) $maintenance->asset_id,
                'cost' => (float) $maintenance->cost,
            ]);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function maintenanceDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $m = MosqueAssetMaintenance::query()->where('business_id', $businessId)->find($id);
            if (! $m) {
                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'asset_maintenance')
                    ->where('ref_id', $id)
                    ->delete();

                return ['success' => true, 'msg' => __('lang_v1.success')];
            }
            $m->delete();

            MosqueFinanceEntry::query()
                ->where('business_id', $businessId)
                ->where('ref_module', 'asset_maintenance')
                ->where('ref_id', $id)
                ->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'asset_maintenance', (int) $id, [
                'asset_id' => (int) $m->asset_id,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'asset maintenance', (int) $id, [
                'asset_id' => (int) $m->asset_id,
                'date' => (string) ($m->date ?? ''),
                'cost' => (float) ($m->cost ?? 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Properties
    public function propertiesData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueProperty::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'address', 'rentable']);

        return DataTables::of($query)
            ->editColumn('rentable', function ($row) {
                return $row->rentable ? __('lang_v1.yes') : __('lang_v1.no');
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'propertiesEdit'], [$row->id], false).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_property_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'propertiesDestroy'], [$row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'property\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}var jq=window.jQuery||window.$;if(jq){jq(\'#mosque_delete_confirm_url\').val(url);jq(\'#mosque_delete_confirm_method\').val(\'DELETE\');jq(\'#mosque_delete_confirm_entity\').val(ent);jq(\'#mosque_delete_confirm_input\').val(\'\');jq(\'#mosque_delete_confirm_btn\').prop(\'disabled\',true);jq(\'#mosque_delete_confirm_modal\').modal(\'show\');return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_property"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function propertiesCreate()
    {
        $this->ensurePermission();
        return view('mosque::assets.properties.create');
    }

    public function propertiesStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string',
            'rentable' => 'nullable|boolean',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            MosqueProperty::query()->create([
                'business_id' => $businessId,
                'name' => $request->input('name'),
                'address' => $request->input('address'),
                'rentable' => (bool) $request->input('rentable', false),
                'notes' => $request->input('notes'),
            ]);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function propertiesEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $property = MosqueProperty::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::assets.properties.edit', compact('property'));
    }

    public function propertiesUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'address' => 'nullable|string',
            'rentable' => 'nullable|boolean',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $property = MosqueProperty::query()->where('business_id', $businessId)->findOrFail($id);
            $property->update([
                'name' => $request->input('name'),
                'address' => $request->input('address'),
                'rentable' => (bool) $request->input('rentable', false),
                'notes' => $request->input('notes'),
            ]);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function propertiesDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $property = MosqueProperty::query()->where('business_id', $businessId)->find($id);
            if (! $property) {
                return ['success' => true, 'msg' => __('lang_v1.success')];
            }
            $property->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'property', (int) $property->id, [
                'name' => (string) $property->name,
                'rentable' => (int) ($property->rentable ? 1 : 0),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'property', (int) $property->id, [
                'name' => (string) $property->name,
                'rentable' => (int) ($property->rentable ? 1 : 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Rentals
    public function rentalsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_rentals as r')
            ->join('mosque_properties as p', function ($join) use ($businessId) {
                $join->on('p.id', '=', 'r.property_id')
                    ->where('p.business_id', '=', $businessId);
            })
            ->where('r.business_id', $businessId)
            ->whereNull('r.deleted_at')
            ->select([
                'r.id',
                'p.name as property_name',
                'r.tenant_name',
                'r.from_date',
                'r.to_date',
                'r.rent_amount',
                'r.paid_amount',
                'r.status',
            ])
            ->orderByDesc('r.from_date');

        if (! empty($request->input('property_id'))) {
            $query->where('r.property_id', $request->input('property_id'));
        }

        return DataTables::of($query)
            ->editColumn('rent_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->rent_amount.'">'.$row->rent_amount.'</span>';
            })
            ->editColumn('paid_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->paid_amount.'">'.$row->paid_amount.'</span>';
            })
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'rentalsPrint'], [$row->id], false).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $pdf = '<a href="'.action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'rentalsPdf'], [$row->id], false).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'rentalsEdit'], [$row->id], false).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_rental_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = action([\Modules\Mosque\Http\Controllers\AssetsController::class, 'rentalsDestroy'], [$row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'rental\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}var jq=window.jQuery||window.$;if(jq){jq(\'#mosque_delete_confirm_url\').val(url);jq(\'#mosque_delete_confirm_method\').val(\'DELETE\');jq(\'#mosque_delete_confirm_entity\').val(ent);jq(\'#mosque_delete_confirm_input\').val(\'\');jq(\'#mosque_delete_confirm_btn\').prop(\'disabled\',true);jq(\'#mosque_delete_confirm_modal\').modal(\'show\');return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_rental"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $print.' '.$pdf.' '.$edit.' '.$delete;
            })
            ->rawColumns(['action', 'rent_amount', 'paid_amount'])
            ->make(true);
    }

    public function rentalsCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $properties = MosqueProperty::query()->where('business_id', $businessId)->orderBy('name')->pluck('name', 'id');
        return view('mosque::assets.rentals.create', compact('properties'));
    }

    public function rentalsStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'property_id' => 'required|integer',
            'tenant_name' => 'required|string|max:255',
            'from_date' => 'required|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
            'rent_amount' => 'required|numeric|min:0',
            'paid_amount' => 'nullable|numeric|min:0',
            'status' => 'nullable|string|max:50',
        ]);

        try {
            $businessId = $this->businessId();
            $rental = MosqueRental::query()->create(array_merge(
                $request->only(['property_id', 'tenant_name', 'from_date', 'to_date', 'rent_amount', 'paid_amount', 'status']),
                ['business_id' => $businessId]
            ));

            $paidAmount = (float) ($rental->paid_amount ?? 0);
            if ($paidAmount > 0) {
                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'income', 'name' => 'Hall/property rent'],
                    ['active' => true, 'sort_order' => 10]
                );

                MosqueFinanceEntry::query()->create([
                    'business_id' => $businessId,
                    'location_id' => null,
                    'type' => 'income',
                    'category_id' => $category->id,
                    'amount' => $paidAmount,
                    'entry_date' => $rental->from_date,
                    'ref_module' => 'rental',
                    'ref_id' => $rental->id,
                    'fund_tag' => null,
                    'note' => 'Rental: '.$rental->tenant_name,
                    'created_by' => auth()->id(),
                ]);
            }

            MosqueAuditUtil::log($businessId, 'create', 'rental', (int) $rental->id, [
                'property_id' => (int) $rental->property_id,
                'paid_amount' => (float) ($rental->paid_amount ?? 0),
            ]);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function rentalsEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $rental = MosqueRental::query()->where('business_id', $businessId)->findOrFail($id);
        $properties = MosqueProperty::query()->where('business_id', $businessId)->orderBy('name')->pluck('name', 'id');
        return view('mosque::assets.rentals.edit', compact('rental', 'properties'));
    }

    public function rentalsUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'property_id' => 'required|integer',
            'tenant_name' => 'required|string|max:255',
            'from_date' => 'required|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
            'rent_amount' => 'required|numeric|min:0',
            'paid_amount' => 'nullable|numeric|min:0',
            'status' => 'nullable|string|max:50',
        ]);

        try {
            $businessId = $this->businessId();
            $rental = MosqueRental::query()->where('business_id', $businessId)->findOrFail($id);
            $rental->update($request->only(['property_id', 'tenant_name', 'from_date', 'to_date', 'rent_amount', 'paid_amount', 'status']));

            $paidAmount = (float) ($rental->paid_amount ?? 0);
            $existing = MosqueFinanceEntry::query()
                ->where('business_id', $businessId)
                ->where('ref_module', 'rental')
                ->where('ref_id', $rental->id)
                ->first();

            if ($paidAmount > 0) {
                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'income', 'name' => 'Hall/property rent'],
                    ['active' => true, 'sort_order' => 10]
                );

                if ($existing) {
                    $existing->update([
                        'category_id' => $category->id,
                        'amount' => $paidAmount,
                        'entry_date' => $rental->from_date,
                        'note' => 'Rental: '.$rental->tenant_name,
                    ]);
                } else {
                    MosqueFinanceEntry::query()->create([
                        'business_id' => $businessId,
                        'location_id' => null,
                        'type' => 'income',
                        'category_id' => $category->id,
                        'amount' => $paidAmount,
                        'entry_date' => $rental->from_date,
                        'ref_module' => 'rental',
                        'ref_id' => $rental->id,
                        'fund_tag' => null,
                        'note' => 'Rental: '.$rental->tenant_name,
                        'created_by' => auth()->id(),
                    ]);
                }
            } else {
                if ($existing) {
                    $existing->delete();
                }
            }

            MosqueAuditUtil::log($businessId, 'update', 'rental', (int) $rental->id, [
                'property_id' => (int) $rental->property_id,
                'paid_amount' => (float) ($rental->paid_amount ?? 0),
            ]);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function rentalsDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $rental = MosqueRental::query()->where('business_id', $businessId)->find($id);
            if (! $rental) {
                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'rental')
                    ->where('ref_id', $id)
                    ->delete();

                return ['success' => true, 'msg' => __('lang_v1.success')];
            }
            $rental->delete();

            MosqueFinanceEntry::query()
                ->where('business_id', $businessId)
                ->where('ref_module', 'rental')
                ->where('ref_id', $rental->id)
                ->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'rental', (int) $rental->id, [
                'property_id' => (int) $rental->property_id,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'rental', (int) $rental->id, [
                'property_id' => (int) $rental->property_id,
                'tenant_name' => (string) ($rental->tenant_name ?? ''),
                'from_date' => (string) ($rental->from_date ?? ''),
                'paid_amount' => (float) ($rental->paid_amount ?? 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function rentalsPrint($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $rental = MosqueRental::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $property = MosqueProperty::query()
            ->where('business_id', $businessId)
            ->findOrFail($rental->property_id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'print', 'rental', (int) $rental->id, [
            'property_id' => $property->id,
            'tenant_name' => $rental->tenant_name,
        ]);

        return view('mosque::assets.rentals.print', compact('rental', 'property', 'profile', 'logoDataUri', 'settings'));
    }

    public function rentalsPdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $rental = MosqueRental::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $property = MosqueProperty::query()
            ->where('business_id', $businessId)
            ->findOrFail($rental->property_id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'pdf', 'rental', (int) $rental->id, [
            'property_id' => $property->id,
            'tenant_name' => $rental->tenant_name,
        ]);

        $pdf = Pdf::loadView('mosque::assets.rentals.print', compact('rental', 'property', 'profile', 'logoDataUri', 'settings'))
            ->setPaper('a4');

        $filename = 'rental_'.$property->id.'_'.$rental->id.'.pdf';

        return $pdf->download($filename);
    }
}
