<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Yajra\DataTables\Facades\DataTables;

class AssetsReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.assets.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function assetsStatusData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_assets as a')
            ->where('a.business_id', $businessId)
            ->whereNull('a.deleted_at')
            ->select([
                'a.status',
                DB::raw('COUNT(a.id) as assets_count'),
                DB::raw('SUM(COALESCE(a.value, 0)) as total_value'),
            ])
            ->groupBy('a.status');

        return DataTables::of($query)
            ->editColumn('total_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->total_value.'">'.$row->total_value.'</span>';
            })
            ->rawColumns(['total_value'])
            ->make(true);
    }

    public function maintenanceSummaryData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_asset_maintenance as m')
            ->join('mosque_assets as a', function ($join) use ($businessId) {
                $join->on('a.id', '=', 'm.asset_id')
                    ->where('a.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_assets', 'deleted_at')) {
                    $join->whereNull('a.deleted_at');
                }
            })
            ->where('m.business_id', $businessId)
            ->whereNull('m.deleted_at')
            ->select([
                'a.name as asset_name',
                'a.tag_no',
                DB::raw('COUNT(m.id) as jobs_count'),
                DB::raw('SUM(COALESCE(m.cost, 0)) as total_cost'),
                DB::raw('MAX(m.date) as last_maintenance_date'),
            ])
            ->groupBy('a.name', 'a.tag_no');

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('m.date', '>=', $request->input('start_date'))
                ->whereDate('m.date', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->editColumn('total_cost', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->total_cost.'">'.$row->total_cost.'</span>';
            })
            ->editColumn('last_maintenance_date', function ($row) {
                return ! empty($row->last_maintenance_date) ? \Carbon\Carbon::parse($row->last_maintenance_date)->format('Y-m-d') : '';
            })
            ->rawColumns(['total_cost'])
            ->make(true);
    }

    public function rentalsIncomeData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_rentals as r')
            ->leftJoin('mosque_properties as p', function ($join) use ($businessId) {
                $join->on('p.id', '=', 'r.property_id')
                    ->where('p.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_properties', 'deleted_at')) {
                    $join->whereNull('p.deleted_at');
                }
            })
            ->where('r.business_id', $businessId)
            ->whereNull('r.deleted_at')
            ->select([
                DB::raw("DATE_FORMAT(r.from_date, '%Y-%m') as month_ym"),
                'p.name as property_name',
                DB::raw('COUNT(r.id) as rentals_count'),
                DB::raw('SUM(COALESCE(r.rent_amount, 0)) as rent_total'),
                DB::raw('SUM(COALESCE(r.paid_amount, 0)) as paid_total'),
            ])
            ->groupBy(DB::raw("DATE_FORMAT(r.from_date, '%Y-%m')"), 'p.name');

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('r.from_date', '>=', $request->input('start_date'))
                ->whereDate('r.from_date', '<=', $request->input('end_date'));
        }
        if (! empty($request->input('property_id'))) {
            $query->where('r.property_id', (int) $request->input('property_id'));
        }

        return DataTables::of($query)
            ->editColumn('rent_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->rent_total.'">'.$row->rent_total.'</span>';
            })
            ->editColumn('paid_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->paid_total.'">'.$row->paid_total.'</span>';
            })
            ->rawColumns(['rent_total', 'paid_total'])
            ->make(true);
    }
}
