<?php

namespace Modules\Mosque\Http\Controllers;

use App\Utils\ModuleUtil;
use DNS2D;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueQrToken;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\MosqueStaff;
use Modules\Mosque\Entities\MosqueStaffAttendance;

class AttendanceQrController extends Controller
{
    private function ensureSubscriptionEnabled(int $businessId): void
    {
        $moduleUtil = new ModuleUtil();
        $enabled = (bool) $moduleUtil->hasThePermissionInSubscription($businessId, 'mosque_module', 'superadmin_package');
        if (! $enabled) {
            abort(403, 'Mosque module is not available for this business.');
        }
    }

    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.staff.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function isQrEnabled(int $businessId): bool
    {
        if (! Schema::hasTable('mosque_settings')) {
            return false;
        }

        $row = MosqueSetting::query()->where('business_id', $businessId)->first();
        $settings = $row?->settings ?: [];

        return ! empty($settings['qr_attendance_enabled']);
    }

    private function buildQrImage(string $qrUrl, string $primaryColor = '#000000'): string
    {
        $hex = ltrim($primaryColor, '#');
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        return 'data:image/png;base64,' . DNS2D::getBarcodePNG($qrUrl, 'QRCODE', 10, 10, [$r, $g, $b]);
    }

    public function index(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $enabled = $this->isQrEnabled($businessId);

        $qrToken = null;
        $qrCode = null;
        $scanUrl = null;
        $primaryColor = '#000000';
        $title = 'Mosque Attendance';

        if ($enabled) {
            $qrToken = MosqueQrToken::firstOrCreate(
                ['business_id' => $businessId],
                [
                    'token' => Str::random(48),
                    'primary_color' => '#000000',
                    'title' => $title,
                    'location_tracking' => true,
                ]
            );

            $primaryColor = $qrToken->primary_color ?: '#000000';
            $title = $qrToken->title ?: $title;
            $scanUrl = route('mosque.qr.mobile', [$qrToken->token]);
            $qrCode = $this->buildQrImage($scanUrl, $primaryColor);
        }

        return view('mosque::attendance_qr.index', compact(
            'enabled',
            'qrToken',
            'qrCode',
            'scanUrl',
            'primaryColor',
            'title'
        ));
    }

    public function regenerate(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        if (! $this->isQrEnabled($businessId)) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => 'QR Attendance is disabled in Mosque Settings.']);
        }

        MosqueQrToken::updateOrCreate(
            ['business_id' => $businessId],
            ['token' => Str::random(48)]
        );

        return redirect()->back()->with('status', ['success' => 1, 'msg' => __('lang_v1.updated_succesfully')]);
    }

    public function showMobileForm(string $token)
    {
        $qrToken = MosqueQrToken::query()->where('token', $token)->first();
        if (empty($qrToken)) {
            abort(404);
        }

        $businessId = (int) $qrToken->business_id;
        $this->ensureSubscriptionEnabled($businessId);
        if (! $this->isQrEnabled($businessId)) {
            return response('QR Attendance is not enabled for this business.', 403);
        }

        $primaryColor = $qrToken->primary_color ?: '#000000';
        $title = $qrToken->title ?: 'Mosque Attendance';

        return view('mosque::attendance_qr.mobile', compact('token', 'primaryColor', 'title'));
    }

    public function submitAttendance(Request $request, string $token)
    {
        $qrToken = MosqueQrToken::query()->where('token', $token)->first();
        if (empty($qrToken)) {
            return response()->json(['success' => 0, 'msg' => __('lang_v1.invalid_link')]);
        }

        $businessId = (int) $qrToken->business_id;
        $this->ensureSubscriptionEnabled($businessId);
        if (! $this->isQrEnabled($businessId)) {
            return response()->json(['success' => 0, 'msg' => 'QR Attendance is not enabled.']);
        }

        $request->validate([
            'staff_id' => 'required|integer|min:1',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);

        $staffId = (int) $request->input('staff_id');
        $staff = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->where('id', $staffId)
            ->first();

        if (empty($staff)) {
            return response()->json(['success' => 0, 'msg' => 'Staff not found.']);
        }

        $cacheKey = 'mosque_qr_attendance:' . $businessId . ':' . $staffId;
        if (Cache::has($cacheKey)) {
            return response()->json(['success' => 0, 'msg' => __('lang_v1.try_again')]);
        }
        Cache::put($cacheKey, true, now()->addSeconds(30));

        $geo = null;
        if ($request->filled('latitude') && $request->filled('longitude') && ! empty($qrToken->location_tracking)) {
            $geo = [
                'lat' => (float) $request->input('latitude'),
                'lng' => (float) $request->input('longitude'),
            ];
        }

        $pending = MosqueStaffAttendance::query()
            ->where('business_id', $businessId)
            ->where('staff_id', $staffId)
            ->whereNull('out_at')
            ->orderByDesc('in_at')
            ->first();

        if (! empty($pending)) {
            $pending->update([
                'out_at' => now(),
                'ip' => $request->ip(),
                'geo' => $geo,
            ]);

            return response()->json(['success' => 1, 'msg' => 'Clocked out successfully.']);
        }

        MosqueStaffAttendance::query()->create([
            'business_id' => $businessId,
            'staff_id' => $staffId,
            'in_at' => now(),
            'out_at' => null,
            'source' => 'qr',
            'ip' => $request->ip(),
            'geo' => $geo,
        ]);

        return response()->json(['success' => 1, 'msg' => 'Clocked in successfully.']);
    }
}
