<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;

class AutomationController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function getSettings(int $businessId): array
    {
        if (! Schema::hasTable('mosque_settings')) {
            return [];
        }

        $row = MosqueSetting::query()->where('business_id', $businessId)->first();
        return $row?->settings ?: [];
    }

    public function remindersIndex()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $settings = $this->getSettings($businessId);

        $periodYm = now()->format('Y-m');

        $dueCount = 0;
        $dueTotal = 0.0;
        if (Schema::hasTable('mosque_member_fees') && Schema::hasTable('mosque_members')) {
            $due = DB::table('mosque_member_fees as f')
                ->join('mosque_members as m', function ($join) use ($businessId) {
                    $join->on('m.id', '=', 'f.member_id')
                        ->where('m.business_id', '=', $businessId);
                })
                ->where('f.business_id', $businessId)
                ->whereIn('f.status', ['pending', 'partial'])
                ->where('f.period_ym', $periodYm)
                ->select([
                    DB::raw('COUNT(*) as cnt'),
                    DB::raw('SUM(GREATEST(f.due_amount - f.paid_amount, 0)) as total_due'),
                ])
                ->first();

            $dueCount = (int) ($due->cnt ?? 0);
            $dueTotal = (float) ($due->total_due ?? 0);
        }

        $donorCount = 0;
        if (Schema::hasTable('mosque_donor_profiles')) {
            $donorCount = (int) DB::table('mosque_donor_profiles')
                ->where('business_id', $businessId)
                ->whereNotNull('email')
                ->where('email', '!=', '')
                ->count();
        }

        return view('mosque::automation.reminders', compact('settings', 'periodYm', 'dueCount', 'dueTotal', 'donorCount'));
    }

    public function sendMembershipDue(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $settings = $this->getSettings($businessId);
        if (empty($settings['reminders_membership_enabled'])) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => 'Membership reminders are disabled in Mosque Settings.']);
        }

        $request->validate([
            'period_ym' => 'required|string|max:7',
            'limit' => 'nullable|integer|min:1|max:500',
        ]);

        $periodYm = (string) $request->input('period_ym');
        $limit = (int) $request->input('limit', 100);
        if (! preg_match('/^\\d{4}-\\d{2}$/', $periodYm)) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => 'Invalid period format. Use YYYY-MM.']);
        }

        if (! Schema::hasTable('mosque_member_fees') || ! Schema::hasTable('mosque_members')) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => __('messages.something_went_wrong')]);
        }

        $rows = DB::table('mosque_member_fees as f')
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'f.member_id')
                    ->where('m.business_id', '=', $businessId);
            })
            ->where('f.business_id', $businessId)
            ->where('f.period_ym', $periodYm)
            ->whereIn('f.status', ['pending', 'partial'])
            ->whereNotNull('m.email')
            ->where('m.email', '!=', '')
            ->select([
                'f.id as fee_id',
                'f.period_ym',
                'f.due_amount',
                'f.paid_amount',
                'm.name as member_name',
                'm.email as email',
            ])
            ->limit($limit)
            ->get();

        $businessName = (string) request()->session()->get('business.name') ?: (string) request()->session()->get('business.business_name') ?: 'Business';
        $mosqueName = $businessName;

        $subject = (string) ($settings['template_membership_reminder_subject'] ?? 'Membership Due Reminder');
        $template = (string) ($settings['template_membership_reminder_email'] ?? '');
        if (trim($template) === '') {
            $template = "Assalamu Alaikum {member},\n\nYour membership fee for {period_ym} is due.\nAmount due: {amount_due}\n\nRegards,\n{mosque}";
        }

        $sent = 0;
        $failed = 0;

        foreach ($rows as $row) {
            $amountDue = max((float) $row->due_amount - (float) $row->paid_amount, 0);
            if ($amountDue <= 0) {
                continue;
            }

            $body = strtr($template, [
                '{business}' => $businessName,
                '{mosque}' => $mosqueName,
                '{member}' => (string) $row->member_name,
                '{period_ym}' => (string) $row->period_ym,
                '{amount_due}' => (string) $amountDue,
                '{date}' => now()->format('Y-m-d'),
            ]);

            try {
                $html = view('mosque::automation.reminder_email', [
                    'title' => $subject,
                    'body' => $body,
                    'footer' => $mosqueName,
                ])->render();

                Mail::send([], [], function ($message) use ($row, $subject, $html) {
                    $message->to($row->email)
                        ->subject($subject)
                        ->setBody($html, 'text/html');
                });

                $sent++;
            } catch (\Exception $e) {
                $failed++;
            }
        }

        MosqueAuditUtil::log($businessId, 'email', 'membership_reminder', null, [
            'period_ym' => $periodYm,
            'attempted' => (int) $rows->count(),
            'sent' => $sent,
            'failed' => $failed,
        ]);

        return redirect()->back()->with('status', ['success' => 1, 'msg' => "Membership reminders: sent {$sent}, failed {$failed}."]);
    }

    public function sendDonationReminder(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $settings = $this->getSettings($businessId);
        if (empty($settings['reminders_donations_enabled'])) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => 'Donation reminders are disabled in Mosque Settings.']);
        }

        $request->validate([
            'limit' => 'nullable|integer|min:1|max:1000',
        ]);

        if (! Schema::hasTable('mosque_donor_profiles')) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => __('messages.something_went_wrong')]);
        }

        $limit = (int) $request->input('limit', 200);

        $donors = DB::table('mosque_donor_profiles')
            ->where('business_id', $businessId)
            ->whereNotNull('email')
            ->where('email', '!=', '')
            ->select(['id', 'name', 'email'])
            ->limit($limit)
            ->get();

        $businessName = (string) request()->session()->get('business.name') ?: (string) request()->session()->get('business.business_name') ?: 'Business';
        $mosqueName = $businessName;

        $subject = (string) ($settings['template_donation_reminder_subject'] ?? 'Donation Reminder');
        $template = (string) ($settings['template_donation_reminder_email'] ?? '');
        if (trim($template) === '') {
            $template = "Assalamu Alaikum {donor},\n\nThis is a gentle reminder for donation support.\n\nRegards,\n{mosque}";
        }

        $sent = 0;
        $failed = 0;

        foreach ($donors as $donor) {
            $body = strtr($template, [
                '{business}' => $businessName,
                '{mosque}' => $mosqueName,
                '{donor}' => (string) ($donor->name ?: 'Donor'),
                '{date}' => now()->format('Y-m-d'),
            ]);

            try {
                $html = view('mosque::automation.reminder_email', [
                    'title' => $subject,
                    'body' => $body,
                    'footer' => $mosqueName,
                ])->render();

                Mail::send([], [], function ($message) use ($donor, $subject, $html) {
                    $message->to($donor->email)
                        ->subject($subject)
                        ->setBody($html, 'text/html');
                });

                $sent++;
            } catch (\Exception $e) {
                $failed++;
            }
        }

        MosqueAuditUtil::log($businessId, 'email', 'donation_reminder', null, [
            'attempted' => (int) $donors->count(),
            'sent' => $sent,
            'failed' => $failed,
        ]);

        return redirect()->back()->with('status', ['success' => 1, 'msg' => "Donation reminders: sent {$sent}, failed {$failed}."]);
    }
}
