<?php

namespace Modules\Mosque\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueBankAccount;
use Modules\Mosque\Entities\MosqueBankCashoutRequest;
use Modules\Mosque\Entities\MosqueBankWalletTransaction;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueBankBalanceUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class BankCashoutController extends Controller
{
    private const ATTACH_DIR = 'mosque_bank_cashout';

    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->hasAnyPermission([
            'mosque.finance.expense',
            'mosque.finance.reports',
            'mosque.finance.cashout.manage',
        ])) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureDir(): void
    {
        try {
            Storage::disk('public')->makeDirectory(self::ATTACH_DIR);
        } catch (\Throwable $e) {
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $banks = [];
        if (Schema::hasTable('mosque_bank_accounts')) {
            $banks = MosqueBankAccount::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('bank_name')
                ->select(['id'])
                ->selectRaw("CONCAT(bank_name,' - ',account_no) as label")
                ->pluck('label', 'id')
                ->toArray();
        }

        return view('mosque::finance.cashouts.index', compact('banks'));
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $base = DB::table('mosque_bank_cashout_requests as c')
            ->join('mosque_bank_accounts as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'c.bank_account_id')
                    ->where('b.business_id', '=', $businessId);
            })
            ->where('c.business_id', $businessId)
            ->whereNull('c.deleted_at');

        if (! empty($request->input('bank_account_id'))) {
            $base->where('c.bank_account_id', $request->input('bank_account_id'));
        }
        if (! empty($request->input('status'))) {
            $base->where('c.status', $request->input('status'));
        }
        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $base->whereDate('c.request_date', '>=', $request->input('start_date'))
                ->whereDate('c.request_date', '<=', $request->input('end_date'));
        }
        if (! empty($request->input('min_amount'))) {
            $base->where('c.amount', '>=', (float) $request->input('min_amount'));
        }
        if (! empty($request->input('max_amount'))) {
            $base->where('c.amount', '<=', (float) $request->input('max_amount'));
        }

        $totalAmount = (float) ((clone $base)->sum('c.amount') ?: 0);

        $query = (clone $base)
            ->select([
                'c.id',
                'c.request_no',
                'c.amount',
                'c.request_date',
                'c.reason',
                'c.status',
                'c.attachment_path',
                'c.approved_at',
                'b.bank_name',
                'b.account_no',
            ])
            ->orderByDesc('c.request_date')
            ->orderByDesc('c.id');

        return DataTables::of($query)
            ->addColumn('bank', function ($row) {
                return e((string) $row->bank_name).' - '.e((string) $row->account_no);
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->addColumn('attachment', function ($row) {
                if (! empty($row->attachment_path)) {
                    $url = route('mosque.finance.cashouts.attachment', [(int) $row->id]);
                    return '<a class="btn btn-xs btn-default" target="_blank" href="'.$url.'"><i class="fa fa-paperclip"></i> '.__('lang_v1.view').'</a>';
                }
                return '-';
            })
            ->addColumn('action', function ($row) {
                $receipt = '<a class="btn btn-xs btn-default" target="_blank" href="'.route('mosque.finance.cashouts.receipt', [(int) $row->id]).'"><i class="fa fa-file-text-o"></i> '.__('mosque::mosque.bank_btn_receipt').'</a>';

                if ((string) $row->status === 'pending') {
                    $approveUrl = route('mosque.finance.cashouts.approve', [(int) $row->id]);
                    $rejectUrl = route('mosque.finance.cashouts.reject', [(int) $row->id]);
                    $approve = '<button data-href="'.$approveUrl.'" class="btn btn-xs btn-success mosque_cashout_approve"><i class="fa fa-check"></i> '.__('mosque::mosque.bank_btn_approve').'</button>';
                    $reject = '<button data-href="'.$rejectUrl.'" class="btn btn-xs btn-danger mosque_cashout_reject"><i class="fa fa-times"></i> '.__('mosque::mosque.bank_btn_reject').'</button>';
                    return $receipt.' '.$approve.' '.$reject;
                }
                if ((string) $row->status === 'approved') {
                    $final = '<a class="btn btn-xs btn-default" target="_blank" href="'.route('mosque.finance.cashouts.final_receipt', [(int) $row->id]).'"><i class="fa fa-check-circle"></i> '.__('mosque::mosque.bank_btn_final').'</a>';
                    return $receipt.' '.$final;
                }

                return $receipt;
            })
            ->with('total_amount', $totalAmount)
            ->rawColumns(['amount', 'attachment', 'action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $banks = [];
        if (Schema::hasTable('mosque_bank_accounts')) {
            $banks = MosqueBankAccount::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('bank_name')
                ->select(['id'])
                ->selectRaw("CONCAT(bank_name,' - ',account_no) as label")
                ->pluck('label', 'id')
                ->toArray();
        }

        return view('mosque::finance.cashouts.create', compact('banks'));
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'bank_account_id' => 'required|integer',
            'amount' => 'required|numeric|min:0.01',
            'request_date' => 'required|date',
            'reason' => 'required|string|max:2000',
            'attachment' => 'nullable|file|mimes:jpg,jpeg,png,webp,pdf|max:5120',
        ]);

        try {
            $businessId = $this->businessId();
            $this->ensureDir();

            $bank = MosqueBankAccount::query()
                ->where('business_id', $businessId)
                ->where('id', $request->input('bank_account_id'))
                ->where('is_active', true)
                ->firstOrFail();

            $amount = (float) $request->input('amount');
            $bankBalance = MosqueBankBalanceUtil::bankWalletBalanceByAccount($businessId, (int) $bank->id);
            if ($amount - $bankBalance > 0.0001) {
                return ['success' => false, 'msg' => __('mosque::mosque.bank_err_insufficient_bank_balance')];
            }

            $attachPath = null;
            if ($request->hasFile('attachment')) {
                $file = $request->file('attachment');
                $ext = strtolower((string) $file->getClientOriginalExtension());
                $name = 'cashout_'.date('YmdHis').'_'.Str::random(8).'.'.$ext;
                $attachPath = $file->storeAs(self::ATTACH_DIR, $name, 'public');
            }

            $cashout = MosqueBankCashoutRequest::query()->create([
                'business_id' => $businessId,
                'bank_account_id' => (int) $bank->id,
                'request_no' => 'CO-'.now()->format('YmdHis').'-'.$businessId,
                'amount' => $amount,
                'request_date' => $request->input('request_date'),
                'reason' => $request->input('reason'),
                'attachment_path' => $attachPath,
                'status' => 'pending',
                'requested_by' => auth()->id(),
                'requested_at' => now(),
            ]);

            MosqueAuditUtil::log($businessId, 'create', 'bank_cashout_request', (int) $cashout->id, [
                'bank_account_id' => (int) $bank->id,
                'amount' => $amount,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function approve(Request $request, int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        try {
            $cashout = MosqueBankCashoutRequest::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            if ((string) $cashout->status !== 'pending') {
                return ['success' => false, 'msg' => __('messages.something_went_wrong')];
            }

            $bankBalance = MosqueBankBalanceUtil::bankWalletBalanceByAccount($businessId, (int) $cashout->bank_account_id);
            if ((float) $cashout->amount - $bankBalance > 0.0001) {
                return ['success' => false, 'msg' => __('mosque::mosque.bank_err_insufficient_bank_balance')];
            }

            $financeEntry = null;
            DB::transaction(function () use ($businessId, $cashout, &$financeEntry) {
                MosqueBankWalletTransaction::query()->create([
                    'business_id' => $businessId,
                    'bank_account_id' => (int) $cashout->bank_account_id,
                    'direction' => 'debit',
                    'amount' => (float) $cashout->amount,
                    'txn_date' => (string) ($cashout->request_date ?? now()->format('Y-m-d')),
                    'ref_module' => 'bank_cashout',
                    'ref_id' => (int) $cashout->id,
                    'note' => (string) ($cashout->reason ?? ''),
                    'created_by' => auth()->id(),
                ]);

                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'expense', 'name' => 'Bank cashout'],
                    ['active' => true, 'sort_order' => 20]
                );

                $financeEntry = MosqueFinanceEntry::query()->create([
                    'business_id' => $businessId,
                    'location_id' => null,
                    'type' => 'expense',
                    'category_id' => $category->id,
                    'amount' => (float) $cashout->amount,
                    'entry_date' => (string) ($cashout->request_date ?? now()->format('Y-m-d')),
                    'ref_module' => 'bank_cashout',
                    'ref_id' => (int) $cashout->id,
                    'fund_tag' => null,
                    'note' => 'Bank cashout: '.trim((string) ($cashout->reason ?? '')),
                    'created_by' => auth()->id(),
                ]);

                $cashout->update([
                    'status' => 'approved',
                    'approved_by' => auth()->id(),
                    'approved_at' => now(),
                    'finance_entry_id' => $financeEntry ? (int) $financeEntry->id : null,
                ]);
            });

            MosqueAuditUtil::log($businessId, 'approve', 'bank_cashout_request', (int) $cashout->id, [
                'amount' => (float) $cashout->amount,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function reject(Request $request, int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $request->validate([
            'reject_reason' => 'required|string|max:2000',
        ]);

        try {
            $cashout = MosqueBankCashoutRequest::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            if ((string) $cashout->status !== 'pending') {
                return ['success' => false, 'msg' => __('messages.something_went_wrong')];
            }

            $cashout->update([
                'status' => 'rejected',
                'reject_reason' => $request->input('reject_reason'),
                'rejected_by' => auth()->id(),
                'rejected_at' => now(),
            ]);

            MosqueAuditUtil::log($businessId, 'reject', 'bank_cashout_request', (int) $cashout->id, [
                'amount' => (float) $cashout->amount,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function receipt(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $cashout = MosqueBankCashoutRequest::query()->where('business_id', $businessId)->findOrFail($id);
        $bank = MosqueBankAccount::query()->where('business_id', $businessId)->find($cashout->bank_account_id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        return view('mosque::finance.cashouts.receipt_request', compact('cashout', 'bank', 'profile', 'settings', 'logoDataUri'));
    }

    public function receiptPdf(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $cashout = MosqueBankCashoutRequest::query()->where('business_id', $businessId)->findOrFail($id);
        $bank = MosqueBankAccount::query()->where('business_id', $businessId)->find($cashout->bank_account_id);
        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $pdf = Pdf::loadView('mosque::finance.cashouts.receipt_request', compact('cashout', 'bank', 'profile', 'settings', 'logoDataUri'))->setPaper('a4');
        return $pdf->download('cashout_request_'.$cashout->id.'.pdf');
    }

    public function finalReceipt(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $cashout = MosqueBankCashoutRequest::query()->where('business_id', $businessId)->findOrFail($id);
        $bank = MosqueBankAccount::query()->where('business_id', $businessId)->find($cashout->bank_account_id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        return view('mosque::finance.cashouts.receipt_final', compact('cashout', 'bank', 'profile', 'settings', 'logoDataUri'));
    }

    public function finalReceiptPdf(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $cashout = MosqueBankCashoutRequest::query()->where('business_id', $businessId)->findOrFail($id);
        $bank = MosqueBankAccount::query()->where('business_id', $businessId)->find($cashout->bank_account_id);
        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $pdf = Pdf::loadView('mosque::finance.cashouts.receipt_final', compact('cashout', 'bank', 'profile', 'settings', 'logoDataUri'))->setPaper('a4');
        return $pdf->download('cashout_final_'.$cashout->id.'.pdf');
    }

    public function attachment(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $cashout = MosqueBankCashoutRequest::query()->where('business_id', $businessId)->findOrFail($id);
        $path = trim((string) ($cashout->attachment_path ?? ''));
        if ($path === '' || ! Storage::disk('public')->exists($path)) {
            abort(404);
        }
        return response()->file(Storage::disk('public')->path($path), [
            'Cache-Control' => 'private, max-age=86400',
            'Content-Disposition' => 'inline; filename="'.basename($path).'"',
        ]);
    }
}
