<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueBankAccount;
use Modules\Mosque\Utils\MosqueBankBalanceUtil;

class BankManagementController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->hasAnyPermission([
            'mosque.finance.bank.manage',
            'mosque.finance.bank.transfer',
            'mosque.finance.cashout.manage',
            'mosque.finance.reports',
        ])) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $banks = [];
        if (Schema::hasTable('mosque_bank_accounts')) {
            $banks = MosqueBankAccount::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('bank_name')
                ->select(['id'])
                ->selectRaw("CONCAT(bank_name,' - ',account_no) as label")
                ->pluck('label', 'id')
                ->toArray();
        }

        $bankSummary = [
            'cash_net_balance' => MosqueBankBalanceUtil::cashNetBalance($businessId),
            'bank_wallet_total' => MosqueBankBalanceUtil::bankWalletTotal($businessId),
            'net_including_bank' => MosqueBankBalanceUtil::overallNetBalance($businessId),
        ];

        return view('mosque::finance.bank_management.index', compact('banks', 'bankSummary'));
    }
}

