<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueBankAccount;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueBankBalanceUtil;
use Yajra\DataTables\Facades\DataTables;

class BankWalletController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->hasAnyPermission([
            'mosque.finance.income',
            'mosque.finance.expense',
            'mosque.finance.reports',
            'mosque.finance.bank.manage',
        ])) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        return view('mosque::finance.banks.index');
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueBankAccount::query()
            ->where('business_id', $businessId)
            ->select(['id', 'bank_name', 'account_no', 'account_holder', 'is_active', 'created_at']);

        return DataTables::of($query)
            ->addColumn('wallet_balance', function ($row) use ($businessId) {
                $bal = MosqueBankBalanceUtil::bankWalletBalanceByAccount($businessId, (int) $row->id);
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$bal.'">'.$bal.'</span>';
            })
            ->addColumn('status', function ($row) {
                return ! empty($row->is_active)
                    ? '<span class="label label-success">'.__('lang_v1.active').'</span>'
                    : '<span class="label label-default">'.__('lang_v1.inactive').'</span>';
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([self::class, 'edit'], [$row->id], false).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_bank_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = action([self::class, 'destroy'], [$row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" class="btn btn-xs btn-danger delete_mosque_bank_account"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['wallet_balance', 'status', 'action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();
        return view('mosque::finance.banks.create');
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'bank_name' => 'required|string|max:255',
            'account_no' => 'required|string|max:255',
            'account_holder' => 'nullable|string|max:255',
            'notes' => 'nullable|string|max:2000',
            'is_active' => 'nullable|boolean',
        ]);

        try {
            $businessId = $this->businessId();
            $acc = MosqueBankAccount::query()->create([
                'business_id' => $businessId,
                'bank_name' => $request->input('bank_name'),
                'account_no' => $request->input('account_no'),
                'account_holder' => $request->input('account_holder'),
                'notes' => $request->input('notes'),
                'is_active' => (bool) $request->boolean('is_active', true),
                'created_by' => auth()->id(),
            ]);

            MosqueAuditUtil::log($businessId, 'create', 'bank_account', (int) $acc->id, [
                'bank_name' => (string) $acc->bank_name,
                'account_no' => (string) $acc->account_no,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $bank = MosqueBankAccount::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::finance.banks.edit', compact('bank'));
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'bank_name' => 'required|string|max:255',
            'account_no' => 'required|string|max:255',
            'account_holder' => 'nullable|string|max:255',
            'notes' => 'nullable|string|max:2000',
            'is_active' => 'nullable|boolean',
        ]);

        try {
            $businessId = $this->businessId();
            $bank = MosqueBankAccount::query()->where('business_id', $businessId)->findOrFail($id);
            $bank->update([
                'bank_name' => $request->input('bank_name'),
                'account_no' => $request->input('account_no'),
                'account_holder' => $request->input('account_holder'),
                'notes' => $request->input('notes'),
                'is_active' => (bool) $request->boolean('is_active', true),
            ]);

            MosqueAuditUtil::log($businessId, 'update', 'bank_account', (int) $bank->id, [
                'bank_name' => (string) $bank->bank_name,
                'account_no' => (string) $bank->account_no,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $bank = MosqueBankAccount::query()->where('business_id', $businessId)->find($id);
            if (! $bank) {
                return ['success' => true, 'msg' => __('lang_v1.success')];
            }

            $balance = MosqueBankBalanceUtil::bankWalletBalanceByAccount($businessId, (int) $bank->id);
            if (abs($balance) > 0.0001) {
                return ['success' => false, 'msg' => __('mosque::mosque.bank_err_non_zero_wallet')];
            }

            $bank->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'bank_account', (int) $bank->id, [
                'bank_name' => (string) $bank->bank_name,
                'account_no' => (string) $bank->account_no,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function summary()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $bankTotal = MosqueBankBalanceUtil::bankWalletTotal($businessId);
        $overallNet = MosqueBankBalanceUtil::overallNetBalance($businessId);
        $cashNet = MosqueBankBalanceUtil::cashNetBalance($businessId);

        $accounts = [];
        if (Schema::hasTable('mosque_bank_accounts')) {
            $rows = MosqueBankAccount::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('bank_name')
                ->orderBy('account_no')
                ->get(['id', 'bank_name', 'account_no', 'account_holder']);

            foreach ($rows as $r) {
                $accounts[] = [
                    'id' => (int) $r->id,
                    'bank_name' => (string) $r->bank_name,
                    'account_no' => (string) $r->account_no,
                    'account_holder' => (string) ($r->account_holder ?? ''),
                    'wallet_balance' => MosqueBankBalanceUtil::bankWalletBalanceByAccount($businessId, (int) $r->id),
                ];
            }
        }

        return [
            'success' => true,
            'cash_net_balance' => $cashNet,
            'bank_wallet_total' => $bankTotal,
            'net_including_bank' => $overallNet,
            'accounts' => $accounts,
        ];
    }
}

