<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueCommittee;
use Modules\Mosque\Entities\MosqueCommitteeMember;
use Modules\Mosque\Entities\MosqueCommitteeRole;
use Modules\Mosque\Entities\MosqueMinute;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class CommitteeController extends Controller
{
    private const MEMBER_PHOTO_DIR = 'uploads/mosque_committee_member_photos';
    private const MEMBER_ID_DOC_DIR = 'uploads/mosque_committee_member_ids';
    private const DEFAULT_WHATSAPP_COUNTRY_CODE = '880';

    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function whatsappDefaultCountryCode(int $businessId): string
    {
        try {
            if (Schema::hasTable('mosque_settings')) {
                $row = MosqueSetting::query()->where('business_id', $businessId)->first();
                $settings = $row?->settings ?: [];
                $cc = preg_replace('/\\D+/', '', (string) ($settings['whatsapp_default_country_code'] ?? ''));
                if (! empty($cc)) {
                    return $cc;
                }
            }
        } catch (\Throwable $e) {
        }

        return self::DEFAULT_WHATSAPP_COUNTRY_CODE;
    }

    private function normalizeWhatsappDigits(string $raw, int $businessId): string
    {
        $digits = preg_replace('/\\D+/', '', (string) $raw);
        $digits = ltrim((string) $digits);
        if ($digits === '') {
            return '';
        }

        // Convert "00CC..." to "CC..."
        if (str_starts_with($digits, '00')) {
            $digits = substr($digits, 2);
        }

        // If user enters local format like "019..." then add country code.
        if (str_starts_with($digits, '0')) {
            $cc = $this->whatsappDefaultCountryCode($businessId);
            $digits = $cc.ltrim($digits, '0');
        }

        return $digits;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.committee.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureDir(string $publicRelativeDir): void
    {
        try {
            $dir = public_path($publicRelativeDir);
            if (! File::isDirectory($dir)) {
                File::makeDirectory($dir, 0755, true);
            }
        } catch (\Throwable $e) {
        }
    }

    private function saveMemberPhoto(Request $request): ?string
    {
        try {
            $this->ensureDir(self::MEMBER_PHOTO_DIR);
            $targetDir = public_path(self::MEMBER_PHOTO_DIR);

            if ($request->hasFile('photo')) {
                $file = $request->file('photo');
                $ext = strtolower((string) $file->getClientOriginalExtension());
                $name = 'committee_member_'.now()->format('YmdHis').'_'.Str::random(8).'.'.$ext;
                $file->move($targetDir, $name);
                return $name;
            }

            $capture = (string) $request->input('photo_capture', '');
            if ($capture !== '' && str_starts_with($capture, 'data:image')) {
                $parts = explode(',', $capture, 2);
                if (count($parts) !== 2) {
                    return null;
                }
                $meta = $parts[0];
                $data = base64_decode($parts[1], true);
                if ($data === false) {
                    return null;
                }

                $ext = 'jpg';
                if (str_contains($meta, 'png')) {
                    $ext = 'png';
                } elseif (str_contains($meta, 'webp')) {
                    $ext = 'webp';
                }

                $name = 'committee_member_'.now()->format('YmdHis').'_'.Str::random(8).'.'.$ext;
                file_put_contents($targetDir.DIRECTORY_SEPARATOR.$name, $data);
                return $name;
            }
        } catch (\Throwable $e) {
            return null;
        }

        return null;
    }

    private function saveMemberIdDoc(Request $request): ?string
    {
        try {
            $this->ensureDir(self::MEMBER_ID_DOC_DIR);
            $targetDir = public_path(self::MEMBER_ID_DOC_DIR);

            if (! $request->hasFile('id_doc_file')) {
                return null;
            }
            $file = $request->file('id_doc_file');
            $ext = strtolower((string) $file->getClientOriginalExtension());
            $name = 'id_'.now()->format('YmdHis').'_'.Str::random(8).'.'.$ext;
            $file->move($targetDir, $name);
            return $name;
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function memberPhotoUrl(?string $photoPath): string
    {
        $photoPath = trim((string) ($photoPath ?? ''));
        if ($photoPath === '') {
            return asset('img/default.png');
        }
        $file = public_path(self::MEMBER_PHOTO_DIR.DIRECTORY_SEPARATOR.$photoPath);
        if (File::exists($file)) {
            return asset(self::MEMBER_PHOTO_DIR.'/'.$photoPath);
        }
        return asset('img/default.png');
    }

    public function index()
    {
        $this->ensurePermission();

        $businessId = $this->businessId();

        $committees = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->withTrashed()
            ->orderBy('name')
            ->pluck('name', 'id');

        $roles = [];
        if (Schema::hasTable('mosque_committee_roles')) {
            $roles = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('sort_order')
                ->orderBy('name')
                ->pluck('name', 'id');
        }

        return view('mosque::committee.index', compact('committees', 'roles'));
    }

    // Committees
    public function committeesData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $status = (string) request()->input('status', 'active'); // active|archived|all
        $query = MosqueCommittee::query()->where('business_id', $businessId);
        if ($status === 'all') {
            $query->withTrashed();
        } elseif ($status === 'archived') {
            $query->onlyTrashed();
        }

        $query->select(['id', 'name', 'term_start', 'term_end', 'deleted_at']);

        $termRange = trim((string) request()->input('term_range', ''));
        if ($termRange !== '' && str_contains($termRange, '-')) {
            $parts = array_map('trim', explode('-', $termRange, 2));
            if (count($parts) === 2 && ! empty($parts[0]) && ! empty($parts[1])) {
                try {
                    $start = \Carbon\Carbon::parse($parts[0])->format('Y-m-d');
                    $end = \Carbon\Carbon::parse($parts[1])->format('Y-m-d');
                    $query->where(function ($q) use ($start, $end) {
                        // Keep committees with missing term dates visible.
                        $q->whereNull('term_start')
                            ->orWhereNull('term_end')
                            ->orWhereBetween('term_start', [$start, $end])
                            ->orWhereBetween('term_end', [$start, $end]);
                    });
                } catch (\Throwable $e) {
                }
            }
        }

        return DataTables::of($query)
            ->addColumn('members_count', function ($row) use ($businessId) {
                return MosqueCommitteeMember::query()
                    ->where('business_id', $businessId)
                    ->where('committee_id', $row->id)
                    ->count();
            })
            ->addColumn('action', function ($row) {
                $invite = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'invite'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-envelope"></i> Invite</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'committeesEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_committee_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';

                if (! empty($row->deleted_at)) {
                    $restore = '<button data-href="'.route('mosque.committee.committees.restore', [$row->id]).'" class="btn btn-xs btn-success restore_mosque_committee"><i class="fa fa-undo"></i> Restore</button>';
                    return $invite.' '.$restore;
                }

                $archive = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'committeesDestroy'], [$row->id]).'" class="btn btn-xs btn-warning delete_mosque_committee"><i class="fa fa-archive"></i> Archive</button>';
                return $invite.' '.$edit.' '.$archive;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function committeesRestore($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();

            $committee = MosqueCommittee::withTrashed()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $committee->restore();

            MosqueAuditUtil::log($businessId, 'restore', 'committee', (int) $committee->id, [
                'name' => (string) $committee->name,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function invite($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $members = MosqueCommitteeMember::query()
            ->where('business_id', $businessId)
            ->where('committee_id', $committee->id)
            ->orderBy('member_name')
            ->get();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        MosqueAuditUtil::log($businessId, 'print', 'committee_invite', (int) $committee->id, [
            'members' => (int) $members->count(),
        ]);

        return view('mosque::committee.invite', compact('committee', 'members', 'profile', 'logoDataUri', 'settings'));
    }

    public function invitePdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $members = MosqueCommitteeMember::query()
            ->where('business_id', $businessId)
            ->where('committee_id', $committee->id)
            ->orderBy('member_name')
            ->get();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        MosqueAuditUtil::log($businessId, 'pdf', 'committee_invite', (int) $committee->id, [
            'members' => (int) $members->count(),
        ]);

        $pdf = Pdf::loadView('mosque::committee.invite', compact('committee', 'members', 'profile', 'logoDataUri', 'settings'))
            ->setPaper('a4');

        $filename = 'committee_invite_'.$committee->id.'.pdf';

        return $pdf->download($filename);
    }

    public function inviteEmail(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $members = MosqueCommitteeMember::query()
            ->where('business_id', $businessId)
            ->where('committee_id', $committee->id)
            ->whereNotNull('email')
            ->where('email', '!=', '')
            ->get();

        if ($members->isEmpty()) {
            return redirect()
                ->back()
                ->with('status', ['success' => 0, 'msg' => 'No committee member emails found.']);
        }

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        $businessName = (string) request()->session()->get('business.name') ?: (string) request()->session()->get('business.business_name') ?: 'Business';
        $headerName = ! empty($profile?->name) ? $profile->name : $businessName;

        $termStart = ! empty($committee->term_start) ? \Carbon\Carbon::parse($committee->term_start)->format('Y-m-d') : '';
        $termEnd = ! empty($committee->term_end) ? \Carbon\Carbon::parse($committee->term_end)->format('Y-m-d') : '';

        $template = (string) ($settings['template_email_message'] ?? '');
        if (trim($template) === '') {
            $template = "Assalamu Alaikum,\n\nYou are invited to join the committee: {committee}\nTerm: {term_start} to {term_end}\n\nRegards,\n{mosque}";
        }

        $subject = 'Committee Invitation - '.$committee->name;

        $sent = 0;
        $failed = 0;
        $whatsappLinks = [];

        foreach ($members as $member) {
            try {
                $body = strtr($template, [
                    '{business}' => $businessName,
                    '{mosque}' => $headerName,
                    '{committee}' => (string) $committee->name,
                    '{term_start}' => (string) $termStart,
                    '{term_end}' => (string) $termEnd,
                    '{member}' => (string) $member->member_name,
                    '{role}' => (string) ($member->role_title ?? ''),
                    '{date}' => now()->format('Y-m-d'),
                ]);

                $html = view('mosque::committee.invite_email', [
                    'committee' => $committee,
                    'member' => $member,
                    'profile' => $profile,
                    'body' => $body,
                ])->render();

                Mail::send([], [], function ($message) use ($member, $subject, $html) {
                    $message->to($member->email)
                        ->subject($subject)
                        ->setBody($html, 'text/html');
                });

                $sent++;
            } catch (\Exception $e) {
                $failed++;
            }

            try {
                $waRaw = trim((string) ($member->whatsapp_number ?? ''));
                if ($waRaw === '' && ! empty($member->whatsapp_same_as_phone) && ! empty($member->phone)) {
                    $waRaw = (string) $member->phone;
                }
                if ($waRaw === '' && ! empty($member->phone)) {
                    $waRaw = (string) $member->phone;
                }

                if ($waRaw !== '') {
                    $digits = $this->normalizeWhatsappDigits($waRaw, $businessId);
                    if ($digits !== '') {
                        $whatsappLinks[] = [
                            'name' => (string) $member->member_name,
                            'url' => route('mosque.committee.invite.whatsapp', [$committee->id, $member->id]),
                            'phone' => $digits,
                        ];
                    }
                }
            } catch (\Throwable $e) {
            }
        }

        MosqueAuditUtil::log($businessId, 'email', 'committee_invite', (int) $committee->id, [
            'sent' => $sent,
            'failed' => $failed,
        ]);

        return redirect()
            ->back()
            ->with('status', [
                'success' => 1,
                'msg' => "Invitations sent: {$sent}. Failed: {$failed}.",
                'whatsapp_links' => $whatsappLinks,
            ]);
    }

    public function inviteWhatsapp($committee_id, $member_id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->findOrFail($committee_id);

        $member = MosqueCommitteeMember::query()
            ->where('business_id', $businessId)
            ->where('committee_id', $committee->id)
            ->findOrFail($member_id);

        $waRaw = trim((string) ($member->whatsapp_number ?? ''));
        if ($waRaw === '' && ! empty($member->whatsapp_same_as_phone) && ! empty($member->phone)) {
            $waRaw = (string) $member->phone;
        }
        if ($waRaw === '' && ! empty($member->phone)) {
            $waRaw = (string) $member->phone;
        }

        $digits = $this->normalizeWhatsappDigits($waRaw, $businessId);
        if (empty($digits)) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => 'Member WhatsApp/phone not found.']);
        }

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        $businessName = (string) request()->session()->get('business.name') ?: (string) request()->session()->get('business.business_name') ?: 'Business';
        $headerName = ! empty($profile?->name) ? $profile->name : $businessName;

        $termStart = ! empty($committee->term_start) ? \Carbon\Carbon::parse($committee->term_start)->format('Y-m-d') : '';
        $termEnd = ! empty($committee->term_end) ? \Carbon\Carbon::parse($committee->term_end)->format('Y-m-d') : '';

        $template = (string) ($settings['template_whatsapp_message'] ?? '');
        if (trim($template) === '') {
            $template = "Assalamu Alaikum {member},\nYou are invited to join committee: {committee}\nTerm: {term_start} to {term_end}\n\n{mosque}";
        }

        $body = strtr($template, [
            '{business}' => $businessName,
            '{mosque}' => $headerName,
            '{committee}' => (string) $committee->name,
            '{term_start}' => (string) $termStart,
            '{term_end}' => (string) $termEnd,
            '{member}' => (string) $member->member_name,
            '{role}' => (string) ($member->role_title ?? ''),
            '{date}' => now()->format('Y-m-d'),
        ]);

        MosqueAuditUtil::log($businessId, 'whatsapp', 'committee_invite', (int) $committee->id, [
            'member_id' => (int) $member->id,
            'phone' => $digits,
        ]);

        $url = 'https://wa.me/'.$digits.'?text='.rawurlencode($body);

        return redirect()->away($url);
    }

    public function committeesCreate()
    {
        $this->ensurePermission();

        return view('mosque::committee.committees.create');
    }

    public function committeesStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'term_start' => 'nullable|date',
            'term_end' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();

            MosqueCommittee::query()->create(array_merge(
                $request->only(['name', 'term_start', 'term_end', 'notes']),
                ['business_id' => $businessId]
            ));

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function committeesEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        return view('mosque::committee.committees.edit', compact('committee'));
    }

    public function committeesUpdate(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'term_start' => 'nullable|date',
            'term_end' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $committee = MosqueCommittee::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $committee->update($request->only(['name', 'term_start', 'term_end', 'notes']));

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function committeesDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $committee = MosqueCommittee::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $committee->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'committee', (int) $committee->id, [
                'name' => (string) $committee->name,
                'term_start' => (string) ($committee->term_start ?? ''),
                'term_end' => (string) ($committee->term_end ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'committee', (int) $committee->id, [
                'name' => (string) $committee->name,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Members
    public function membersData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_committee_members as m')
            ->where('m.business_id', $businessId)
            ->whereNull('m.deleted_at');

        $joinRoles = false;
        if (Schema::hasTable('mosque_committee_roles') && Schema::hasColumn('mosque_committee_members', 'role_id')) {
            $query->leftJoin('mosque_committee_roles as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 'm.role_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            });
            $joinRoles = true;
        }

        $select = [
            'm.id',
            'm.committee_id',
            'm.member_name',
            'm.phone',
            'm.whatsapp_number',
            'm.whatsapp_same_as_phone',
            'm.photo_path',
            'm.email',
        ];
        if ($joinRoles) {
            $select[] = DB::raw("COALESCE(NULLIF(r.name,''), m.role_title) as role_title");
        } else {
            $select[] = 'm.role_title';
        }
        $query->select($select);

        if (! empty($request->input('committee_id'))) {
            $query->where('m.committee_id', $request->input('committee_id'));
        }
        if (! empty($request->input('role_id')) && Schema::hasColumn('mosque_committee_members', 'role_id')) {
            $query->where('m.role_id', (int) $request->input('role_id'));
        }

        $hasWhatsapp = (string) $request->input('has_whatsapp', '');
        if ($hasWhatsapp === 'yes') {
            $query->where(function ($q) {
                $q->whereNotNull('m.whatsapp_number')
                    ->where('m.whatsapp_number', '!=', '')
                    ->orWhere(function ($q2) {
                        $q2->whereNotNull('m.phone')
                            ->where('m.phone', '!=', '')
                            ->where('m.whatsapp_same_as_phone', true);
                    });
            });
        } elseif ($hasWhatsapp === 'no') {
            $query->where(function ($q) {
                $q->whereNull('m.whatsapp_number')->orWhere('m.whatsapp_number', '=', '');
            });
        }

        return DataTables::of($query)
            ->addColumn('photo', function ($row) {
                $src = $this->memberPhotoUrl($row->photo_path ?? null);
                $fallback = asset('public/img/default.png');
                return '<img src="'.e($src).'" onerror="this.onerror=null;this.src=\''.e($fallback).'\';" style="width:32px;height:32px;border-radius:6px;object-fit:cover;border:1px solid #e6eef7;background:#fff;" alt="photo">';
            })
            ->addColumn('committee', function ($row) use ($businessId) {
                $committee = MosqueCommittee::withTrashed()
                    ->where('business_id', $businessId)
                    ->where('id', $row->committee_id)
                    ->first();
                if (! $committee) {
                    return '';
                }
                return (string) $committee->name.(($committee->deleted_at ?? null) ? ' (Archived)' : '');
            })
            ->editColumn('whatsapp_number', function ($row) {
                $phone = trim((string) ($row->phone ?? ''));
                $wa = trim((string) ($row->whatsapp_number ?? ''));
                if ($wa === '' && ! empty($row->whatsapp_same_as_phone) && $phone !== '') {
                    $wa = $phone;
                }
                if ($wa === '') {
                    return '-';
                }
                $digits = $this->normalizeWhatsappDigits($wa, $this->businessId());
                if ($digits === '') {
                    return '<span><i class="fa fa-whatsapp" style="margin-right:4px;"></i>'.e($wa).'</span>';
                }

                // Invite link opens WhatsApp with prefilled template message.
                $inviteUrl = route('mosque.committee.invite.whatsapp', [(int) $row->committee_id, (int) $row->id]);
                return '<a href="'.$inviteUrl.'" target="_blank" rel="noopener" title="Send WhatsApp invite"><i class="fa fa-whatsapp" style="margin-right:4px;"></i>'.e($wa).'</a>';
            })
            ->addColumn('action', function ($row) {
                $card = '<a target="_blank" href="'.route('mosque.committee.members.card', [(int) $row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-id-card"></i> '.__('mosque::mosque.committee_id_card').'</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'membersEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_committee_member_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'membersDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_committee_member"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $card.' '.$edit.' '.$delete;
            })
            ->rawColumns(['photo', 'whatsapp_number', 'action'])
            ->make(true);
    }

    public function membersCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $committees = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        $roles = [];
        if (Schema::hasTable('mosque_committee_roles')) {
            $roles = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('sort_order')
                ->orderBy('name')
                ->pluck('name', 'id');
        }

        return view('mosque::committee.members.create', compact('committees', 'roles'));
    }

    public function membersStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'committee_id' => 'required|integer',
            'member_name' => 'required|string|max:255',
            'role_id' => 'nullable|integer',
            'role_title' => 'nullable|string|max:255',
            'photo' => 'nullable|image|max:5120',
            'photo_capture' => 'nullable|string',
            'id_doc_type' => 'required|string|in:nid,passport,other',
            'id_doc_no' => 'required|string|max:64',
            'id_doc_file' => 'required|file|mimes:jpg,jpeg,png,webp,pdf|max:5120',
            'phone' => 'nullable|string|max:50',
            'whatsapp_number' => 'nullable|string|max:50',
            'whatsapp_same_as_phone' => 'nullable|boolean',
            'email' => 'nullable|email|max:255',
        ]);

        try {
            $businessId = $this->businessId();
            $roleId = $request->input('role_id');
            $roleTitle = trim((string) $request->input('role_title'));
            if (! empty($roleId) && Schema::hasTable('mosque_committee_roles')) {
                $role = MosqueCommitteeRole::query()
                    ->where('business_id', $businessId)
                    ->where('is_active', true)
                    ->find($roleId);
                if ($role) {
                    $roleTitle = (string) $role->name;
                }
            } else {
                $roleId = null;
            }

            $photoPath = $this->saveMemberPhoto($request);
            if (empty($photoPath)) {
                return ['success' => false, 'msg' => __('mosque::mosque.committee_member_photo').' '.__('lang_v1.required')];
            }
            $idDocPath = $this->saveMemberIdDoc($request);
            if (empty($idDocPath)) {
                return ['success' => false, 'msg' => __('mosque::mosque.committee_id_doc_file').' '.__('lang_v1.required')];
            }

            $phone = trim((string) $request->input('phone', ''));
            $waSame = (bool) $request->input('whatsapp_same_as_phone', false);
            $whatsapp = trim((string) $request->input('whatsapp_number', ''));
            if ($waSame && $phone !== '') {
                $whatsapp = $phone;
            }

            $member = MosqueCommitteeMember::query()->create([
                'business_id' => $businessId,
                'committee_id' => (int) $request->input('committee_id'),
                'member_name' => (string) $request->input('member_name'),
                'role_title' => $roleTitle !== '' ? $roleTitle : null,
                'role_id' => $roleId ? (int) $roleId : null,
                'photo_path' => $photoPath,
                'id_doc_type' => (string) $request->input('id_doc_type'),
                'id_doc_no' => (string) $request->input('id_doc_no'),
                'id_doc_path' => $idDocPath,
                'phone' => $phone !== '' ? $phone : null,
                'whatsapp_number' => $whatsapp !== '' ? $whatsapp : null,
                'whatsapp_same_as_phone' => $waSame,
                'email' => $request->input('email'),
            ]);

            if (Schema::hasColumn('mosque_committee_members', 'card_no') && empty($member->card_no)) {
                $member->update([
                    'card_no' => 'CM-'.$businessId.'-'.$member->id,
                ]);
            }

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function membersEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $member = MosqueCommitteeMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $committees = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        $roles = [];
        if (Schema::hasTable('mosque_committee_roles')) {
            $roles = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('sort_order')
                ->orderBy('name')
                ->pluck('name', 'id');
        }

        $selectedRoleId = null;
        if (Schema::hasColumn('mosque_committee_members', 'role_id') && ! empty($member->role_id)) {
            $selectedRoleId = (int) $member->role_id;
        } elseif (! empty($roles) && ! empty($member->role_title)) {
            foreach ($roles as $rid => $rname) {
                if (trim(mb_strtolower((string) $rname)) === trim(mb_strtolower((string) $member->role_title))) {
                    $selectedRoleId = (int) $rid;
                    break;
                }
            }
        }

        return view('mosque::committee.members.edit', compact('member', 'committees', 'roles', 'selectedRoleId'));
    }

    public function membersUpdate(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'committee_id' => 'required|integer',
            'member_name' => 'required|string|max:255',
            'role_id' => 'nullable|integer',
            'role_title' => 'nullable|string|max:255',
            'photo' => 'nullable|image|max:5120',
            'photo_capture' => 'nullable|string',
            'id_doc_type' => 'required|string|in:nid,passport,other',
            'id_doc_no' => 'required|string|max:64',
            'id_doc_file' => 'nullable|file|mimes:jpg,jpeg,png,webp,pdf|max:5120',
            'phone' => 'nullable|string|max:50',
            'whatsapp_number' => 'nullable|string|max:50',
            'whatsapp_same_as_phone' => 'nullable|boolean',
            'email' => 'nullable|email|max:255',
        ]);

        try {
            $businessId = $this->businessId();
            $member = MosqueCommitteeMember::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $roleId = $request->input('role_id');
            $roleTitle = trim((string) $request->input('role_title'));
            if (! empty($roleId) && Schema::hasTable('mosque_committee_roles')) {
                $role = MosqueCommitteeRole::query()
                    ->where('business_id', $businessId)
                    ->where('is_active', true)
                    ->find($roleId);
                if ($role) {
                    $roleTitle = (string) $role->name;
                }
            } else {
                $roleId = null;
            }

            $update = [
                'committee_id' => (int) $request->input('committee_id'),
                'member_name' => (string) $request->input('member_name'),
                'role_title' => $roleTitle !== '' ? $roleTitle : null,
                'role_id' => $roleId ? (int) $roleId : null,
                'id_doc_type' => (string) $request->input('id_doc_type'),
                'id_doc_no' => (string) $request->input('id_doc_no'),
                'phone' => null,
                'whatsapp_number' => null,
                'whatsapp_same_as_phone' => false,
                'email' => $request->input('email'),
            ];

            $phone = trim((string) $request->input('phone', ''));
            $waSame = (bool) $request->input('whatsapp_same_as_phone', false);
            $whatsapp = trim((string) $request->input('whatsapp_number', ''));
            if ($waSame && $phone !== '') {
                $whatsapp = $phone;
            }

            $update['phone'] = $phone !== '' ? $phone : null;
            $update['whatsapp_number'] = $whatsapp !== '' ? $whatsapp : null;
            $update['whatsapp_same_as_phone'] = $waSame;

            $photoPath = $this->saveMemberPhoto($request);
            if (! empty($photoPath)) {
                $update['photo_path'] = $photoPath;
            }
            if (empty($member->photo_path) && empty($update['photo_path'])) {
                return ['success' => false, 'msg' => __('mosque::mosque.committee_member_photo').' '.__('lang_v1.required')];
            }

            $idDocPath = $this->saveMemberIdDoc($request);
            if (! empty($idDocPath)) {
                $update['id_doc_path'] = $idDocPath;
            }
            if (empty($member->id_doc_path) && empty($update['id_doc_path'])) {
                return ['success' => false, 'msg' => __('mosque::mosque.committee_id_doc_file').' '.__('lang_v1.required')];
            }

            if (Schema::hasColumn('mosque_committee_members', 'card_no') && empty($member->card_no)) {
                $update['card_no'] = 'CM-'.$businessId.'-'.$member->id;
            }

            $member->update($update);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function memberCard(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $member = MosqueCommitteeMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->find($member->committee_id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        $photoUrl = $this->memberPhotoUrl($member->photo_path ?? null);

        $sigFile = trim((string) ($settings['payslip_signature'] ?? ''));
        $sigUrl = $sigFile !== '' ? asset('uploads/mosque/signatures/'.$sigFile) : null;
        $authName = trim((string) ($settings['payslip_authority_name'] ?? ''));
        $authTitle = trim((string) ($settings['payslip_authority_title'] ?? ''));

        return view('mosque::committee.members.card', compact('member', 'committee', 'profile', 'logoDataUri', 'settings', 'photoUrl', 'sigUrl', 'authName', 'authTitle'));
    }

    public function membersDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $member = MosqueCommitteeMember::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $member->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'committee_member', (int) $member->id, [
                'committee_id' => (int) $member->committee_id,
                'member_name' => (string) $member->member_name,
                'role_title' => (string) ($member->role_title ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'committee member', (int) $member->id, [
                'committee_id' => (int) $member->committee_id,
                'member_name' => (string) $member->member_name,
                'role_title' => (string) ($member->role_title ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Minutes
    public function minutesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueMinute::query()
            ->where('business_id', $businessId)
            ->select(['id', 'committee_id', 'minute_no', 'date']);

        if (! empty($request->input('committee_id'))) {
            $query->where('committee_id', $request->input('committee_id'));
        }

        return DataTables::of($query)
            ->addColumn('committee', function ($row) use ($businessId) {
                $committee = MosqueCommittee::query()
                    ->where('business_id', $businessId)
                    ->where('id', $row->committee_id)
                    ->first();
                return $committee?->name ?? '';
            })
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'minutesPrint'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $pdf = '<a href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'minutesPdf'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'minutesEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_minute_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\CommitteeController::class, 'minutesDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_minute"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $print.' '.$pdf.' '.$edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function minutesPrint($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $minute = MosqueMinute::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->where('id', $minute->committee_id)
            ->first();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        MosqueAuditUtil::log($businessId, 'print', 'committee_minute', (int) $minute->id, [
            'minute_no' => $minute->minute_no,
        ]);

        return view('mosque::committee.minutes.print', compact('minute', 'committee', 'profile', 'logoDataUri'));
    }

    public function minutesPdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $minute = MosqueMinute::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $committee = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->where('id', $minute->committee_id)
            ->first();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        MosqueAuditUtil::log($businessId, 'pdf', 'committee_minute', (int) $minute->id, [
            'minute_no' => $minute->minute_no,
        ]);

        $pdf = Pdf::loadView('mosque::committee.minutes.print', compact('minute', 'committee', 'profile', 'logoDataUri'))
            ->setPaper('a4');

        $filename = 'committee_minute_'.$minute->id.'.pdf';

        return $pdf->download($filename);
    }

    public function minutesCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $committees = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::committee.minutes.create', compact('committees'));
    }

    public function minutesStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'committee_id' => 'required|integer',
            'minute_no' => 'nullable|string|max:100',
            'date' => 'required|date',
            'decisions' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            MosqueMinute::query()->create(array_merge(
                $request->only(['committee_id', 'minute_no', 'date', 'decisions']),
                ['business_id' => $businessId]
            ));

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function minutesEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $minute = MosqueMinute::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $committees = MosqueCommittee::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::committee.minutes.edit', compact('minute', 'committees'));
    }

    public function minutesUpdate(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'committee_id' => 'required|integer',
            'minute_no' => 'nullable|string|max:100',
            'date' => 'required|date',
            'decisions' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $minute = MosqueMinute::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $minute->update($request->only(['committee_id', 'minute_no', 'date', 'decisions']));

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function minutesDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $minute = MosqueMinute::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $minute->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'committee_minute', (int) $minute->id, [
                'committee_id' => (int) $minute->committee_id,
                'minute_no' => (string) ($minute->minute_no ?? ''),
                'date' => (string) ($minute->date ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'committee minute', (int) $minute->id, [
                'committee_id' => (int) $minute->committee_id,
                'minute_no' => (string) ($minute->minute_no ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
