<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Yajra\DataTables\Facades\DataTables;

class CommitteeReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.committee.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function committeesSummaryData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $membersCountSql = "(SELECT COUNT(*) FROM mosque_committee_members m WHERE m.business_id = {$businessId} AND m.committee_id = c.id";
        if (Schema::hasColumn('mosque_committee_members', 'deleted_at')) {
            $membersCountSql .= " AND m.deleted_at IS NULL";
        }
        $membersCountSql .= ')';

        $minutesCountSql = "(SELECT COUNT(*) FROM mosque_minutes mm WHERE mm.business_id = {$businessId} AND mm.committee_id = c.id";
        if (Schema::hasColumn('mosque_minutes', 'deleted_at')) {
            $minutesCountSql .= " AND mm.deleted_at IS NULL";
        }
        $minutesCountSql .= ')';

        $query = DB::table('mosque_committees as c')
            ->where('c.business_id', $businessId)
            ->whereNull('c.deleted_at')
            ->select([
                'c.id',
                'c.name',
                'c.term_start',
                'c.term_end',
                DB::raw($membersCountSql.' as members_count'),
                DB::raw($minutesCountSql.' as minutes_count'),
            ]);

        if (! empty($request->input('committee_id'))) {
            $query->where('c.id', (int) $request->input('committee_id'));
        }

        return DataTables::of($query)->make(true);
    }

    public function rolesSummaryData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_committee_members as m')
            ->leftJoin('mosque_committees as c', function ($join) use ($businessId) {
                $join->on('c.id', '=', 'm.committee_id')
                    ->where('c.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_committees', 'deleted_at')) {
                    $join->whereNull('c.deleted_at');
                }
            })
            ->where('m.business_id', $businessId)
            ->whereNull('m.deleted_at')
            ->select([
                'm.committee_id',
                'c.name as committee_name',
                DB::raw("COALESCE(NULLIF(m.role_title,''),'Unknown') as role_title"),
                DB::raw('COUNT(m.id) as members_count'),
            ])
            ->groupBy('m.committee_id', 'c.name', DB::raw("COALESCE(NULLIF(m.role_title,''),'Unknown')"));

        if (! empty($request->input('committee_id'))) {
            $query->where('m.committee_id', (int) $request->input('committee_id'));
        }

        return DataTables::of($query)->make(true);
    }

    public function minutesArchiveData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_minutes as mm')
            ->leftJoin('mosque_committees as c', function ($join) use ($businessId) {
                $join->on('c.id', '=', 'mm.committee_id')
                    ->where('c.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_committees', 'deleted_at')) {
                    $join->whereNull('c.deleted_at');
                }
            })
            ->where('mm.business_id', $businessId)
            ->whereNull('mm.deleted_at')
            ->select([
                'mm.committee_id',
                'c.name as committee_name',
                DB::raw('YEAR(mm.date) as year'),
                DB::raw('COUNT(mm.id) as minutes_count'),
                DB::raw('MAX(mm.date) as last_meeting_date'),
            ])
            ->groupBy('mm.committee_id', 'c.name', DB::raw('YEAR(mm.date)'));

        if (! empty($request->input('committee_id'))) {
            $query->where('mm.committee_id', (int) $request->input('committee_id'));
        }
        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('mm.date', '>=', $request->input('start_date'))
                ->whereDate('mm.date', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->editColumn('last_meeting_date', function ($row) {
                return ! empty($row->last_meeting_date) ? \Carbon\Carbon::parse($row->last_meeting_date)->format('Y-m-d') : '';
            })
            ->make(true);
    }
}
