<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueCommitteeMember;
use Modules\Mosque\Entities\MosqueCommitteeRole;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Yajra\DataTables\Facades\DataTables;

class CommitteeRoleController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.committee.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function data()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        if (! Schema::hasTable('mosque_committee_roles')) {
            return response()->json([
                'draw' => (int) request()->input('draw', 0),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
            ]);
        }

        try {
            $query = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->select(['id', 'name', 'is_active', 'sort_order']);

            return DataTables::of($query)
                ->editColumn('is_active', function ($row) {
                    return ! empty($row->is_active) ? __('lang_v1.active') : __('lang_v1.inactive');
                })
                ->addColumn('action', function ($row) {
                    $edit = '<button data-href="'.route('mosque.committee.roles.edit', [(int) $row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_committee_role_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                    $delete = '<button data-href="'.route('mosque.committee.roles.destroy', [(int) $row->id]).'" class="btn btn-xs btn-danger delete_mosque_committee_role"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                    return $edit.' '.$delete;
                })
                ->rawColumns(['action'])
                ->make(true);
        } catch (\Throwable $e) {
            Log::error('Committee roles datatable error', [
                'business_id' => $businessId,
                'url' => (string) request()->fullUrl(),
                'message' => $e->getMessage(),
            ]);

            return response()->json([
                'draw' => (int) request()->input('draw', 0),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
            ]);
        }
    }

    public function create()
    {
        $this->ensurePermission();
        return view('mosque::committee.roles.create');
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'is_active' => 'nullable',
            'sort_order' => 'nullable|integer|min:0|max:1000000',
            'notes' => 'nullable|string|max:2000',
        ]);

        try {
            $businessId = $this->businessId();
            $name = trim((string) $request->input('name'));

            $exists = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->whereRaw('LOWER(name) = ?', [mb_strtolower($name)])
                ->exists();
            if ($exists) {
                return ['success' => false, 'msg' => __('mosque::mosque.committee_role_exists')];
            }

            $role = MosqueCommitteeRole::query()->create([
                'business_id' => $businessId,
                'name' => $name,
                'is_active' => (bool) $request->boolean('is_active', true),
                'sort_order' => (int) ($request->input('sort_order') ?? 0),
                'notes' => $request->input('notes'),
            ]);

            MosqueAuditUtil::log($businessId, 'create', 'committee_role', (int) $role->id, [
                'name' => (string) $role->name,
                'is_active' => (bool) $role->is_active,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $role = MosqueCommitteeRole::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        return view('mosque::committee.roles.edit', compact('role'));
    }

    public function update(Request $request, int $id)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'is_active' => 'nullable',
            'sort_order' => 'nullable|integer|min:0|max:1000000',
            'notes' => 'nullable|string|max:2000',
        ]);

        try {
            $businessId = $this->businessId();
            $role = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $name = trim((string) $request->input('name'));
            $exists = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->where('id', '!=', (int) $role->id)
                ->whereRaw('LOWER(name) = ?', [mb_strtolower($name)])
                ->exists();
            if ($exists) {
                return ['success' => false, 'msg' => __('mosque::mosque.committee_role_exists')];
            }

            $role->update([
                'name' => $name,
                'is_active' => (bool) $request->boolean('is_active', false),
                'sort_order' => (int) ($request->input('sort_order') ?? 0),
                'notes' => $request->input('notes'),
            ]);

            MosqueAuditUtil::log($businessId, 'update', 'committee_role', (int) $role->id, [
                'name' => (string) $role->name,
                'is_active' => (bool) $role->is_active,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy(int $id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $role = MosqueCommitteeRole::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $roleName = (string) $role->name;

            DB::transaction(function () use ($businessId, $role, $roleName) {
                if (Schema::hasTable('mosque_committee_members') && Schema::hasColumn('mosque_committee_members', 'role_id')) {
                    MosqueCommitteeMember::query()
                        ->where('business_id', $businessId)
                        ->where('role_id', (int) $role->id)
                        ->where(function ($q) {
                            $q->whereNull('role_title')->orWhere('role_title', '=', '');
                        })
                        ->update(['role_title' => $roleName]);

                    MosqueCommitteeMember::query()
                        ->where('business_id', $businessId)
                        ->where('role_id', (int) $role->id)
                        ->update(['role_id' => null]);
                }

                $role->delete();
            });

            MosqueAuditUtil::log($businessId, 'delete', 'committee_role', (int) $role->id, [
                'name' => $roleName,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'committee role', (int) $role->id, [
                'name' => $roleName,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
