<?php

namespace Modules\Mosque\Http\Controllers;

use App\Utils\ModuleUtil;
use Illuminate\Support\Facades\Schema;
use Menu;
use Modules\Mosque\Entities\MosqueLicense;
use Modules\Mosque\Entities\MosqueSetting;

class DataController
{
    /**
     * Defines module as a superadmin package.
     * Used by Superadmin -> packages to enable/disable module per business.
     *
     * @return array
     */
    public function superadmin_package(): array
    {
        return [
            [
                'name' => 'mosque_module',
                'label' => __('mosque::mosque.mosque'),
                'default' => false,
            ],
        ];
    }

    public function modifyAdminMenu(): void
    {
        if (! auth()->check()) {
            return;
        }

        $businessId = (int) session()->get('user.business_id');
        if (empty($businessId)) {
            return;
        }

        if (! Schema::hasTable('mosque_licenses')) {
            return;
        }

        $moduleUtil = new ModuleUtil();
        $isMosqueEnabled = (bool) $moduleUtil->hasThePermissionInSubscription($businessId, 'mosque_module', 'superadmin_package');
        if (! $isMosqueEnabled) {
            return;
        }

        $hasLicense = MosqueLicense::query()
            ->where('business_id', $businessId)
            ->exists();

        $anyPermission = auth()->user()->hasAnyPermission([
            'mosque.members.manage',
            'mosque.subscriptions.manage',
            'mosque.donations.manage',
            'mosque.pledges.view',
            'mosque.pledges.create',
            'mosque.pledges.edit',
            'mosque.pledges.fulfill',
            'mosque.pledges.remind',
            'mosque.pledges.delete',
            'mosque.pledges.reports',
            'mosque.finance.income',
            'mosque.finance.expense',
            'mosque.finance.reports',
            'mosque.finance.bank.manage',
            'mosque.finance.bank.transfer',
            'mosque.finance.cashout.manage',
            'mosque.committee.manage',
            'mosque.staff.manage',
            'mosque.marriage.manage',
            'mosque.death.manage',
            'mosque.assets.manage',
            'mosque.events.manage',
            'mosque.tenancy.view',
            'mosque.tenancy.edit',
            'mosque.tenancy.allocate',
            'mosque.tenancy.billing.run',
            'mosque.tenancy.requests.manage',
            'mosque.tenancy.cases.view',
            'mosque.tenancy.settings.manage',
            'mosque.audit.view',
            'mosque.manage',
            'mosque.license.manage',
        ]);

        if (! $anyPermission) {
            return;
        }

        Menu::modify('admin-sidebar-menu', function ($menu) {
            $businessId = (int) session()->get('user.business_id');
            $licensed = Schema::hasTable('mosque_licenses') && MosqueLicense::query()->where('business_id', $businessId)->exists();

            // Allow owners/admins to reach activation page even if unlicensed.
            if (! $licensed && auth()->user()->hasAnyPermission(['mosque.manage', 'mosque.license.manage'])) {
                $menu->url(
                    action([\Modules\Mosque\Http\Controllers\ActivationController::class, 'index']),
                    __('mosque::mosque.activate_title'),
                    ['icon' => 'fa fas fa-key', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'activate']
                )->order(88);
                return;
            }

            if (! $licensed) {
                return;
            }

            $menu->dropdown(
                __('mosque::mosque.mosque'),
                function ($sub) {
                    if (auth()->user()->can('mosque.members.manage') ||
                        auth()->user()->can('mosque.subscriptions.manage') ||
                        auth()->user()->can('mosque.donations.manage') ||
                        auth()->user()->can('mosque.finance.income') ||
                        auth()->user()->can('mosque.finance.expense') ||
                        auth()->user()->can('mosque.finance.reports') ||
                        auth()->user()->can('mosque.finance.bank.manage') ||
                        auth()->user()->can('mosque.finance.bank.transfer') ||
                        auth()->user()->can('mosque.finance.cashout.manage') ||
                        auth()->user()->can('mosque.committee.manage') ||
                        auth()->user()->can('mosque.staff.manage') ||
                        auth()->user()->can('mosque.marriage.manage') ||
                        auth()->user()->can('mosque.death.manage') ||
                        auth()->user()->can('mosque.assets.manage') ||
                        auth()->user()->can('mosque.events.manage') ||
                        auth()->user()->can('mosque.audit.view') ||
                        auth()->user()->can('mosque.manage') ||
                        auth()->user()->can('mosque.license.manage')
                    ) {
                        $sub->url(
                            route('mosque.dashboard'),
                            __('mosque::mosque.menu_dashboard'),
                            ['icon' => 'fa fas fa-tachometer-alt', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'dashboard']
                        );
                    }

                    if (auth()->user()->can('mosque.members.manage')) {
                        $sub->url(
                            route('mosque.members'),
                            __('mosque::mosque.menu_members'),
                            ['icon' => 'fa fas fa-users', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'members']
                        );
                    }

                    if (auth()->user()->can('mosque.subscriptions.manage')) {
                        $sub->url(
                            route('mosque.subscriptions'),
                            __('mosque::mosque.menu_membership_fees'),
                            ['icon' => 'fa fas fa-id-card', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'subscriptions']
                        );
                    }

                    if (auth()->user()->can('mosque.donations.manage')) {
                        $sub->url(
                            route('mosque.donations'),
                            __('mosque::mosque.menu_donations'),
                            ['icon' => 'fa fas fa-hand-holding-heart', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'donations']
                        );
                    }

                    $pledgesEnabled = true;
                    if (Schema::hasTable('mosque_settings')) {
                        $settingsRow = MosqueSetting::query()->where('business_id', (int) session()->get('user.business_id'))->first();
                        $settings = $settingsRow?->settings ?: [];
                        if (array_key_exists('pledges_enabled', $settings)) {
                            $pledgesEnabled = (bool) $settings['pledges_enabled'];
                        }
                    }

                    if ($pledgesEnabled && auth()->user()->can('mosque.pledges.view')) {
                        $sub->url(
                            route('mosque.pledges'),
                            __('mosque::mosque.menu_pledges'),
                            ['icon' => 'fa fas fa-handshake', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'pledges']
                        );
                    }

                    if (auth()->user()->can('mosque.finance.income') ||
                        auth()->user()->can('mosque.finance.expense') ||
                        auth()->user()->can('mosque.finance.reports') ||
                        auth()->user()->can('mosque.finance.bank.manage') ||
                        auth()->user()->can('mosque.finance.bank.transfer') ||
                        auth()->user()->can('mosque.finance.cashout.manage')) {
                        $sub->url(
                            route('mosque.finance'),
                            __('mosque::mosque.menu_finance'),
                            ['icon' => 'fa fas fa-wallet', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'finance']
                        );
                    }

                    if (auth()->user()->can('mosque.finance.bank.manage') ||
                        auth()->user()->can('mosque.finance.bank.transfer') ||
                        auth()->user()->can('mosque.finance.cashout.manage')) {
                        $sub->url(
                            route('mosque.finance.bank_management'),
                            __('mosque::mosque.menu_bank_management'),
                            ['icon' => 'fa fas fa-university', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'finance' && request()->segment(3) === 'bank-management']
                        );
                    }

                    if (auth()->user()->can('mosque.committee.manage')) {
                        $sub->url(
                            route('mosque.committee'),
                            __('mosque::mosque.menu_committee'),
                            ['icon' => 'fa fas fa-users-cog', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'committee']
                        );
                    }

                    if (auth()->user()->can('mosque.staff.manage')) {
                        $sub->url(
                            route('mosque.staff'),
                            __('mosque::mosque.menu_staff'),
                            ['icon' => 'fa fas fa-user-tie', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'staff']
                        );
                    }

                    if (auth()->user()->can('mosque.marriage.manage')) {
                        $sub->url(
                            route('mosque.marriage'),
                            __('mosque::mosque.menu_marriage'),
                            ['icon' => 'fa fas fa-ring', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'marriage']
                        );
                    }

                    if (auth()->user()->can('mosque.death.manage')) {
                        $sub->url(
                            route('mosque.death'),
                            __('mosque::mosque.menu_death'),
                            ['icon' => 'fa fas fa-book-dead', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'death']
                        );
                    }

                    if (auth()->user()->can('mosque.assets.manage')) {
                        $sub->url(
                            route('mosque.assets'),
                            __('mosque::mosque.menu_assets'),
                            ['icon' => 'fa fas fa-building', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'assets']
                        );
                    }

                    if (auth()->user()->can('mosque.tenancy.view') ||
                        auth()->user()->can('mosque.tenancy.edit') ||
                        auth()->user()->can('mosque.tenancy.allocate') ||
                        auth()->user()->can('mosque.tenancy.billing.run') ||
                        auth()->user()->can('mosque.tenancy.requests.manage') ||
                        auth()->user()->can('mosque.tenancy.cases.view') ||
                        auth()->user()->can('mosque.tenancy.settings.manage')) {
                        $sub->url(
                            route('mosque.tenancy.dashboard'),
                            __('mosque::mosque.menu_tenancy'),
                            ['icon' => 'fa fas fa-bed', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'tenancy']
                        );
                    }

                    if (auth()->user()->can('mosque.events.manage')) {
                        $sub->url(
                            route('mosque.events'),
                            __('mosque::mosque.menu_events'),
                            ['icon' => 'fa fas fa-calendar', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'events']
                        );
                    }

                    if (auth()->user()->can('mosque.audit.view')) {
                        $sub->url(
                            route('mosque.audit'),
                            __('mosque::mosque.menu_audit'),
                            ['icon' => 'fa fas fa-shield-alt', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'audit']
                        );
                    }

                    if (auth()->user()->can('mosque.manage') || auth()->user()->can('mosque.license.manage')) {
                        $sub->url(
                            route('mosque.settings'),
                            __('mosque::mosque.menu_settings'),
                            ['icon' => 'fa fas fa-cog', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'settings']
                        );
                    }

                    $sub->url(
                        route('mosque.reports'),
                        __('mosque::mosque.menu_reports'),
                        ['icon' => 'fa fas fa-chart-bar', 'active' => request()->segment(1) === 'mosque' && request()->segment(2) === 'reports']
                    );
                },
                ['icon' => 'fa fas fa-mosque', 'active' => request()->segment(1) === 'mosque']
            )->order(88);
        });
    }

    /**
     * Returns addtional js, css, html and files which
     * will be included in the app layout
     */
    public function get_additional_script(): array
    {
        $additional_css = '';

        $additional_html = '
        <div class="modal fade" id="mosque_delete_confirm_modal" tabindex="-1" role="dialog" aria-labelledby="mosqueDeleteConfirmLabel">
            <div class="modal-dialog" role="document">
                <div class="modal-content">
                    <div class="modal-header">
                        <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
                        <h4 class="modal-title" id="mosqueDeleteConfirmLabel">Confirm Delete</h4>
                    </div>
                    <div class="modal-body">
                        <p class="text-danger" style="margin-bottom:8px;"><strong>This action cannot be undone.</strong></p>
                        <p>Type <strong>CONFIRM</strong> to delete.</p>
                        <input type="text" class="form-control" id="mosque_delete_confirm_input" autocomplete="off" placeholder="Type CONFIRM"
                               oninput="var v=(this.value||&quot;&quot;).trim().toUpperCase(); var b=document.getElementById(&quot;mosque_delete_confirm_btn&quot;); if(b){b.disabled=(v!==&quot;CONFIRM&quot;);}"
                               onkeyup="var v=(this.value||&quot;&quot;).trim().toUpperCase(); var b=document.getElementById(&quot;mosque_delete_confirm_btn&quot;); if(b){b.disabled=(v!==&quot;CONFIRM&quot;);}"
                               onchange="var v=(this.value||&quot;&quot;).trim().toUpperCase(); var b=document.getElementById(&quot;mosque_delete_confirm_btn&quot;); if(b){b.disabled=(v!==&quot;CONFIRM&quot;);}">
                        <input type="hidden" id="mosque_delete_confirm_url" value="">
                        <input type="hidden" id="mosque_delete_confirm_method" value="DELETE">
                        <input type="hidden" id="mosque_delete_confirm_entity" value="">
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-danger" id="mosque_delete_confirm_btn" disabled>Delete</button>
                    </div>
                </div>
            </div>
        </div>

        <div class="modal fade" id="mosque_whatsapp_links_modal" tabindex="-1" role="dialog" aria-labelledby="mosqueWhatsAppLinksLabel">
            <div class="modal-dialog" role="document">
                <div class="modal-content">
                    <div class="modal-header">
                        <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
                        <h4 class="modal-title" id="mosqueWhatsAppLinksLabel">WhatsApp Notifications</h4>
                    </div>
                    <div class="modal-body">
                        <p class="help-block">WhatsApp web links are generated. Click each to send.</p>
                        <div id="mosque_whatsapp_links_container"></div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-primary" data-dismiss="modal">OK</button>
                    </div>
                </div>
            </div>
        </div>
        ';

        $additional_js = <<<'HTML'
<script>
(function () {
    function isMosquePage() {
        try {
            var p = (window.location && window.location.pathname) ? String(window.location.pathname) : '';
            return p.indexOf('/mosque') !== -1;
        } catch (e) {
            return false;
        }
    }

    function getJq() {
        return window.jQuery || window.$ || null;
    }

    function toast(type, msg) {
        if (window.toastr && typeof window.toastr[type] === 'function') {
            window.toastr[type](msg);
            return;
        }
        alert(msg);
    }

    function getCsrfToken() {
        var meta = document.querySelector('meta[name="csrf-token"]');
        if (meta && meta.getAttribute('content')) {
            return meta.getAttribute('content');
        }
        var tokenInput = document.querySelector('input[name="_token"]');
        if (tokenInput && tokenInput.value) {
            return tokenInput.value;
        }
        return '';
    }

    function reloadAllDataTables() {
        var jq = getJq();
        if (!jq || !jq.fn || !jq.fn.dataTable) {
            window.location.reload();
            return;
        }
        var api = jq.fn.dataTable.tables({ visible: true, api: true });
        if (api && api.length) {
            api.ajax.reload();
        } else {
            window.location.reload();
        }
    }

    function openWhatsAppLinksModal(links) {
        var jq = getJq();
        if (!jq || !links || !links.length) {
            return;
        }
        var html = '<ul style="padding-left:18px;">' + links.map(function (l) {
            var name = (l.name || 'Member').toString();
            var url = (l.url || '').toString();
            var safeName = jq('<div>').text(name).html();
            var safeUrl = jq('<div>').text(url).html();
            return '<li style="margin-bottom:6px;"><a href="' + safeUrl + '" target="_blank" rel="noopener">' + safeName + '</a></li>';
        }).join('') + '</ul>';
        jq('#mosque_whatsapp_links_container').html(html);
        jq('#mosque_whatsapp_links_modal').modal('show');
    }

    function openConfirmModal(url, method, entity) {
        var jq = getJq();
        if (!jq) {
            return;
        }
        jq('#mosque_delete_confirm_url').val(url || '');
        jq('#mosque_delete_confirm_method').val(method || 'DELETE');
        jq('#mosque_delete_confirm_entity').val(entity || '');
        jq('#mosque_delete_confirm_input').val('');
        jq('#mosque_delete_confirm_btn').prop('disabled', true);
        jq('#mosque_delete_confirm_modal').modal('show');
    }
    window.mosqueOpenDeleteConfirm = openConfirmModal;

    function isConfirmTextOk(value) {
        return String(value || '').trim().toUpperCase() === 'CONFIRM';
    }

    function setConfirmDeleteEnabled(enabled) {
        var btn = document.getElementById('mosque_delete_confirm_btn');
        if (!btn) {
            return;
        }
        btn.disabled = !enabled;
    }

    function getEntityFromClassName(className) {
        var cls = String(className || '');
        var m = cls.match(/delete_mosque_([a-z0-9_\-]+)/i);
        if (m && m[1]) {
            return m[1].replace(/_/g, ' ');
        }
        return 'item';
    }

    function hideModal() {
        var jq = getJq();
        if (jq) {
            jq('#mosque_delete_confirm_modal').modal('hide');
        }
    }

    async function submitDelete(url, method) {
        var token = getCsrfToken();
        var headers = { 'X-Requested-With': 'XMLHttpRequest', 'Accept': 'application/json' };
        if (token) {
            headers['X-CSRF-TOKEN'] = token;
        }

        var jq = getJq();
        if (jq && typeof jq.ajax === 'function') {
            return new Promise(function (resolve, reject) {
                jq.ajax({
                    method: method || 'DELETE',
                    url: url,
                    dataType: 'json',
                    headers: headers,
                    data: token ? { _token: token } : {},
                    success: resolve,
                    error: reject
                });
            });
        }

        var body = token ? ('_token=' + encodeURIComponent(token)) : '';
        var resp = await fetch(url, {
            method: method || 'DELETE',
            headers: Object.assign({ 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' }, headers),
            body: body
        });
        return await resp.json();
    }

    document.addEventListener('input', function (e) {
        if (!e || !e.target || e.target.id !== 'mosque_delete_confirm_input') {
            return;
        }
        setConfirmDeleteEnabled(isConfirmTextOk(e.target.value));
    }, true);

    document.addEventListener('click', async function (e) {
        if (!e || !e.target || e.target.id !== 'mosque_delete_confirm_btn') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();

        var urlEl = document.getElementById('mosque_delete_confirm_url');
        var methodEl = document.getElementById('mosque_delete_confirm_method');
        var url = urlEl ? urlEl.value : '';
        var method = methodEl ? (methodEl.value || 'DELETE') : 'DELETE';
        if (!url) {
            hideModal();
            return;
        }

        if (window.__mosqueDeleteConfirmSubmitting) {
            return;
        }
        window.__mosqueDeleteConfirmSubmitting = true;

        var btn = e.target;
        btn.disabled = true;

        try {
            var result = await submitDelete(url, method);
            if (result && result.success) {
                toast('success', result.msg || (window.LANG ? LANG.success : 'Deleted'));
                hideModal();
                reloadAllDataTables();
                if (result.whatsapp_links && result.whatsapp_links.length) {
                    openWhatsAppLinksModal(result.whatsapp_links);
                }
            } else {
                toast('error', (result && result.msg) ? result.msg : (window.LANG ? LANG.something_went_wrong : 'Something went wrong'));
                btn.disabled = false;
            }
        } catch (xhr) {
            btn.disabled = false;
            try {
                if (xhr && xhr.status === 419) {
                    toast('error', 'Session expired. Please refresh the page and try again.');
                    return;
                }
            } catch (e2) {}
            toast('error', (window.LANG ? LANG.something_went_wrong : 'Something went wrong'));
        }
        finally {
            window.__mosqueDeleteConfirmSubmitting = false;
        }
    }, true);

    document.addEventListener('click', function (e) {
        if (!isMosquePage()) {
            return;
        }
        var t = e && e.target ? e.target : null;
        if (t && t.nodeType === 3) {
            t = t.parentElement;
        }
        if (!t || typeof t.closest !== 'function') {
            return;
        }
        var el = t.closest('[class*="delete_mosque_"]');
        if (!el) {
            return;
        }
        var url = el.getAttribute('data-href') || el.getAttribute('href');
        if (!url) {
            return;
        }
        e.preventDefault();
        e.stopImmediatePropagation();
        openConfirmModal(url, 'DELETE', getEntityFromClassName(el.className));
    }, true);
})();
</script>
HTML;

        return [
            'additional_js' => $additional_js,
            'additional_css' => $additional_css,
            'additional_html' => $additional_html,
            'additional_views' => [],
        ];
    }

    /**
     * Defines user permissions for the module (used by Role & Package screens).
     */
    public function user_permissions(): array
    {
        return [
            ['value' => 'mosque.members.manage', 'label' => __('mosque::mosque.menu_members'), 'default' => false],
            ['value' => 'mosque.subscriptions.manage', 'label' => __('mosque::mosque.menu_membership_fees'), 'default' => false],
            ['value' => 'mosque.membership.cancel', 'label' => __('mosque::mosque.permission_membership_cancel'), 'default' => false],
            ['value' => 'mosque.membership.reactivate', 'label' => __('mosque::mosque.permission_membership_reactivate'), 'default' => false],
            ['value' => 'mosque.membership.delete', 'label' => __('mosque::mosque.permission_membership_delete'), 'default' => false],
            ['value' => 'mosque.donations.manage', 'label' => __('mosque::mosque.menu_donations'), 'default' => false],
            ['value' => 'mosque.pledges.view', 'label' => __('mosque::mosque.menu_pledges'), 'default' => false],
            ['value' => 'mosque.pledges.create', 'label' => __('mosque::mosque.permission_pledges_create'), 'default' => false],
            ['value' => 'mosque.pledges.edit', 'label' => __('mosque::mosque.permission_pledges_edit'), 'default' => false],
            ['value' => 'mosque.pledges.fulfill', 'label' => __('mosque::mosque.permission_pledges_fulfill'), 'default' => false],
            ['value' => 'mosque.pledges.remind', 'label' => __('mosque::mosque.permission_pledges_remind'), 'default' => false],
            ['value' => 'mosque.pledges.delete', 'label' => __('mosque::mosque.permission_pledges_delete'), 'default' => false],
            ['value' => 'mosque.pledges.reports', 'label' => __('mosque::mosque.permission_pledges_reports'), 'default' => false],
            ['value' => 'mosque.finance.income', 'label' => __('mosque::mosque.permission_finance_income'), 'default' => false],
            ['value' => 'mosque.finance.expense', 'label' => __('mosque::mosque.permission_finance_expense'), 'default' => false],
            ['value' => 'mosque.finance.reports', 'label' => __('mosque::mosque.permission_finance_reports'), 'default' => false],
            ['value' => 'mosque.finance.bank.manage', 'label' => __('mosque::mosque.permission_finance_bank_manage'), 'default' => false],
            ['value' => 'mosque.finance.bank.transfer', 'label' => __('mosque::mosque.permission_finance_bank_transfer'), 'default' => false],
            ['value' => 'mosque.finance.cashout.manage', 'label' => __('mosque::mosque.permission_finance_cashout_manage'), 'default' => false],
            ['value' => 'mosque.committee.manage', 'label' => __('mosque::mosque.menu_committee'), 'default' => false],
            ['value' => 'mosque.staff.manage', 'label' => __('mosque::mosque.menu_staff'), 'default' => false],
            ['value' => 'mosque.marriage.manage', 'label' => __('mosque::mosque.menu_marriage'), 'default' => false],
            ['value' => 'mosque.death.manage', 'label' => __('mosque::mosque.menu_death'), 'default' => false],
            ['value' => 'mosque.assets.manage', 'label' => __('mosque::mosque.menu_assets'), 'default' => false],
            ['value' => 'mosque.events.manage', 'label' => __('mosque::mosque.menu_events'), 'default' => false],
            ['value' => 'mosque.tenancy.view', 'label' => __('mosque::mosque.permission_tenancy_view'), 'default' => false],
            ['value' => 'mosque.tenancy.edit', 'label' => __('mosque::mosque.permission_tenancy_edit'), 'default' => false],
            ['value' => 'mosque.tenancy.allocate', 'label' => __('mosque::mosque.permission_tenancy_allocate'), 'default' => false],
            ['value' => 'mosque.tenancy.billing.run', 'label' => __('mosque::mosque.permission_tenancy_billing_run'), 'default' => false],
            ['value' => 'mosque.tenancy.requests.manage', 'label' => __('mosque::mosque.permission_tenancy_requests_manage'), 'default' => false],
            ['value' => 'mosque.tenancy.cases.view', 'label' => __('mosque::mosque.permission_tenancy_cases_view'), 'default' => false],
            ['value' => 'mosque.tenancy.settings.manage', 'label' => __('mosque::mosque.permission_tenancy_settings_manage'), 'default' => false],
            ['value' => 'mosque.audit.view', 'label' => __('mosque::mosque.permission_audit_view'), 'default' => false],
            ['value' => 'mosque.manage', 'label' => __('mosque::mosque.menu_settings'), 'default' => false],
            ['value' => 'mosque.license.manage', 'label' => __('mosque::mosque.permission_license_manage'), 'default' => false],
        ];
    }
}
