<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Barryvdh\DomPDF\Facade\Pdf;
use Modules\Mosque\Entities\MosqueDeath;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class DeathController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.death.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        return view('mosque::death.index');
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueDeath::query()
            ->where('business_id', $businessId)
            ->select(['id', 'certificate_no', 'date', 'name', 'burial_place']);

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('date', '>=', $request->input('start_date'))
                ->whereDate('date', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\DeathController::class, 'print'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $pdf = '<a href="'.action([\Modules\Mosque\Http\Controllers\DeathController::class, 'pdf'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\DeathController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_death_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\DeathController::class, 'destroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_death"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $print.' '.$pdf.' '.$edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();
        return view('mosque::death.create');
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'date' => 'required|date',
            'certificate_no' => 'nullable|string|max:100',
            'name' => 'required|string|max:255',
            'burial_place' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $certificateNo = $request->input('certificate_no');
            if (empty($certificateNo)) {
                $certificateNo = 'DEATH-'.now()->format('YmdHis').'-'.$businessId;
            }

            MosqueDeath::query()->create([
                'business_id' => $businessId,
                'date' => $request->input('date'),
                'certificate_no' => $certificateNo,
                'name' => $request->input('name'),
                'burial_place' => $request->input('burial_place'),
                'notes' => $request->input('notes'),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $death = MosqueDeath::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::death.edit', compact('death'));
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'date' => 'required|date',
            'certificate_no' => 'nullable|string|max:100',
            'name' => 'required|string|max:255',
            'burial_place' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $death = MosqueDeath::query()->where('business_id', $businessId)->findOrFail($id);

            $certificateNo = $request->input('certificate_no');
            if (empty($certificateNo)) {
                $certificateNo = $death->certificate_no;
            }

            $death->update([
                'date' => $request->input('date'),
                'certificate_no' => $certificateNo,
                'name' => $request->input('name'),
                'burial_place' => $request->input('burial_place'),
                'notes' => $request->input('notes'),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $death = MosqueDeath::query()->where('business_id', $businessId)->findOrFail($id);
            $death->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'death', (int) $death->id, [
                'certificate_no' => (string) ($death->certificate_no ?? ''),
                'date' => (string) ($death->date ?? ''),
                'name' => (string) ($death->name ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'death', (int) $death->id, [
                'certificate_no' => (string) ($death->certificate_no ?? ''),
                'name' => (string) ($death->name ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function print($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $death = MosqueDeath::query()->where('business_id', $businessId)->findOrFail($id);

        MosqueAuditUtil::log($businessId, 'print', 'death_certificate', (int) $death->id, [
            'certificate_no' => $death->certificate_no,
        ]);

        return view('mosque::death.print', compact('profile', 'logoDataUri', 'death'));
    }

    public function pdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $death = MosqueDeath::query()->where('business_id', $businessId)->findOrFail($id);

        MosqueAuditUtil::log($businessId, 'pdf', 'death_certificate', (int) $death->id, [
            'certificate_no' => $death->certificate_no,
        ]);

        $pdf = Pdf::loadView('mosque::death.print', compact('profile', 'logoDataUri', 'death'))
            ->setPaper('a4');

        $filename = 'death_certificate_'.$death->certificate_no.'.pdf';

        return $pdf->download($filename);
    }
}
