<?php

namespace Modules\Mosque\Http\Controllers;

use App\Utils\ModuleUtil;
use DNS2D;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueDonation;
use Modules\Mosque\Entities\MosqueDonationLink;
use Modules\Mosque\Entities\MosqueDonorProfile;
use Modules\Mosque\Entities\MosqueDonationQrRequest;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueLicense;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\QrPngUtil;
use Yajra\DataTables\Facades\DataTables;

class DonationQrController extends Controller
{
    private function ensureSubscriptionEnabled(int $businessId): void
    {
        $moduleUtil = new ModuleUtil();
        $enabled = (bool) $moduleUtil->hasThePermissionInSubscription($businessId, 'mosque_module', 'superadmin_package');
        if (! $enabled) {
            abort(403, 'Mosque module is not available for this business.');
        }
    }

    private function ensureQrStorageDirExists(): void
    {
        try {
            Storage::disk('public')->makeDirectory('mosque_donation_qr');
        } catch (\Throwable $e) {
            // Ignore: directory creation can fail in some environments, and storeAs may still work.
        }
    }

    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.donations.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function normalizeHexColor(string $hexColor): string
    {
        $hex = ltrim(trim($hexColor), '#');
        if (strlen($hex) === 3) {
            $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        }
        if (! preg_match('/^[0-9a-fA-F]{6}$/', $hex)) {
            $hex = '000000';
        }
        return strtoupper($hex);
    }

    private function buildQrPng(string $qrUrl, string $primaryColor = '#000000'): string
    {
        $hex = $this->normalizeHexColor($primaryColor);
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        return QrPngUtil::barcodeOutputToPng((string) DNS2D::getBarcodePNG($qrUrl, 'QRCODE', 10, 10, [$r, $g, $b]));
    }

    private function renderQrPngWithTitle(string $qrUrl, string $title, string $primaryColor = '#000000'): string
    {
        $png = $this->buildQrPng($qrUrl, $primaryColor);
        $title = trim((string) $title);
        if ($title === '') {
            return $png;
        }

        if (! function_exists('imagecreatefromstring')) {
            return $png;
        }

        $qrImg = @imagecreatefromstring($png);
        if (! $qrImg) {
            return $png;
        }

        $qrW = imagesx($qrImg);
        $qrH = imagesy($qrImg);

        $font = 5; // built-in font
        $charW = imagefontwidth($font);
        $charH = imagefontheight($font);
        $strlen = function (string $s): int {
            return function_exists('mb_strlen') ? (int) mb_strlen($s) : (int) strlen($s);
        };

        $padding = 12;
        $maxChars = max(10, (int) floor(($qrW - ($padding * 2)) / max(1, $charW)));
        $words = preg_split('/\s+/', $title) ?: [];
        $lines = [];
        $line = '';

        foreach ($words as $word) {
            $test = $line === '' ? $word : ($line.' '.$word);
            if ($strlen($test) <= $maxChars) {
                $line = $test;
                continue;
            }
            if ($line !== '') {
                $lines[] = $line;
            }
            $line = $word;
        }
        if ($line !== '') {
            $lines[] = $line;
        }
        $lines = array_slice($lines, 0, 2);

        $titleH = ($charH * count($lines)) + $padding + 6;
        $outH = $qrH + $titleH + $padding;
        $outW = $qrW;

        $out = imagecreatetruecolor($outW, $outH);
        if (! $out) {
            imagedestroy($qrImg);
            return $png;
        }

        $white = imagecolorallocate($out, 255, 255, 255);
        $black = imagecolorallocate($out, 0, 0, 0);
        imagefilledrectangle($out, 0, 0, $outW, $outH, $white);

        $y = (int) floor($padding / 2);
        foreach ($lines as $l) {
            $l = trim((string) $l);
            $lW = $charW * $strlen($l);
            $x = max(0, (int) floor(($outW - $lW) / 2));
            imagestring($out, $font, $x, $y, $l, $black);
            $y += $charH + 4;
        }

        imagecopy($out, $qrImg, 0, $titleH, 0, 0, $qrW, $qrH);
        imagedestroy($qrImg);

        ob_start();
        imagepng($out);
        $outPng = (string) ob_get_clean();
        imagedestroy($out);

        return $outPng !== '' ? $outPng : $png;
    }

    public function adminIndex(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        if (! Schema::hasTable('mosque_donation_links')) {
            abort(500, 'Donation link table missing.');
        }

        $link = MosqueDonationLink::query()->firstOrCreate(
            ['business_id' => $businessId],
            [
                'token' => Str::random(48),
                'primary_color' => '#000000',
                'title' => 'Mosque Donation',
                'enabled' => true,
                'created_by' => auth()->id(),
            ]
        );

        $primaryColor = $link->primary_color ?: '#000000';
        $title = $link->title ?: 'Mosque Donation';
        $donateUrl = route('mosque.donate.public', [$link->token]);

        $qrImageUrl = route('mosque.donations.qr.image');
        $qrDownloadUrl = route('mosque.donations.qr.download');

        return view('mosque::donations.qr', compact('link', 'primaryColor', 'title', 'donateUrl', 'qrImageUrl', 'qrDownloadUrl'));
    }

    public function adminRequestsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        if (! Schema::hasTable('mosque_donation_qr_requests')) {
            abort(500, 'QR request table missing.');
        }

        $query = MosqueDonationQrRequest::query()
            ->where('business_id', $businessId)
            ->select([
                'id',
                'type',
                'channel',
                'amount',
                'anonymous',
                'donor_name',
                'donor_phone',
                'transaction_ref',
                'transaction_image_path',
                'status',
                'reject_reason',
                'donation_id',
                'created_at',
                'approved_at',
                'rejected_at',
            ]);

        if (! empty($request->input('status'))) {
            $query->where('status', $request->input('status'));
        }
        if (! empty($request->input('type'))) {
            $query->where('type', $request->input('type'));
        }
        if (! empty($request->input('channel'))) {
            $query->where('channel', $request->input('channel'));
        }

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('created_at', '>=', $request->input('start_date'))
                ->whereDate('created_at', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->addColumn('donor', function ($row) {
                if (! empty($row->anonymous)) {
                    return __('lang_v1.none');
                }
                $name = trim((string) ($row->donor_name ?? ''));
                $phone = trim((string) ($row->donor_phone ?? ''));
                if ($name === '' && $phone === '') {
                    return __('lang_v1.none');
                }
                $html = '<div>';
                if ($name !== '') {
                    $html .= '<strong>'.e($name).'</strong>';
                }
                if ($phone !== '') {
                    $html .= ($name !== '' ? '<br>' : '').'<span><i class="fa fa-phone" style="margin-right:4px;"></i>'.e($phone).'</span>';
                }
                $html .= '</div>';
                return $html;
            })
            ->addColumn('transaction', function ($row) {
                $ref = trim((string) ($row->transaction_ref ?? ''));
                $img = trim((string) ($row->transaction_image_path ?? ''));
                $html = '<div>';
                if ($ref !== '') {
                    $html .= '<div><strong>Ref:</strong> '.e($ref).'</div>';
                }
                if ($img !== '') {
                    $url = route('mosque.donations.qr.requests.image', [(int) $row->id]);
                    $html .= '<div><a href="'.$url.'" target="_blank" rel="noopener" class="btn btn-xs btn-default"><i class="fa fa-image"></i> View</a></div>';
                }
                $html .= '</div>';
                return $html;
            })
            ->addColumn('action', function ($row) {
                $buttons = '';
                if ($row->status === 'pending') {
                    $buttons .= '<button class="btn btn-xs btn-success mosque_qr_approve" data-id="'.$row->id.'"><i class="fa fa-check"></i> Approve</button> ';
                    $buttons .= '<button class="btn btn-xs btn-danger mosque_qr_reject" data-id="'.$row->id.'"><i class="fa fa-times"></i> Reject</button>';
                } elseif ($row->status === 'approved' && ! empty($row->donation_id)) {
                    $buttons .= '<a class="btn btn-xs btn-default" target="_blank" href="'.action([DonationsController::class, 'print'], [$row->donation_id]).'"><i class="fa fa-print"></i> Receipt</a>';
                } else {
                    $buttons .= '-';
                }
                return $buttons;
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->rawColumns(['donor', 'transaction', 'amount', 'action'])
            ->make(true);
    }

    public function adminApprove(Request $request, int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        if (! Schema::hasTable('mosque_donation_qr_requests')) {
            abort(500, 'QR request table missing.');
        }

        $qr = MosqueDonationQrRequest::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        if ($qr->status !== 'pending') {
            return ['success' => false, 'msg' => 'Already processed.'];
        }

        $fundTag = $qr->type === 'zakat' ? 'zakat' : null;
        $receiptNo = 'DON-'.now()->format('YmdHis').'-'.$businessId;

        DB::transaction(function () use ($businessId, $qr, $fundTag, $receiptNo) {
            $donorId = null;

            if (! $qr->anonymous) {
                $donorPhone = trim((string) ($qr->donor_phone ?? ''));
                $donorEmail = trim((string) ($qr->donor_email ?? ''));
                $donorName = trim((string) ($qr->donor_name ?? ''));

                if ($donorEmail !== '') {
                    $donor = MosqueDonorProfile::query()->firstOrCreate(
                        ['business_id' => $businessId, 'email' => $donorEmail],
                        [
                            'name' => $donorName !== '' ? $donorName : 'Donor',
                            'phone' => $donorPhone !== '' ? $donorPhone : null,
                            'address' => $qr->donor_address,
                        ]
                    );
                    $donorId = $donor->id;
                } elseif ($donorPhone !== '') {
                    $donor = MosqueDonorProfile::query()->firstOrCreate(
                        ['business_id' => $businessId, 'phone' => $donorPhone],
                        [
                            'name' => $donorName !== '' ? $donorName : 'Donor',
                            'email' => null,
                            'address' => $qr->donor_address,
                        ]
                    );
                    $donorId = $donor->id;
                } elseif ($donorName !== '') {
                    $donor = MosqueDonorProfile::query()->create([
                        'business_id' => $businessId,
                        'name' => $donorName,
                        'phone' => $donorPhone !== '' ? $donorPhone : null,
                        'email' => null,
                        'address' => $qr->donor_address,
                    ]);
                    $donorId = $donor->id;
                }
            }

            $donation = MosqueDonation::query()->create([
                'business_id' => $businessId,
                'donor_id' => $donorId,
                'type' => $qr->type,
                'channel' => $qr->channel,
                'amount' => $qr->amount,
                'date' => now()->format('Y-m-d'),
                'anonymous' => (bool) $qr->anonymous,
                'fund_tag' => $fundTag,
                'receipt_no' => $receiptNo,
                'notes' => trim((string) ($qr->notes ?? '')),
            ]);

            $category = MosqueFinanceCategory::query()->firstOrCreate(
                ['business_id' => $businessId, 'type' => 'income', 'name' => 'Donations'],
                ['active' => true, 'sort_order' => 1]
            );

            MosqueFinanceEntry::query()->create([
                'business_id' => $businessId,
                'location_id' => null,
                'type' => 'income',
                'category_id' => $category->id,
                'amount' => $donation->amount,
                'entry_date' => $donation->date,
                'ref_module' => 'donation',
                'ref_id' => $donation->id,
                'fund_tag' => $donation->fund_tag,
                'note' => $donation->receipt_no,
                'created_by' => auth()->id(),
            ]);

            $qr->update([
                'status' => 'approved',
                'approved_by' => auth()->id(),
                'approved_at' => now(),
                'donation_id' => $donation->id,
            ]);

            MosqueAuditUtil::log($businessId, 'approve', 'donation_qr_request', (int) $qr->id, [
                'donation_id' => (int) $donation->id,
                'receipt_no' => (string) $donation->receipt_no,
                'amount' => (float) $donation->amount,
                'type' => (string) $donation->type,
            ]);
        });

        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function adminReject(Request $request, int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $request->validate([
            'reason' => 'required|string|max:2000',
        ]);

        if (! Schema::hasTable('mosque_donation_qr_requests')) {
            abort(500, 'QR request table missing.');
        }

        $qr = MosqueDonationQrRequest::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        if ($qr->status !== 'pending') {
            return ['success' => false, 'msg' => 'Already processed.'];
        }

        $qr->update([
            'status' => 'rejected',
            'rejected_by' => auth()->id(),
            'rejected_at' => now(),
            'reject_reason' => $request->input('reason'),
        ]);

        MosqueAuditUtil::log($businessId, 'reject', 'donation_qr_request', (int) $qr->id, [
            'reason' => (string) $request->input('reason'),
        ]);

        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function adminUpdate(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $request->validate([
            'title' => 'nullable|string|max:255',
            'primary_color' => 'nullable|string|max:20',
            'enabled' => 'nullable|boolean',
        ]);

        $link = MosqueDonationLink::query()->firstOrCreate(
            ['business_id' => $businessId],
            ['token' => Str::random(48), 'created_by' => auth()->id()]
        );

        $link->update([
            'title' => $request->input('title'),
            'primary_color' => $request->input('primary_color', '#000000'),
            'enabled' => (bool) $request->input('enabled', true),
        ]);

        MosqueAuditUtil::log($businessId, 'update', 'donation_link', (int) $link->id, [
            'enabled' => $link->enabled ? 1 : 0,
        ]);

        return redirect()->back()->with('status', ['success' => 1, 'msg' => __('lang_v1.updated_succesfully')]);
    }

    public function adminRegenerate(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $link = MosqueDonationLink::query()->firstOrCreate(
            ['business_id' => $businessId],
            [
                'token' => Str::random(48),
                'primary_color' => '#000000',
                'title' => 'Mosque Donation',
                'enabled' => true,
                'created_by' => auth()->id(),
            ]
        );

        $currentCount = (int) ($link->regenerate_count ?? 0);
        if ($currentCount >= 1) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => 'You can regenerate the donation link only once.']);
        }

        $link->token = Str::random(48);
        if (Schema::hasColumn('mosque_donation_links', 'regenerate_count')) {
            $link->regenerate_count = $currentCount + 1;
        }
        if (Schema::hasColumn('mosque_donation_links', 'regenerated_at')) {
            $link->regenerated_at = now();
        }
        $link->save();

        MosqueAuditUtil::log($businessId, 'update', 'donation_link', (int) $link->id, ['regenerated' => 1]);

        return redirect()->back()->with('status', ['success' => 1, 'msg' => __('lang_v1.updated_succesfully')]);
    }

    public function adminQrImage()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $link = MosqueDonationLink::query()->firstOrCreate(
            ['business_id' => $businessId],
            [
                'token' => Str::random(48),
                'primary_color' => '#000000',
                'title' => 'Mosque Donation',
                'enabled' => true,
                'created_by' => auth()->id(),
            ]
        );
        $donateUrl = route('mosque.donate.public', [$link->token]);
        $title = (string) ($link->title ?: 'Mosque Donation');
        $primaryColor = (string) ($link->primary_color ?: '#000000');

        $png = $this->renderQrPngWithTitle($donateUrl, $title, $primaryColor);
        if ($png === '') {
            abort(500, 'Failed to generate QR image.');
        }

        return response($png, 200, [
            'Content-Type' => 'image/png',
            'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        ]);
    }

    public function adminQrDownload()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $link = MosqueDonationLink::query()->firstOrCreate(
            ['business_id' => $businessId],
            [
                'token' => Str::random(48),
                'primary_color' => '#000000',
                'title' => 'Mosque Donation',
                'enabled' => true,
                'created_by' => auth()->id(),
            ]
        );
        $donateUrl = route('mosque.donate.public', [$link->token]);
        $title = (string) ($link->title ?: 'Mosque Donation');
        $primaryColor = (string) ($link->primary_color ?: '#000000');

        $png = $this->renderQrPngWithTitle($donateUrl, $title, $primaryColor);
        if ($png === '') {
            abort(500, 'Failed to generate QR image.');
        }
        $filename = 'donation_qr_'.Str::slug($title ?: 'mosque').'.png';

        return response($png, 200, [
            'Content-Type' => 'image/png',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"',
            'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        ]);
    }

    public function adminRequestImage(int $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $qr = MosqueDonationQrRequest::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $path = trim((string) ($qr->transaction_image_path ?? ''));
        if ($path === '') {
            abort(404);
        }

        if (! Storage::disk('public')->exists($path)) {
            abort(404);
        }

        $fullPath = Storage::disk('public')->path($path);
        return response()->file($fullPath, [
            'Cache-Control' => 'private, max-age=86400',
            'Content-Disposition' => 'inline; filename="'.basename($path).'"',
        ]);
    }

    public function showPublic(string $token)
    {
        if (! Schema::hasTable('mosque_donation_links')) {
            abort(404);
        }

        $link = MosqueDonationLink::query()
            ->where('token', $token)
            ->first();

        if (empty($link) || ! $link->enabled) {
            abort(404);
        }

        $businessId = (int) $link->business_id;
        $this->ensureSubscriptionEnabled($businessId);
        if (! Schema::hasTable('mosque_licenses') || ! MosqueLicense::query()->where('business_id', $businessId)->exists()) {
            return response('Mosque module is not activated for this business.', 403);
        }

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        return view('mosque::donations.public', compact('link', 'profile', 'settings'));
    }

    public function submitPublic(Request $request, string $token)
    {
        if (! Schema::hasTable('mosque_donation_links')) {
            abort(404);
        }

        $link = MosqueDonationLink::query()
            ->where('token', $token)
            ->first();

        if (empty($link) || ! $link->enabled) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => __('lang_v1.invalid_link')]);
        }

        $businessId = (int) $link->business_id;
        $this->ensureSubscriptionEnabled($businessId);
        if (! Schema::hasTable('mosque_licenses') || ! MosqueLicense::query()->where('business_id', $businessId)->exists()) {
            return redirect()->back()->with('status', ['success' => 0, 'msg' => 'Module not activated.']);
        }

        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'type' => 'required|in:zakat,sadaqah,fitrah,general,event,waqf',
            'channel' => 'required|in:cash,bank,card,online',
            'anonymous' => 'nullable|boolean',
            'donor_name' => 'nullable|string|max:255',
            'donor_email' => 'nullable|email|max:255',
            'donor_phone' => 'nullable|string|max:50',
            'donor_address' => 'nullable|string',
            'transaction_ref' => 'required|string|max:120',
            'transaction_image' => 'required|file|mimes:jpg,jpeg,png,webp|max:5120',
            'notes' => 'nullable|string|max:2000',
        ]);

        $isAnonymous = (bool) $request->input('anonymous', false);

        $imgPath = null;
        if ($request->hasFile('transaction_image')) {
            $this->ensureQrStorageDirExists();
            $file = $request->file('transaction_image');
            $ext = strtolower((string) $file->getClientOriginalExtension());
            $name = 'qr_'.date('YmdHis').'_'.Str::random(10).'.'.$ext;
            $imgPath = $file->storeAs('mosque_donation_qr', $name, 'public');
        }

        MosqueDonationQrRequest::query()->create([
            'business_id' => $businessId,
            'type' => $request->input('type'),
            'channel' => $request->input('channel'),
            'amount' => $request->input('amount'),
            'anonymous' => $isAnonymous,
            'donor_name' => $isAnonymous ? null : $request->input('donor_name'),
            'donor_email' => $isAnonymous ? null : $request->input('donor_email'),
            'donor_phone' => $isAnonymous ? null : $request->input('donor_phone'),
            'donor_address' => $isAnonymous ? null : $request->input('donor_address'),
            'transaction_ref' => $request->input('transaction_ref'),
            'transaction_image_path' => $imgPath,
            'notes' => $request->input('notes'),
            'status' => 'pending',
        ]);

        return redirect()
            ->back()
            ->with('status', ['success' => 1, 'msg' => 'Thank you. Your donation was submitted for approval.']);
    }
}
