<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Barryvdh\DomPDF\Facade\Pdf;
use Modules\Mosque\Entities\MosqueDonation;
use Modules\Mosque\Entities\MosqueDonorProfile;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class DonationsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.donations.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $donors = MosqueDonorProfile::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::donations.index', compact('donors'));
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $donationsHasDeletedAt = Schema::hasColumn('mosque_donations', 'deleted_at');
        $donorHasDeletedAt = Schema::hasColumn('mosque_donor_profiles', 'deleted_at');

        $summaryKeys = ['zakat', 'sadaqah', 'fitrah', 'general', 'event', 'waqf'];
        $typeTotals = array_fill_keys($summaryKeys, 0.0);

        $totalsQuery = MosqueDonation::query()
            ->from('mosque_donations as d')
            ->withoutGlobalScope(SoftDeletingScope::class)
            ->where('d.business_id', $businessId)
            ->when($donationsHasDeletedAt, function ($q) {
                return $q->whereNull('d.deleted_at');
            });

        $query = MosqueDonation::query()
            // Always remove the SoftDeletes scope so it won't reference the base table name
            // (it would generate `mosque_donations`.`deleted_at` which breaks when we use an alias).
            ->withoutGlobalScope(SoftDeletingScope::class)
            ->from('mosque_donations as d')
            ->leftJoin('mosque_donor_profiles as p', function ($join) use ($businessId, $donorHasDeletedAt) {
                $join->on('p.id', '=', 'd.donor_id')
                    ->where('p.business_id', '=', $businessId);
                if ($donorHasDeletedAt) {
                    $join->whereNull('p.deleted_at');
                }
            })
            ->where('d.business_id', $businessId)
            ->when($donationsHasDeletedAt, function ($q) {
                return $q->whereNull('d.deleted_at');
            })
            ->select([
                'd.id',
                'd.donor_id',
                'd.type',
                'd.channel',
                'd.amount',
                'd.date',
                'd.anonymous',
                'd.receipt_no',
                'p.name as donor_name',
                'p.phone as donor_phone',
                'p.whatsapp_number as donor_whatsapp_number',
                'p.whatsapp_same_as_phone as donor_whatsapp_same_as_phone',
            ]);

        if (! empty($request->input('type'))) {
            $query->where('d.type', $request->input('type'));
            $totalsQuery->where('d.type', $request->input('type'));
        }
        if (! empty($request->input('channel'))) {
            $query->where('d.channel', $request->input('channel'));
            $totalsQuery->where('d.channel', $request->input('channel'));
        }
        if (! empty($request->input('donor_id'))) {
            $query->where('d.donor_id', $request->input('donor_id'));
            $totalsQuery->where('d.donor_id', $request->input('donor_id'));
        }

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('d.date', '>=', $request->input('start_date'))
                ->whereDate('d.date', '<=', $request->input('end_date'));
            $totalsQuery->whereDate('d.date', '>=', $request->input('start_date'))
                ->whereDate('d.date', '<=', $request->input('end_date'));
        }

        $totalsRows = $totalsQuery
            ->selectRaw('d.type, COALESCE(SUM(d.amount), 0) as total_amount')
            ->groupBy('d.type')
            ->get();

        foreach ($totalsRows as $r) {
            $t = (string) ($r->type ?? '');
            if ($t !== '' && array_key_exists($t, $typeTotals)) {
                $typeTotals[$t] = (float) $r->total_amount;
            }
        }

        $summary = [
            'type_totals' => $typeTotals,
            'total_amount' => (float) array_sum($typeTotals),
        ];

        return DataTables::of($query)
            ->addColumn('donor', function ($row) {
                if ($row->anonymous || empty($row->donor_id)) {
                    return __('lang_v1.none');
                }

                return (string) ($row->donor_name ?? '');
            })
            ->addColumn('contact', function ($row) {
                if ($row->anonymous || empty($row->donor_id)) {
                    return __('lang_v1.none');
                }

                $name = trim((string) ($row->donor_name ?? ''));
                if ($name === '') {
                    return __('lang_v1.none');
                }

                $phone = trim((string) ($row->donor_phone ?? ''));
                $wa = trim((string) ($row->donor_whatsapp_number ?? ''));
                if ($wa === '' && ! empty($row->donor_whatsapp_same_as_phone) && $phone !== '') {
                    $wa = $phone;
                }

                $html = '<div class="text-left">';
                $html .= '<strong>'.e($name).'</strong>';

                if ($phone !== '' && $phone !== $wa) {
                    $html .= '<br><span><i class="fa fa-phone" style="margin-right:4px;"></i>'.e($phone).'</span>';
                }

                if ($wa !== '') {
                    $digits = preg_replace('/\\D+/', '', $wa);
                    if (! empty($digits)) {
                        $html .= '<br><a href="https://wa.me/'.$digits.'" target="_blank" rel="noopener"><i class="fa fa-whatsapp" style="margin-right:4px;"></i>'.e($wa).'</a>';
                    } else {
                        $html .= '<br><span><i class="fa fa-whatsapp" style="margin-right:4px;"></i>'.e($wa).'</span>';
                    }
                }

                $html .= '</div>';

                return $html;
            })
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\DonationsController::class, 'print'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\DonationsController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_donation_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\DonationsController::class, 'destroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_donation"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $print.' '.$edit.' '.$delete;
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->rawColumns(['contact', 'action', 'amount'])
            ->with(['summary' => $summary])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $donors = MosqueDonorProfile::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::donations.create', compact('donors'));
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'donor_id' => 'nullable|integer',
            'type' => 'required|in:zakat,sadaqah,fitrah,general,event,waqf',
            'channel' => 'required|in:cash,bank,card,online',
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'anonymous' => 'nullable|boolean',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $isAnonymous = (bool) $request->input('anonymous', false);

            $receiptNo = 'DON-'.now()->format('YmdHis').'-'.$businessId;
            $fundTag = $request->input('type') === 'zakat' ? 'zakat' : null;

            DB::transaction(function () use ($businessId, $request, $isAnonymous, $receiptNo, $fundTag) {
                $donation = MosqueDonation::query()->create([
                    'business_id' => $businessId,
                    'donor_id' => $isAnonymous ? null : $request->input('donor_id'),
                    'type' => $request->input('type'),
                    'channel' => $request->input('channel'),
                    'amount' => $request->input('amount'),
                    'date' => $request->input('date'),
                    'anonymous' => $isAnonymous,
                    'fund_tag' => $fundTag,
                    'receipt_no' => $receiptNo,
                    'notes' => $request->input('notes'),
                ]);

                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'income', 'name' => 'Donations'],
                    ['active' => true, 'sort_order' => 1]
                );

                MosqueFinanceEntry::query()->create([
                    'business_id' => $businessId,
                    'location_id' => null,
                    'type' => 'income',
                    'category_id' => $category->id,
                    'amount' => $donation->amount,
                    'entry_date' => $donation->date,
                    'ref_module' => 'donation',
                    'ref_id' => $donation->id,
                    'fund_tag' => $donation->fund_tag,
                    'note' => $donation->receipt_no,
                    'created_by' => auth()->id(),
                ]);
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $donation = MosqueDonation::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $donors = MosqueDonorProfile::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::donations.edit', compact('donation', 'donors'));
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'donor_id' => 'nullable|integer',
            'type' => 'required|in:zakat,sadaqah,fitrah,general,event,waqf',
            'channel' => 'required|in:cash,bank,card,online',
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'anonymous' => 'nullable|boolean',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $isAnonymous = (bool) $request->input('anonymous', false);
            $fundTag = $request->input('type') === 'zakat' ? 'zakat' : null;

            DB::transaction(function () use ($businessId, $request, $id, $isAnonymous, $fundTag) {
                $donation = MosqueDonation::query()
                    ->where('business_id', $businessId)
                    ->findOrFail($id);

                $donation->update([
                    'donor_id' => $isAnonymous ? null : $request->input('donor_id'),
                    'type' => $request->input('type'),
                    'channel' => $request->input('channel'),
                    'amount' => $request->input('amount'),
                    'date' => $request->input('date'),
                    'anonymous' => $isAnonymous,
                    'fund_tag' => $fundTag,
                    'notes' => $request->input('notes'),
                ]);

                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'income', 'name' => 'Donations'],
                    ['active' => true, 'sort_order' => 1]
                );

                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'donation')
                    ->where('ref_id', $donation->id)
                    ->update([
                        'category_id' => $category->id,
                        'amount' => $donation->amount,
                        'entry_date' => $donation->date,
                        'fund_tag' => $donation->fund_tag,
                        'note' => $donation->receipt_no,
                    ]);
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();

            $donation = MosqueDonation::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $meta = [
                'receipt_no' => (string) $donation->receipt_no,
                'type' => (string) $donation->type,
                'amount' => (float) $donation->amount,
                'date' => (string) $donation->date,
                'fund_tag' => (string) ($donation->fund_tag ?? ''),
            ];

            DB::transaction(function () use ($businessId, $id) {
                $donation = MosqueDonation::query()
                    ->where('business_id', $businessId)
                    ->findOrFail($id);
                $donation->delete();

                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'donation')
                    ->where('ref_id', $id)
                    ->delete();
            });

            MosqueAuditUtil::log($businessId, 'delete', 'donation', (int) $id, $meta);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'donation', (int) $id, $meta);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function print($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }

        $donation = MosqueDonation::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $donor = null;
        if (! $donation->anonymous && ! empty($donation->donor_id)) {
            $donor = MosqueDonorProfile::query()
                ->where('business_id', $businessId)
                ->where('id', $donation->donor_id)
                ->first();
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        MosqueAuditUtil::log($businessId, 'print', 'donation_receipt', (int) $donation->id, [
            'receipt_no' => $donation->receipt_no,
        ]);

        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        return view('mosque::donations.print', compact('profile', 'logoDataUri', 'donation', 'donor', 'settings'));
    }

    public function pdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $donation = MosqueDonation::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $donor = null;
        if (! $donation->anonymous && ! empty($donation->donor_id)) {
            $donor = MosqueDonorProfile::query()
                ->where('business_id', $businessId)
                ->where('id', $donation->donor_id)
                ->first();
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        MosqueAuditUtil::log($businessId, 'pdf', 'donation_receipt', (int) $donation->id, [
            'receipt_no' => $donation->receipt_no,
        ]);

        $pdf = Pdf::loadView('mosque::donations.print', compact('profile', 'logoDataUri', 'donation', 'donor', 'settings'))
            ->setPaper('a4');

        $filename = 'donation_receipt_'.$donation->receipt_no.'.pdf';

        return $pdf->download($filename);
    }
}
