<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Modules\Mosque\Utils\MosqueQueryUtil;
use Yajra\DataTables\Facades\DataTables;

class DonationsReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->hasAnyPermission(['mosque.donations.manage', 'mosque.finance.reports'])) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        return view('mosque::donations.reports');
    }

    public function donorsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_donations as d')
            ->leftJoin('mosque_donor_profiles as p', function ($join) use ($businessId) {
                $join->on('p.id', '=', 'd.donor_id')
                    ->where('p.business_id', '=', $businessId);

                if (\Illuminate\Support\Facades\Schema::hasColumn('mosque_donor_profiles', 'deleted_at')) {
                    $join->whereNull('p.deleted_at');
                }
            })
            ->where('d.business_id', $businessId)
            ->whereNull('d.deleted_at')
            ->select([
                DB::raw("CASE
                    WHEN d.anonymous = 1 THEN 'Anonymous'
                    WHEN d.donor_id IS NULL THEN 'Unknown'
                    ELSE COALESCE(p.name, 'Unknown')
                END as donor_name"),
                DB::raw('COUNT(d.id) as donations_count'),
                DB::raw('SUM(d.amount) as total_amount'),
                DB::raw("SUM(CASE WHEN d.type='zakat' OR d.fund_tag='zakat' THEN d.amount ELSE 0 END) as zakat_amount"),
                DB::raw('MAX(d.date) as last_donation_date'),
            ])
            ->groupBy(DB::raw("CASE
                WHEN d.anonymous = 1 THEN 'Anonymous'
                WHEN d.donor_id IS NULL THEN 'Unknown'
                ELSE COALESCE(p.name, 'Unknown')
            END"));

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('d.date', '>=', $request->input('start_date'))
                ->whereDate('d.date', '<=', $request->input('end_date'));
        }
        if (! empty($request->input('type'))) {
            $query->where('d.type', $request->input('type'));
        }
        if (! empty($request->input('channel'))) {
            $query->where('d.channel', $request->input('channel'));
        }

        return DataTables::of($query)
            ->editColumn('total_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->total_amount.'">'.$row->total_amount.'</span>';
            })
            ->editColumn('zakat_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->zakat_amount.'">'.$row->zakat_amount.'</span>';
            })
            ->rawColumns(['total_amount', 'zakat_amount'])
            ->make(true);
    }

    public function typesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_donations as d')
            ->where('d.business_id', $businessId)
            ->whereNull('d.deleted_at')
            ->select([
                'd.type',
                DB::raw('COUNT(d.id) as donations_count'),
                DB::raw('SUM(d.amount) as total_amount'),
                DB::raw("SUM(CASE WHEN d.type='zakat' OR d.fund_tag='zakat' THEN d.amount ELSE 0 END) as zakat_amount"),
            ])
            ->groupBy('d.type');

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('d.date', '>=', $request->input('start_date'))
                ->whereDate('d.date', '<=', $request->input('end_date'));
        }
        if (! empty($request->input('channel'))) {
            $query->where('d.channel', $request->input('channel'));
        }

        return DataTables::of($query)
            ->editColumn('type', function ($row) {
                return ucfirst((string) $row->type);
            })
            ->editColumn('total_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->total_amount.'">'.$row->total_amount.'</span>';
            })
            ->editColumn('zakat_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->zakat_amount.'">'.$row->zakat_amount.'</span>';
            })
            ->rawColumns(['total_amount', 'zakat_amount'])
            ->make(true);
    }

    public function annualData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $donorNameExpr = "CASE
            WHEN d.anonymous = 1 THEN 'Anonymous'
            WHEN d.donor_id IS NULL THEN 'Unknown'
            ELSE COALESCE(p.name, 'Unknown')
        END";

        $query = DB::table('mosque_donations as d')
            ->leftJoin('mosque_donor_profiles as p', function ($join) use ($businessId) {
                $join->on('p.id', '=', 'd.donor_id')
                    ->where('p.business_id', '=', $businessId);

                if (\Illuminate\Support\Facades\Schema::hasColumn('mosque_donor_profiles', 'deleted_at')) {
                    $join->whereNull('p.deleted_at');
                }
            })
            ->where('d.business_id', $businessId)
            ->whereNull('d.deleted_at')
            ->select([
                DB::raw('YEAR(d.date) as year'),
                DB::raw($donorNameExpr.' as donor_name'),
                DB::raw('COUNT(d.id) as donations_count'),
                DB::raw('SUM(d.amount) as total_amount'),
                DB::raw("SUM(CASE WHEN d.type='zakat' OR d.fund_tag='zakat' THEN d.amount ELSE 0 END) as zakat_amount"),
            ])
            ->groupBy(DB::raw('YEAR(d.date)'))
            ->groupBy(DB::raw($donorNameExpr));

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('d.date', '>=', $request->input('start_date'))
                ->whereDate('d.date', '<=', $request->input('end_date'));
        }
        if (! empty($request->input('year'))) {
            $query->whereYear('d.date', (int) $request->input('year'));
        }
        if (! empty($request->input('type'))) {
            $query->where('d.type', $request->input('type'));
        }
        if (! empty($request->input('channel'))) {
            $query->where('d.channel', $request->input('channel'));
        }

        return DataTables::of($query)
            ->editColumn('total_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->total_amount.'">'.$row->total_amount.'</span>';
            })
            ->editColumn('zakat_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->zakat_amount.'">'.$row->zakat_amount.'</span>';
            })
            ->rawColumns(['total_amount', 'zakat_amount'])
            ->make(true);
    }

    public function receiptsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $donorNameExpr = "CASE
            WHEN d.anonymous = 1 THEN 'Anonymous'
            WHEN d.donor_id IS NULL THEN 'Unknown'
            ELSE COALESCE(p.name, 'Unknown')
        END";

        $query = DB::table('mosque_donations as d')
            ->leftJoin('mosque_donor_profiles as p', function ($join) use ($businessId) {
                $join->on('p.id', '=', 'd.donor_id')
                    ->where('p.business_id', '=', $businessId);

                if (\Illuminate\Support\Facades\Schema::hasColumn('mosque_donor_profiles', 'deleted_at')) {
                    $join->whereNull('p.deleted_at');
                }
            })
            ->where('d.business_id', $businessId)
            ->whereNull('d.deleted_at')
            ->select([
                'd.id',
                'd.date',
                'd.receipt_no',
                DB::raw($donorNameExpr.' as donor_name'),
                'd.type',
                'd.channel',
                'd.amount',
            ]);

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('d.date', '>=', $request->input('start_date'))
                ->whereDate('d.date', '<=', $request->input('end_date'));
        }
        if (! empty($request->input('type'))) {
            $query->where('d.type', $request->input('type'));
        }
        if (! empty($request->input('channel'))) {
            $query->where('d.channel', $request->input('channel'));
        }

        return DataTables::of($query)
            ->editColumn('date', function ($row) {
                return ! empty($row->date) ? \Carbon\Carbon::parse($row->date)->format('Y-m-d') : '';
            })
            ->editColumn('type', function ($row) {
                return ucfirst((string) $row->type);
            })
            ->editColumn('channel', function ($row) {
                return ucfirst((string) $row->channel);
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->addColumn('action', function ($row) {
                $print = '<a href="'.route('mosque.donations.print', [$row->id]).'" class="btn btn-xs btn-default" target="_blank"><i class="fa fa-print"></i> Print</a>';
                $pdf = '<a href="'.route('mosque.donations.pdf', [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                return $print.' '.$pdf;
            })
            ->rawColumns(['amount', 'action'])
            ->make(true);
    }

    public function zakatSummary(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $donationsQuery = DB::table('mosque_donations as d')
            ->where('d.business_id', $businessId)
            ->where(function ($q) {
                $q->where('d.type', 'zakat')
                    ->orWhere('d.fund_tag', 'zakat');
            });

        $donationsQuery = MosqueQueryUtil::whereNotDeleted($donationsQuery, 'mosque_donations', 'd');

        if (! empty($startDate) && ! empty($endDate)) {
            $donationsQuery->whereDate('d.date', '>=', $startDate)
                ->whereDate('d.date', '<=', $endDate);
        }

        $zakatCollected = (float) $donationsQuery->sum('d.amount');

        $expenseQuery = DB::table('mosque_finance_entries as e')
            ->where('e.business_id', $businessId)
            ->where('e.type', 'expense')
            ->where('e.fund_tag', 'zakat');

        $expenseQuery = MosqueQueryUtil::whereNotDeleted($expenseQuery, 'mosque_finance_entries', 'e');

        if (! empty($startDate) && ! empty($endDate)) {
            $expenseQuery->whereDate('e.entry_date', '>=', $startDate)
                ->whereDate('e.entry_date', '<=', $endDate);
        }

        $zakatDistributed = (float) $expenseQuery->sum('e.amount');

        return [
            'success' => true,
            'collected' => $zakatCollected,
            'distributed' => $zakatDistributed,
            'balance' => $zakatCollected - $zakatDistributed,
        ];
    }
}
