<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueDonorProfile;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Yajra\DataTables\Facades\DataTables;

class DonorsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.donations.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        return view('mosque::donors.index');
    }

    public function data()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueDonorProfile::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'phone', 'whatsapp_number', 'whatsapp_same_as_phone', 'email', 'tax_id']);

        return DataTables::of($query)
            ->editColumn('whatsapp_number', function ($row) {
                $phone = trim((string) ($row->phone ?? ''));
                $wa = trim((string) ($row->whatsapp_number ?? ''));
                if (empty($wa) && ! empty($row->whatsapp_same_as_phone) && $phone !== '') {
                    $wa = $phone;
                }
                if ($wa === '') {
                    return '';
                }
                $digits = preg_replace('/\D+/', '', $wa);
                if (empty($digits)) {
                    return e($wa);
                }
                $url = 'https://wa.me/'.$digits;
                return '<a href="'.$url.'" target="_blank" rel="noopener"><i class="fa fa-whatsapp" style="margin-right:4px;"></i>'.e($wa).'</a>';
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\DonorsController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_donor_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\DonorsController::class, 'destroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_donor"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['whatsapp_number', 'action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();

        return view('mosque::donors.create');
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'whatsapp_number' => 'nullable|string|max:50',
            'whatsapp_same_as_phone' => 'nullable|boolean',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'tax_id' => 'nullable|string|max:100',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $phone = trim((string) $request->input('phone', ''));
            $waSame = (bool) $request->input('whatsapp_same_as_phone', false);
            $whatsapp = trim((string) $request->input('whatsapp_number', ''));
            if ($waSame) {
                $whatsapp = $phone;
            }

            MosqueDonorProfile::query()->create(array_merge(
                $request->only(['name', 'phone', 'email', 'address', 'tax_id', 'notes']),
                [
                    'whatsapp_number' => $whatsapp !== '' ? $whatsapp : null,
                    'whatsapp_same_as_phone' => $waSame,
                ],
                ['business_id' => $businessId]
            ));

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $donor = MosqueDonorProfile::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        return view('mosque::donors.edit', compact('donor'));
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'whatsapp_number' => 'nullable|string|max:50',
            'whatsapp_same_as_phone' => 'nullable|boolean',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'tax_id' => 'nullable|string|max:100',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $donor = MosqueDonorProfile::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $phone = trim((string) $request->input('phone', ''));
            $waSame = (bool) $request->input('whatsapp_same_as_phone', false);
            $whatsapp = trim((string) $request->input('whatsapp_number', ''));
            if ($waSame) {
                $whatsapp = $phone;
            }

            $donor->update(array_merge(
                $request->only(['name', 'phone', 'email', 'address', 'tax_id', 'notes']),
                [
                    'whatsapp_number' => $whatsapp !== '' ? $whatsapp : null,
                    'whatsapp_same_as_phone' => $waSame,
                ]
            ));

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $donor = MosqueDonorProfile::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $donor->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'donor', (int) $donor->id, [
                'name' => (string) $donor->name,
                'phone' => (string) ($donor->phone ?? ''),
                'email' => (string) ($donor->email ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'donor', (int) $donor->id, [
                'name' => (string) $donor->name,
                'phone' => (string) ($donor->phone ?? ''),
                'email' => (string) ($donor->email ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function importForm()
    {
        $this->ensurePermission();

        return view('mosque::donors.import');
    }

    public function importStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:2048',
        ]);

        $businessId = $this->businessId();

        $file = $request->file('file');
        $path = $file?->getRealPath();
        if (empty($path) || ! is_file($path)) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $created = 0;
        $updated = 0;
        $skipped = 0;

        $handle = fopen($path, 'r');
        if ($handle === false) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        try {
            $header = fgetcsv($handle);
            if (empty($header) || ! is_array($header)) {
                return ['success' => false, 'msg' => 'Invalid CSV header.'];
            }

            $map = [];
            foreach ($header as $index => $col) {
                $key = Str::of((string) $col)->trim()->lower()->replace(' ', '_')->replace('-', '_')->toString();
                $map[$index] = $key;
            }

            DB::beginTransaction();

            while (($row = fgetcsv($handle)) !== false) {
                if (! is_array($row) || count(array_filter($row, fn ($v) => trim((string) $v) !== '')) === 0) {
                    continue;
                }

                $data = [];
                foreach ($map as $i => $key) {
                    $data[$key] = array_key_exists($i, $row) ? trim((string) $row[$i]) : null;
                }

                $name = (string) ($data['name'] ?? '');
                if ($name === '') {
                    $skipped++;
                    continue;
                }

                $payload = [
                    'name' => $name,
                    'phone' => $data['phone'] ?? null,
                    'whatsapp_number' => $data['whatsapp_number'] ?? ($data['whatsapp'] ?? null),
                    'whatsapp_same_as_phone' => array_key_exists('whatsapp_same_as_phone', $data) ? (bool) $data['whatsapp_same_as_phone'] : false,
                    'email' => $data['email'] ?? null,
                    'address' => $data['address'] ?? null,
                    'tax_id' => $data['tax_id'] ?? null,
                    'notes' => $data['notes'] ?? null,
                ];

                $email = trim((string) ($payload['email'] ?? ''));
                $phone = trim((string) ($payload['phone'] ?? ''));
                $taxId = trim((string) ($payload['tax_id'] ?? ''));
                $waSame = ! empty($payload['whatsapp_same_as_phone']);
                $wa = trim((string) ($payload['whatsapp_number'] ?? ''));
                if ($waSame && $phone !== '') {
                    $payload['whatsapp_number'] = $phone;
                } elseif ($wa === '') {
                    $payload['whatsapp_number'] = null;
                }

                $existing = null;
                if ($email !== '') {
                    $existing = MosqueDonorProfile::query()
                        ->where('business_id', $businessId)
                        ->where('email', $email)
                        ->first();
                } elseif ($phone !== '') {
                    $existing = MosqueDonorProfile::query()
                        ->where('business_id', $businessId)
                        ->where('phone', $phone)
                        ->first();
                } elseif ($taxId !== '') {
                    $existing = MosqueDonorProfile::query()
                        ->where('business_id', $businessId)
                        ->where('tax_id', $taxId)
                        ->first();
                } else {
                    $skipped++;
                    continue;
                }

                if (! empty($existing)) {
                    $existing->update($payload);
                    $updated++;
                } else {
                    MosqueDonorProfile::query()->create(array_merge($payload, [
                        'business_id' => $businessId,
                    ]));
                    $created++;
                }
            }

            DB::commit();

            MosqueAuditUtil::log($businessId, 'import', 'donors', null, [
                'created' => $created,
                'updated' => $updated,
                'skipped' => $skipped,
            ]);

            $msg = "Import complete. Created: {$created}, Updated: {$updated}, Skipped: {$skipped}.";
            return ['success' => true, 'msg' => $msg];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        } finally {
            fclose($handle);
        }
    }
}
