<?php

namespace Modules\Mosque\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueEvent;
use Modules\Mosque\Entities\MosqueEventEntry;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class EventsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.events.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $events = MosqueEvent::query()
            ->where('business_id', $businessId)
            ->orderByDesc('start_date')
            ->pluck('name', 'id');

        return view('mosque::events.index', compact('events'));
    }

    // Events
    public function eventsData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueEvent::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'type', 'start_date', 'end_date']);

        return DataTables::of($query)
            ->addColumn('pl', function ($row) use ($businessId) {
                $income = MosqueEventEntry::query()
                    ->where('business_id', $businessId)
                    ->where('event_id', $row->id)
                    ->where('type', 'income')
                    ->sum('amount');

                $expense = MosqueEventEntry::query()
                    ->where('business_id', $businessId)
                    ->where('event_id', $row->id)
                    ->where('type', 'expense')
                    ->sum('amount');

                $net = (float) $income - (float) $expense;
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$net.'">'.$net.'</span>';
            })
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\EventsController::class, 'summaryPrint'], [$row->id], false).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $pdf = '<a href="'.action([\Modules\Mosque\Http\Controllers\EventsController::class, 'summaryPdf'], [$row->id], false).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\EventsController::class, 'eventsEdit'], [$row->id], false).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_event_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = action([\Modules\Mosque\Http\Controllers\EventsController::class, 'eventsDestroy'], [$row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'event\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}var jq=window.jQuery||window.$;if(jq){jq(\'#mosque_delete_confirm_url\').val(url);jq(\'#mosque_delete_confirm_method\').val(\'DELETE\');jq(\'#mosque_delete_confirm_entity\').val(ent);jq(\'#mosque_delete_confirm_input\').val(\'\');jq(\'#mosque_delete_confirm_btn\').prop(\'disabled\',true);jq(\'#mosque_delete_confirm_modal\').modal(\'show\');return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_event"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $print.' '.$pdf.' '.$edit.' '.$delete;
            })
            ->rawColumns(['action', 'pl'])
            ->make(true);
    }

    public function summaryPrint($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $event = MosqueEvent::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $entries = MosqueEventEntry::query()
            ->where('business_id', $businessId)
            ->where('event_id', $event->id)
            ->orderBy('date')
            ->get();

        $income = (float) $entries->where('type', 'income')->sum('amount');
        $expense = (float) $entries->where('type', 'expense')->sum('amount');
        $net = $income - $expense;

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'print', 'event_summary', (int) $event->id, [
            'income' => $income,
            'expense' => $expense,
        ]);

        return view('mosque::events.summary_print', compact('event', 'entries', 'income', 'expense', 'net', 'profile', 'logoDataUri', 'settings'));
    }

    public function summaryPdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $event = MosqueEvent::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $entries = MosqueEventEntry::query()
            ->where('business_id', $businessId)
            ->where('event_id', $event->id)
            ->orderBy('date')
            ->get();

        $income = (float) $entries->where('type', 'income')->sum('amount');
        $expense = (float) $entries->where('type', 'expense')->sum('amount');
        $net = $income - $expense;

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'pdf', 'event_summary', (int) $event->id, [
            'income' => $income,
            'expense' => $expense,
        ]);

        $pdf = Pdf::loadView('mosque::events.summary_print', compact('event', 'entries', 'income', 'expense', 'net', 'profile', 'logoDataUri', 'settings'))
            ->setPaper('a4');

        $filename = 'event_summary_'.$event->id.'.pdf';

        return $pdf->download($filename);
    }

    public function eventsCreate()
    {
        $this->ensurePermission();
        return view('mosque::events.events.create');
    }

    public function eventsStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'nullable|string|max:100',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            MosqueEvent::query()->create(array_merge(
                $request->only(['name', 'type', 'start_date', 'end_date', 'notes']),
                ['business_id' => $businessId]
            ));
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function eventsEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $event = MosqueEvent::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::events.events.edit', compact('event'));
    }

    public function eventsUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'nullable|string|max:100',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $event = MosqueEvent::query()->where('business_id', $businessId)->findOrFail($id);
            $event->update($request->only(['name', 'type', 'start_date', 'end_date', 'notes']));
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function eventsDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();

            $event = MosqueEvent::query()->where('business_id', $businessId)->find($id);
            if (! $event) {
                return ['success' => true, 'msg' => __('lang_v1.success')];
            }
            $meta = [
                'name' => (string) $event->name,
                'type' => (string) ($event->type ?? ''),
                'start_date' => (string) ($event->start_date ?? ''),
                'end_date' => (string) ($event->end_date ?? ''),
            ];

            DB::transaction(function () use ($businessId, $id) {
                $entryIds = MosqueEventEntry::query()
                    ->where('business_id', $businessId)
                    ->where('event_id', $id)
                    ->pluck('id')
                    ->toArray();

                $event = MosqueEvent::query()->where('business_id', $businessId)->find($id);
                if ($event) {
                    $event->delete();
                }

                MosqueEventEntry::query()
                    ->where('business_id', $businessId)
                    ->where('event_id', $id)
                    ->delete();

                if (! empty($entryIds)) {
                    MosqueFinanceEntry::query()
                        ->where('business_id', $businessId)
                        ->where('ref_module', 'event')
                        ->whereIn('ref_id', $entryIds)
                        ->delete();
                }
            });

            MosqueAuditUtil::log($businessId, 'delete', 'event', (int) $id, $meta);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'event', (int) $id, $meta);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Entries
    public function entriesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_event_entries as ee')
            ->join('mosque_events as e', function ($join) use ($businessId) {
                $join->on('e.id', '=', 'ee.event_id')
                    ->where('e.business_id', '=', $businessId);
            })
            ->where('ee.business_id', $businessId)
            ->whereNull('ee.deleted_at')
            ->select([
                'ee.id',
                'e.name as event_name',
                'ee.type',
                'ee.amount',
                'ee.date',
                'ee.note',
                'ee.event_id',
            ])
            ->orderByDesc('ee.date');

        if (! empty($request->input('event_id'))) {
            $query->where('ee.event_id', $request->input('event_id'));
        }
        if (! empty($request->input('type'))) {
            $query->where('ee.type', $request->input('type'));
        }

        return DataTables::of($query)
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\EventsController::class, 'entriesEdit'], [$row->id], false).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_event_entry_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $deleteUrl = action([\Modules\Mosque\Http\Controllers\EventsController::class, 'entriesDestroy'], [$row->id], false);
                $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'event entry\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}var jq=window.jQuery||window.$;if(jq){jq(\'#mosque_delete_confirm_url\').val(url);jq(\'#mosque_delete_confirm_method\').val(\'DELETE\');jq(\'#mosque_delete_confirm_entity\').val(ent);jq(\'#mosque_delete_confirm_input\').val(\'\');jq(\'#mosque_delete_confirm_btn\').prop(\'disabled\',true);jq(\'#mosque_delete_confirm_modal\').modal(\'show\');return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_event_entry"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action', 'amount'])
            ->make(true);
    }

    public function entriesCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $events = MosqueEvent::query()->where('business_id', $businessId)->orderBy('name')->pluck('name', 'id');
        return view('mosque::events.entries.create', compact('events'));
    }

    public function entriesStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'event_id' => 'required|integer',
            'type' => 'required|in:income,expense',
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'note' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();

            DB::transaction(function () use ($businessId, $request) {
                $entry = MosqueEventEntry::query()->create(array_merge(
                    $request->only(['event_id', 'type', 'amount', 'date', 'note']),
                    ['business_id' => $businessId]
                ));

                $event = MosqueEvent::query()->where('business_id', $businessId)->findOrFail($entry->event_id);
                $categoryName = $entry->type === 'income' ? 'Event income' : 'Event Expense';
                $categoryType = $entry->type;

                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => $categoryType, 'name' => $categoryName],
                    ['active' => true, 'sort_order' => 50]
                );

                MosqueFinanceEntry::query()->create([
                    'business_id' => $businessId,
                    'location_id' => null,
                    'type' => $entry->type,
                    'category_id' => $category->id,
                    'amount' => $entry->amount,
                    'entry_date' => $entry->date,
                    'ref_module' => 'event',
                    'ref_id' => $entry->id,
                    'fund_tag' => null,
                    'note' => 'Event: '.$event->name,
                    'created_by' => auth()->id(),
                ]);
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function entriesEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $entry = MosqueEventEntry::query()->where('business_id', $businessId)->findOrFail($id);
        $events = MosqueEvent::query()->where('business_id', $businessId)->orderBy('name')->pluck('name', 'id');
        return view('mosque::events.entries.edit', compact('entry', 'events'));
    }

    public function entriesUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'event_id' => 'required|integer',
            'type' => 'required|in:income,expense',
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'note' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();

            DB::transaction(function () use ($businessId, $request, $id) {
                $entry = MosqueEventEntry::query()->where('business_id', $businessId)->findOrFail($id);
                $entry->update($request->only(['event_id', 'type', 'amount', 'date', 'note']));

                $event = MosqueEvent::query()->where('business_id', $businessId)->findOrFail($entry->event_id);
                $categoryName = $entry->type === 'income' ? 'Event income' : 'Event Expense';
                $categoryType = $entry->type;
                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => $categoryType, 'name' => $categoryName],
                    ['active' => true, 'sort_order' => 50]
                );

                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'event')
                    ->where('ref_id', $entry->id)
                    ->update([
                        'type' => $entry->type,
                        'category_id' => $category->id,
                        'amount' => $entry->amount,
                        'entry_date' => $entry->date,
                        'note' => 'Event: '.$event->name,
                    ]);
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function entriesDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();

            $entry = MosqueEventEntry::query()->where('business_id', $businessId)->find($id);
            if (! $entry) {
                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'event')
                    ->where('ref_id', $id)
                    ->delete();

                return ['success' => true, 'msg' => __('lang_v1.success')];
            }
            $meta = [
                'event_id' => (int) $entry->event_id,
                'type' => (string) $entry->type,
                'amount' => (float) $entry->amount,
                'date' => (string) $entry->date,
            ];

            DB::transaction(function () use ($businessId, $id) {
                $entry = MosqueEventEntry::query()->where('business_id', $businessId)->find($id);
                if ($entry) {
                    $entry->delete();
                }

                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'event')
                    ->where('ref_id', $id)
                    ->delete();
            });

            MosqueAuditUtil::log($businessId, 'delete', 'event_entry', (int) $id, $meta);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'event entry', (int) $id, $meta);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
