<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Yajra\DataTables\Facades\DataTables;

class EventsReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.events.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function eventsSummaryData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $incomeSql = "(SELECT COALESCE(SUM(x.amount),0) FROM mosque_event_entries x WHERE x.business_id = {$businessId} AND x.event_id = e.id AND x.type='income'";
        if (Schema::hasColumn('mosque_event_entries', 'deleted_at')) {
            $incomeSql .= " AND x.deleted_at IS NULL";
        }
        $incomeSql .= ')';

        $expenseSql = "(SELECT COALESCE(SUM(x.amount),0) FROM mosque_event_entries x WHERE x.business_id = {$businessId} AND x.event_id = e.id AND x.type='expense'";
        if (Schema::hasColumn('mosque_event_entries', 'deleted_at')) {
            $expenseSql .= " AND x.deleted_at IS NULL";
        }
        $expenseSql .= ')';

        $base = DB::table('mosque_events as e')
            ->where('e.business_id', $businessId)
            ->whereNull('e.deleted_at')
            ->select([
                'e.id',
                'e.name',
                'e.type',
                'e.start_date',
                'e.end_date',
                DB::raw($incomeSql.' as income_total'),
                DB::raw($expenseSql.' as expense_total'),
            ]);

        if (! empty($request->input('event_type'))) {
            $base->where('e.type', $request->input('event_type'));
        }

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $start = $request->input('start_date');
            $end = $request->input('end_date');
            $base->where(function ($q) use ($start, $end) {
                $q->whereBetween('e.start_date', [$start, $end])
                    ->orWhereBetween('e.end_date', [$start, $end]);
            });
        }

        return DataTables::of($base)
            ->addColumn('net_total', function ($row) {
                return ((float) $row->income_total) - ((float) $row->expense_total);
            })
            ->editColumn('income_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->income_total.'">'.$row->income_total.'</span>';
            })
            ->editColumn('expense_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->expense_total.'">'.$row->expense_total.'</span>';
            })
            ->editColumn('net_total', function ($row) {
                $net = ((float) $row->income_total) - ((float) $row->expense_total);
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$net.'">'.$net.'</span>';
            })
            ->rawColumns(['income_total', 'expense_total', 'net_total'])
            ->make(true);
    }

    public function monthlyProfitLossData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_event_entries as ee')
            ->where('ee.business_id', $businessId)
            ->whereNull('ee.deleted_at')
            ->select([
                DB::raw("DATE_FORMAT(ee.date, '%Y-%m') as month_ym"),
                DB::raw("SUM(CASE WHEN ee.type='income' THEN ee.amount ELSE 0 END) as income_total"),
                DB::raw("SUM(CASE WHEN ee.type='expense' THEN ee.amount ELSE 0 END) as expense_total"),
                DB::raw("SUM(CASE WHEN ee.type='income' THEN ee.amount ELSE 0 END) - SUM(CASE WHEN ee.type='expense' THEN ee.amount ELSE 0 END) as net_total"),
            ])
            ->groupBy(DB::raw("DATE_FORMAT(ee.date, '%Y-%m')"));

        if (! empty($request->input('event_id'))) {
            $query->where('ee.event_id', (int) $request->input('event_id'));
        }
        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('ee.date', '>=', $request->input('start_date'))
                ->whereDate('ee.date', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->editColumn('income_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->income_total.'">'.$row->income_total.'</span>';
            })
            ->editColumn('expense_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->expense_total.'">'.$row->expense_total.'</span>';
            })
            ->editColumn('net_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->net_total.'">'.$row->net_total.'</span>';
            })
            ->rawColumns(['income_total', 'expense_total', 'net_total'])
            ->make(true);
    }
}
