<?php

namespace Modules\Mosque\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueFamily;
use Modules\Mosque\Entities\MosqueFamilyCode;
use Modules\Mosque\Entities\MosqueFamilyMember;
use Modules\Mosque\Entities\MosqueMember;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class FamiliesController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId) && auth()->check()) {
            $businessId = (int) (auth()->user()->business_id ?? 0);
        }
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function relationOptions(): array
    {
        return [
            'Head' => 'Head',
            'Father' => 'Father',
            'Mother' => 'Mother',
            'Parent' => 'Parent',
            'Husband' => 'Husband',
            'Wife' => 'Wife',
            'Son' => 'Son',
            'Daughter' => 'Daughter',
            'Brother' => 'Brother',
            'Sister' => 'Sister',
            'Guardian' => 'Guardian',
            'Other' => 'Other',
        ];
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.members.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        return view('mosque::families.index');
    }

    public function data()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueFamily::query()
            ->where('business_id', $businessId)
            ->select(['id', 'family_code', 'head_member_id', 'address', 'created_at']);

        if (Schema::hasColumn('mosque_families', 'deleted_at')) {
            $query->whereNull('deleted_at');
        }

        $familyCode = trim((string) request()->input('family_code', ''));
        if ($familyCode !== '') {
            $query->where('family_code', $familyCode);
        }

        $hasHead = (string) request()->input('has_head', '');
        if ($hasHead === 'yes') {
            $query->whereNotNull('head_member_id');
        } elseif ($hasHead === 'no') {
            $query->whereNull('head_member_id');
        }

        return DataTables::of($query)
            ->addColumn('head', function ($row) use ($businessId) {
                if (empty($row->head_member_id)) {
                    return '';
                }

                $member = MosqueMember::query()
                    ->where('business_id', $businessId)
                    ->where('id', $row->head_member_id)
                    ->first();

                return $member?->name ?? '';
            })
            ->addColumn('members_count', function ($row) use ($businessId) {
                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    return (int) DB::table('mosque_family_members as fm')
                        ->join('mosque_members as m', function ($join) use ($businessId) {
                            $join->on('m.id', '=', 'fm.member_id')
                                ->where('m.business_id', '=', $businessId)
                                ->whereNull('m.deleted_at');
                        })
                        ->where('fm.business_id', $businessId)
                        ->where('fm.family_id', $row->id)
                        ->count();
                }

                return (int) MosqueFamilyMember::query()
                    ->where('business_id', $businessId)
                    ->where('family_id', $row->id)
                    ->count();
            })
            ->addColumn('action', function ($row) {
                $card = '<a href="'.route('mosque.families.card.print', [$row->id]).'" class="btn btn-xs btn-default" target="_blank"><i class="fa fa-id-card"></i> Card</a>';
                $pdf = '<a href="'.route('mosque.families.card.pdf', [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FamiliesController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_family_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FamiliesController::class, 'destroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_family"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $card.' '.$pdf.' '.$edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function codesSearch(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $term = trim((string) $request->input('q', ''));

        if (Schema::hasTable('mosque_family_codes')) {
            $query = DB::table('mosque_family_codes')
                ->where('business_id', $businessId)
                ->where('active', 1)
                ->whereNull('deleted_at');

            if ($term !== '') {
                $query->where('code', 'like', '%'.$term.'%');
            }

            $codes = $query->orderBy('code')
                ->limit(20)
                ->pluck('code')
                ->all();
        } else {
            $query = DB::table('mosque_families')
                ->where('business_id', $businessId);

            if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                $query->whereNull('deleted_at');
            }

            if ($term !== '') {
                $query->where('family_code', 'like', '%'.$term.'%');
            }

            $codes = $query->whereNotNull('family_code')
                ->where('family_code', '!=', '')
                ->distinct()
                ->orderBy('family_code')
                ->limit(20)
                ->pluck('family_code')
                ->all();
        }

        $results = array_map(function ($c) {
            return ['id' => $c, 'text' => $c];
        }, $codes);

        return response()->json(['results' => $results]);
    }

    public function create()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $lastFamilyCode = (string) (MosqueFamily::query()
            ->where('business_id', $businessId)
            ->orderByDesc('id')
            ->value('family_code') ?? '');

        $suggestedFamilyCode = $this->suggestNextFamilyCode($lastFamilyCode);

        // Do not preload all members; use searchable selector in UI.
        $members = collect();

        $relationOptions = $this->relationOptions();
        return view('mosque::families.create', compact('members', 'relationOptions', 'lastFamilyCode', 'suggestedFamilyCode'));
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'family_code' => 'nullable|string|max:50',
            'head_member_id' => 'nullable|integer',
            'head_relation' => 'nullable|string|max:50',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
            'member_ids' => 'array',
            'member_ids.*' => 'nullable|integer',
            'member_relations' => 'array',
            'member_relations.*' => 'nullable|string|max:50',
        ]);

        try {
            $businessId = $this->businessId();
            $familyCode = trim((string) $request->input('family_code', ''));
            $requestedHeadMemberId = (int) $request->input('head_member_id', 0);

            if ($requestedHeadMemberId > 0) {
                $headQuery = MosqueMember::query()
                    ->where('business_id', $businessId)
                    ->where('id', $requestedHeadMemberId);
                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $headQuery->whereNull('deleted_at');
                }
                if (! $headQuery->exists()) {
                    return ['success' => false, 'msg' => __('messages.something_went_wrong')];
                }
            }

            if ($familyCode !== '' && Schema::hasTable('mosque_family_codes')) {
                MosqueFamilyCode::query()->updateOrCreate(
                    ['business_id' => $businessId, 'code' => $familyCode],
                    ['active' => true, 'created_by' => (int) (auth()->id() ?? 0) ?: null]
                );
            }

            $family = MosqueFamily::query()->create([
                'business_id' => $businessId,
                'family_code' => $familyCode !== '' ? $familyCode : null,
                'head_member_id' => $requestedHeadMemberId > 0 ? $requestedHeadMemberId : null,
                'address' => $request->input('address'),
                'notes' => $request->input('notes'),
            ]);

            $headMemberId = (int) ($family->head_member_id ?? 0);
            if ($headMemberId > 0) {
                $fm = MosqueFamilyMember::query()->firstOrCreate(
                    [
                        'business_id' => $businessId,
                        'family_id' => $family->id,
                        'member_id' => $headMemberId,
                    ],
                    ['relation' => 'Head']
                );

                $headRelation = trim((string) $request->input('head_relation', ''));
                if ($headRelation !== '') {
                    $fm->relation = $headRelation;
                    $fm->save();
                }
            }

            $memberIds = (array) $request->input('member_ids', []);
            $memberRelations = (array) $request->input('member_relations', []);
            $memberLines = [];
            foreach ($memberIds as $idx => $mid) {
                $mid = (int) $mid;
                if ($mid <= 0 || ($headMemberId > 0 && $mid === $headMemberId)) {
                    continue;
                }
                $rel = trim((string) ($memberRelations[$idx] ?? ''));
                $memberLines[$mid] = $rel !== '' ? $rel : 'Member';
            }

            if (! empty($memberLines)) {
                $validIdsQuery = MosqueMember::query()
                    ->where('business_id', $businessId)
                    ->whereIn('id', array_keys($memberLines));
                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $validIdsQuery->whereNull('deleted_at');
                }
                $validIds = $validIdsQuery->pluck('id')->all();

                foreach ($validIds as $mid) {
                    $relation = $memberLines[(int) $mid] ?? 'Member';
                    MosqueFamilyMember::query()->updateOrCreate(
                        [
                            'business_id' => $businessId,
                            'family_id' => $family->id,
                            'member_id' => (int) $mid,
                        ],
                        ['relation' => $relation]
                    );
                }
            }

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $family = MosqueFamily::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        // Do not preload all members; use searchable selector in UI.
        $members = collect();
        $headMember = null;
        if (! empty($family->head_member_id)) {
            $headMember = MosqueMember::query()
                ->where('business_id', $businessId)
                ->where('id', $family->head_member_id)
                ->first();
        }

        $relationOptions = $this->relationOptions();
        $headRelation = null;
        if (! empty($family->head_member_id)) {
            $headRelation = MosqueFamilyMember::query()
                ->where('business_id', $businessId)
                ->where('family_id', $family->id)
                ->where('member_id', $family->head_member_id)
                ->value('relation');
        }

        $familyMembers = DB::table('mosque_family_members as fm')
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'fm.member_id')
                    ->where('m.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $join->whereNull('m.deleted_at');
                }
            })
            ->where('fm.business_id', $businessId)
            ->where('fm.family_id', $family->id)
            ->when(! empty($family->head_member_id), function ($q) use ($family) {
                $q->where('fm.member_id', '!=', (int) $family->head_member_id);
            })
            ->select(['fm.member_id', 'fm.relation', 'm.name', 'm.phone'])
            ->orderBy('m.name')
            ->get();

        return view('mosque::families.edit', compact('family', 'members', 'relationOptions', 'headRelation', 'headMember', 'familyMembers'));
    }

    private function suggestNextFamilyCode(string $lastFamilyCode): string
    {
        $lastFamilyCode = trim($lastFamilyCode);
        if ($lastFamilyCode === '') {
            return '';
        }

        if (preg_match('/^(.*?)(\d+)$/', $lastFamilyCode, $m) !== 1) {
            return '';
        }

        $prefix = (string) $m[1];
        $numStr = (string) $m[2];
        $width = strlen($numStr);
        $nextNum = (int) $numStr + 1;

        return $prefix.str_pad((string) $nextNum, $width, '0', STR_PAD_LEFT);
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'family_code' => 'nullable|string|max:50',
            'head_member_id' => 'nullable|integer',
            'head_relation' => 'nullable|string|max:50',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
            'member_ids' => 'array',
            'member_ids.*' => 'nullable|integer',
            'member_relations' => 'array',
            'member_relations.*' => 'nullable|string|max:50',
        ]);

        try {
            $businessId = $this->businessId();
            $family = MosqueFamily::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $requestedHeadMemberId = (int) $request->input('head_member_id', 0);

            if ($requestedHeadMemberId > 0) {
                $headQuery = MosqueMember::query()
                    ->where('business_id', $businessId)
                    ->where('id', $requestedHeadMemberId);
                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $headQuery->whereNull('deleted_at');
                }
                if (! $headQuery->exists()) {
                    return ['success' => false, 'msg' => __('messages.something_went_wrong')];
                }
            }

            DB::transaction(function () use ($family, $businessId, $request, $requestedHeadMemberId) {
                $familyCode = trim((string) $request->input('family_code', ''));
                if ($familyCode !== '' && Schema::hasTable('mosque_family_codes')) {
                    MosqueFamilyCode::query()->updateOrCreate(
                        ['business_id' => $businessId, 'code' => $familyCode],
                        ['active' => true, 'created_by' => (int) (auth()->id() ?? 0) ?: null]
                    );
                }

                $family->update([
                    'family_code' => $familyCode !== '' ? $familyCode : null,
                    'head_member_id' => $requestedHeadMemberId > 0 ? $requestedHeadMemberId : null,
                    'address' => $request->input('address'),
                    'notes' => $request->input('notes'),
                ]);

                $desired = [];
                $headMemberId = (int) ($family->head_member_id ?? 0);
                if ($headMemberId > 0) {
                    $fm = MosqueFamilyMember::query()->firstOrCreate(
                        [
                            'business_id' => $businessId,
                            'family_id' => $family->id,
                            'member_id' => $headMemberId,
                        ],
                        ['relation' => 'Head']
                    );

                    $headRelation = trim((string) $request->input('head_relation', ''));
                    if ($headRelation !== '') {
                        $fm->relation = $headRelation;
                        $fm->save();
                    }
                    $desired[$headMemberId] = $headRelation !== '' ? $headRelation : 'Head';
                }

                $memberIds = (array) $request->input('member_ids', []);
                $memberRelations = (array) $request->input('member_relations', []);
                foreach ($memberIds as $idx => $mid) {
                    $mid = (int) $mid;
                    if ($mid <= 0 || ($headMemberId > 0 && $mid === $headMemberId)) {
                        continue;
                    }
                    $rel = trim((string) ($memberRelations[$idx] ?? ''));
                    $desired[$mid] = $rel !== '' ? $rel : 'Member';
                }

                if (! empty($desired)) {
                    $validIdsQuery = MosqueMember::query()
                        ->where('business_id', $businessId)
                        ->whereIn('id', array_keys($desired));
                    if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                        $validIdsQuery->whereNull('deleted_at');
                    }
                    $validIds = array_map('intval', $validIdsQuery->pluck('id')->all());

                    $desiredValid = [];
                    foreach ($validIds as $vid) {
                        $desiredValid[$vid] = $desired[$vid] ?? 'Member';
                    }

                    MosqueFamilyMember::query()
                        ->where('business_id', $businessId)
                        ->where('family_id', $family->id)
                        ->whereNotIn('member_id', array_keys($desiredValid))
                        ->delete();

                    foreach ($desiredValid as $mid => $rel) {
                        MosqueFamilyMember::query()->updateOrCreate(
                            [
                                'business_id' => $businessId,
                                'family_id' => $family->id,
                                'member_id' => (int) $mid,
                            ],
                            ['relation' => $rel]
                        );
                    }
                }
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $family = MosqueFamily::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $family->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'family', (int) $family->id, [
                'family_code' => (string) $family->family_code,
                'head_member_id' => (int) ($family->head_member_id ?? 0),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'family', (int) $family->id, [
                'family_code' => (string) $family->family_code,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function cardPrint($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $family = MosqueFamily::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $familyCode = trim((string) ($family->family_code ?? ''));
        $familyIdsForCard = [$family->id];
        if ($familyCode !== '') {
            $familyIdsQuery = MosqueFamily::query()
                ->where('business_id', $businessId)
                ->where('family_code', $familyCode);
            if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                $familyIdsQuery->whereNull('deleted_at');
            }
            $familyIdsForCard = $familyIdsQuery->pluck('id')->map(fn ($v) => (int) $v)->all();
            if (empty($familyIdsForCard)) {
                $familyIdsForCard = [$family->id];
            }
        }

        $head = null;
        if (! empty($family->head_member_id)) {
            $head = MosqueMember::query()
                ->where('business_id', $businessId)
                ->where('id', $family->head_member_id)
                ->first();
        }

        $memberRows = DB::table('mosque_family_members as fm')
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'fm.member_id')
                    ->where('m.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $join->whereNull('m.deleted_at');
                }
            })
            ->where('fm.business_id', $businessId)
            ->whereIn('fm.family_id', $familyIdsForCard)
            ->select([
                'm.id',
                'm.name',
                'm.phone',
                'm.status',
                'fm.relation',
                'fm.family_id',
            ])
            ->orderByRaw('CASE WHEN fm.member_id = ? THEN 0 ELSE 1 END, m.name', [(int) ($family->head_member_id ?? 0)])
            ->get();

        // Deduplicate members across the same family_code (a member can be linked multiple times).
        $membersById = [];
        foreach ($memberRows as $row) {
            $mid = (int) ($row->id ?? 0);
            if ($mid <= 0) {
                continue;
            }

            if (! isset($membersById[$mid])) {
                $membersById[$mid] = $row;
                continue;
            }

            $existing = $membersById[$mid];
            $existingRel = trim((string) ($existing->relation ?? ''));
            $newRel = trim((string) ($row->relation ?? ''));

            // Prefer relation from the requested family_id when available, otherwise keep first non-empty.
            if ((int) ($row->family_id ?? 0) === (int) $family->id && $newRel !== '') {
                $existing->relation = $newRel;
                $membersById[$mid] = $existing;
            } elseif ($existingRel === '' && $newRel !== '') {
                $existing->relation = $newRel;
                $membersById[$mid] = $existing;
            }
        }

        $members = collect(array_values($membersById))
            ->sortBy(function ($m) use ($family) {
                if ((int) ($m->id ?? 0) === (int) ($family->head_member_id ?? 0)) {
                    return '0_'.$m->name;
                }
                return '1_'.(string) ($m->name ?? '');
            })
            ->values();

        MosqueAuditUtil::log($businessId, 'print', 'family_card', (int) $family->id, [
            'family_code' => $family->family_code,
        ]);

        return view('mosque::families.card', compact('family', 'head', 'members', 'profile', 'logoDataUri'));
    }

    public function cardPdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $family = MosqueFamily::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $familyCode = trim((string) ($family->family_code ?? ''));
        $familyIdsForCard = [$family->id];
        if ($familyCode !== '') {
            $familyIdsQuery = MosqueFamily::query()
                ->where('business_id', $businessId)
                ->where('family_code', $familyCode);
            if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                $familyIdsQuery->whereNull('deleted_at');
            }
            $familyIdsForCard = $familyIdsQuery->pluck('id')->map(fn ($v) => (int) $v)->all();
            if (empty($familyIdsForCard)) {
                $familyIdsForCard = [$family->id];
            }
        }

        $head = null;
        if (! empty($family->head_member_id)) {
            $head = MosqueMember::query()
                ->where('business_id', $businessId)
                ->where('id', $family->head_member_id)
                ->first();
        }

        $memberRows = DB::table('mosque_family_members as fm')
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'fm.member_id')
                    ->where('m.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $join->whereNull('m.deleted_at');
                }
            })
            ->where('fm.business_id', $businessId)
            ->whereIn('fm.family_id', $familyIdsForCard)
            ->select([
                'm.id',
                'm.name',
                'm.phone',
                'm.status',
                'fm.relation',
                'fm.family_id',
            ])
            ->orderByRaw('CASE WHEN fm.member_id = ? THEN 0 ELSE 1 END, m.name', [(int) ($family->head_member_id ?? 0)])
            ->get();

        $membersById = [];
        foreach ($memberRows as $row) {
            $mid = (int) ($row->id ?? 0);
            if ($mid <= 0) {
                continue;
            }

            if (! isset($membersById[$mid])) {
                $membersById[$mid] = $row;
                continue;
            }

            $existing = $membersById[$mid];
            $existingRel = trim((string) ($existing->relation ?? ''));
            $newRel = trim((string) ($row->relation ?? ''));

            if ((int) ($row->family_id ?? 0) === (int) $family->id && $newRel !== '') {
                $existing->relation = $newRel;
                $membersById[$mid] = $existing;
            } elseif ($existingRel === '' && $newRel !== '') {
                $existing->relation = $newRel;
                $membersById[$mid] = $existing;
            }
        }

        $members = collect(array_values($membersById))
            ->sortBy(function ($m) use ($family) {
                if ((int) ($m->id ?? 0) === (int) ($family->head_member_id ?? 0)) {
                    return '0_'.$m->name;
                }
                return '1_'.(string) ($m->name ?? '');
            })
            ->values();

        MosqueAuditUtil::log($businessId, 'pdf', 'family_card', (int) $family->id, [
            'family_code' => $family->family_code,
        ]);

        $pdf = Pdf::loadView('mosque::families.card', compact('family', 'head', 'members', 'profile', 'logoDataUri'))
            ->setPaper('a4');

        $filename = 'family_card_'.($family->family_code ?: $family->id).'.pdf';

        return $pdf->download($filename);
    }
}
