<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Yajra\DataTables\Facades\DataTables;

class FamiliesReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.members.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        return view('mosque::families.reports');
    }

    public function familiesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $familyDeletedFilter = ' AND f2.deleted_at IS NULL';
        $memberDeletedFilter = Schema::hasColumn('mosque_members', 'deleted_at') ? ' AND m2.deleted_at IS NULL' : '';

        $membersCountSql = '(SELECT COUNT(DISTINCT fm.member_id) '.
            'FROM mosque_family_members fm '.
            'JOIN mosque_families f2 ON f2.id = fm.family_id AND f2.business_id = '.$businessId.$familyDeletedFilter.' '.
            'JOIN mosque_members m2 ON m2.id = fm.member_id AND m2.business_id = '.$businessId.$memberDeletedFilter.' '.
            'WHERE fm.business_id = '.$businessId.' AND ( '.
                '((f.family_code IS NOT NULL AND f.family_code != \'\') AND f2.family_code = f.family_code) '.
                'OR ((f.family_code IS NULL OR f.family_code = \'\') AND fm.family_id = f.id) '.
            '))';

        $membersListSql = '(SELECT GROUP_CONCAT(DISTINCT '.
                'TRIM(CONCAT(m2.name, CASE WHEN m2.phone IS NOT NULL AND m2.phone != \'\' THEN CONCAT(\' (\', m2.phone, \')\') ELSE \'\' END)) '.
                'ORDER BY m2.name SEPARATOR \', \') '.
            'FROM mosque_family_members fm '.
            'JOIN mosque_families f2 ON f2.id = fm.family_id AND f2.business_id = '.$businessId.$familyDeletedFilter.' '.
            'JOIN mosque_members m2 ON m2.id = fm.member_id AND m2.business_id = '.$businessId.$memberDeletedFilter.' '.
            'WHERE fm.business_id = '.$businessId.' AND ( '.
                '((f.family_code IS NOT NULL AND f.family_code != \'\') AND f2.family_code = f.family_code) '.
                'OR ((f.family_code IS NULL OR f.family_code = \'\') AND fm.family_id = f.id) '.
            '))';

        $query = DB::table('mosque_families as f')
            ->leftJoin('mosque_members as head', function ($join) use ($businessId) {
                $join->on('head.id', '=', 'f.head_member_id')
                    ->where('head.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $join->whereNull('head.deleted_at');
                }
            })
            ->where('f.business_id', $businessId)
            ->whereNull('f.deleted_at')
            ->select([
                'f.id',
                'f.family_code',
                'f.address',
                'head.name as head_name',
                DB::raw($membersCountSql.' as members_count'),
                DB::raw($membersListSql.' as members_list'),
            ]);

        if (! empty($request->input('has_head'))) {
            if ($request->input('has_head') === 'yes') {
                $query->whereNotNull('f.head_member_id');
            } elseif ($request->input('has_head') === 'no') {
                $query->whereNull('f.head_member_id');
            }
        }

        $familyCode = trim((string) $request->input('family_code', ''));
        if ($familyCode !== '') {
            $query->where('f.family_code', $familyCode);
        }

        $headSearch = trim((string) $request->input('head_search', ''));
        if ($headSearch !== '') {
            $query->where('head.name', 'like', '%'.$headSearch.'%');
        }

        $memberSearch = trim((string) $request->input('member_search', ''));
        if ($memberSearch !== '') {
            $query->whereExists(function ($q) use ($businessId, $memberSearch) {
                $q->select(DB::raw(1))
                    ->from('mosque_family_members as fm')
                    ->join('mosque_families as f2', function ($join) use ($businessId) {
                        $join->on('f2.id', '=', 'fm.family_id')
                            ->where('f2.business_id', '=', $businessId)
                            ->whereNull('f2.deleted_at');
                    })
                    ->join('mosque_members as m2', function ($join) use ($businessId) {
                        $join->on('m2.id', '=', 'fm.member_id')
                            ->where('m2.business_id', '=', $businessId);
                        if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                            $join->whereNull('m2.deleted_at');
                        }
                    })
                    ->whereColumn('fm.business_id', 'f.business_id')
                    ->where(function ($where) {
                        $where->whereRaw("((f.family_code IS NOT NULL AND f.family_code != '') AND f2.family_code = f.family_code)")
                            ->orWhereRaw("((f.family_code IS NULL OR f.family_code = '') AND fm.family_id = f.id)");
                    })
                    ->where(function ($w) use ($memberSearch) {
                        $w->where('m2.name', 'like', '%'.$memberSearch.'%')
                            ->orWhere('m2.phone', 'like', '%'.$memberSearch.'%')
                            ->orWhere('m2.email', 'like', '%'.$memberSearch.'%');
                    });
            });
        }

        $minMembers = trim((string) $request->input('members_min', ''));
        if ($minMembers !== '' && is_numeric($minMembers)) {
            $query->havingRaw('members_count >= ?', [(int) $minMembers]);
        }
        $maxMembers = trim((string) $request->input('members_max', ''));
        if ($maxMembers !== '' && is_numeric($maxMembers)) {
            $query->havingRaw('members_count <= ?', [(int) $maxMembers]);
        }

        return DataTables::of($query)
            ->addColumn('members_details', function ($row) {
                $list = trim((string) ($row->members_list ?? ''));
                if ($list === '') {
                    return '<span class="text-muted">-</span>';
                }

                $items = array_filter(array_map('trim', explode(',', $list)));
                if (empty($items)) {
                    return '<span class="text-muted">-</span>';
                }

                $out = '<div style="max-width:520px;white-space:normal;">';
                foreach ($items as $i => $txt) {
                    $out .= '<div>'.e($txt).'</div>';
                    if ($i >= 9 && count($items) > 10) {
                        $out .= '<div class="text-muted">+'.(count($items) - 10).' more</div>';
                        break;
                    }
                }
                $out .= '</div>';

                return $out;
            })
            ->addColumn('action', function ($row) {
                $card = '<a href="'.route('mosque.families.card.print', [$row->id]).'" class="btn btn-xs btn-default" target="_blank"><i class="fa fa-id-card"></i> Card</a>';
                $pdf = '<a href="'.route('mosque.families.card.pdf', [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FamiliesController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_family_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.view').'</button>';
                return $card.' '.$pdf.' '.$edit;
            })
            ->rawColumns(['members_details', 'action'])
            ->make(true);
    }
}
