<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueFamilyCode;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Yajra\DataTables\Facades\DataTables;

class FamilyCodesController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId) && auth()->check()) {
            $businessId = (int) (auth()->user()->business_id ?? 0);
        }
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.members.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_family_codes')) {
            abort(404);
        }

        return view('mosque::families.codes.index');
    }

    public function data()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        if (! Schema::hasTable('mosque_family_codes')) {
            abort(404);
        }

        $query = MosqueFamilyCode::query()
            ->where('business_id', $businessId)
            ->select(['id', 'code', 'active', 'description', 'created_at']);

        return DataTables::of($query)
            ->editColumn('active', function ($row) {
                return ! empty($row->active) ? __('lang_v1.yes') : __('lang_v1.no');
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.route('mosque.families.codes.edit', [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_family_code_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.route('mosque.families.codes.destroy', [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_family_code"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_family_codes')) {
            abort(404);
        }

        return view('mosque::families.codes.create');
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_family_codes')) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $request->validate([
            'code' => 'required|string|max:50',
            'active' => 'nullable|boolean',
            'description' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $code = trim((string) $request->input('code'));

            $row = MosqueFamilyCode::query()->updateOrCreate(
                [
                    'business_id' => $businessId,
                    'code' => $code,
                ],
                [
                    'active' => (bool) $request->boolean('active', true),
                    'description' => $request->input('description'),
                    'created_by' => (int) (auth()->id() ?? 0) ?: null,
                ]
            );

            MosqueAuditUtil::log($businessId, 'create', 'family_code', (int) $row->id, [
                'code' => (string) $row->code,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_family_codes')) {
            abort(404);
        }

        $businessId = $this->businessId();
        $code = MosqueFamilyCode::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        return view('mosque::families.codes.edit', compact('code'));
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_family_codes')) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $request->validate([
            'code' => 'required|string|max:50',
            'active' => 'nullable|boolean',
            'description' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();

            $row = MosqueFamilyCode::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $row->update([
                'code' => trim((string) $request->input('code')),
                'active' => (bool) $request->boolean('active', true),
                'description' => $request->input('description'),
            ]);

            MosqueAuditUtil::log($businessId, 'update', 'family_code', (int) $row->id, [
                'code' => (string) $row->code,
                'active' => (int) ($row->active ? 1 : 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_family_codes')) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        try {
            $businessId = $this->businessId();

            $row = MosqueFamilyCode::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $row->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'family_code', (int) $row->id, [
                'code' => (string) $row->code,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'family_code', (int) $row->id, [
                'code' => (string) $row->code,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}

