<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Yajra\DataTables\Facades\DataTables;

class FinanceController extends Controller
{
    private const OPENING_BALANCE_REF_MODULE = 'opening_balance';

    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->hasAnyPermission(['mosque.finance.income', 'mosque.finance.expense', 'mosque.finance.reports'])) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureTypePermission(string $type): void
    {
        if ($type === 'income' && ! auth()->user()->hasAnyPermission(['mosque.finance.income', 'mosque.finance.reports'])) {
            abort(403, 'Unauthorized action.');
        }
        if ($type === 'expense' && ! auth()->user()->hasAnyPermission(['mosque.finance.expense', 'mosque.finance.reports'])) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        $businessId = $this->businessId();

        $categories = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->where('active', true)
            ->orderBy('type')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        return view('mosque::finance.index', compact('categories'));
    }

    public function entriesCreate(string $type)
    {
        $this->ensurePermission();
        $type = strtolower($type);
        if (! in_array($type, ['income', 'expense'], true)) {
            abort(404);
        }
        $this->ensureTypePermission($type);

        $businessId = $this->businessId();
        $categories = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->where('type', $type)
            ->where('active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::finance.entries.create', compact('type', 'categories'));
    }

    public function entriesStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'type' => 'required|in:income,expense',
            'category_id' => 'required|integer',
            'amount' => 'required|numeric|min:0',
            'entry_date' => 'required|date',
            'note' => 'nullable|string|max:2000',
        ]);

        try {
            $businessId = $this->businessId();
            $type = (string) $request->input('type');
            $this->ensureTypePermission($type);

            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->where('id', $request->input('category_id'))
                ->where('type', $type)
                ->firstOrFail();

            $entry = MosqueFinanceEntry::query()->create([
                'business_id' => $businessId,
                'location_id' => null,
                'type' => $type,
                'category_id' => $category->id,
                'amount' => $request->input('amount'),
                'entry_date' => $request->input('entry_date'),
                'ref_module' => 'custom',
                'ref_id' => null,
                'fund_tag' => null,
                'note' => $request->input('note'),
                'created_by' => auth()->id(),
            ]);

            MosqueAuditUtil::log($businessId, 'create', 'finance_entry', (int) $entry->id, [
                'type' => $entry->type,
                'amount' => $entry->amount,
                'entry_date' => (string) $entry->entry_date,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function openingBalanceCreate()
    {
        $this->ensurePermission();

        return view('mosque::finance.opening_balance.create');
    }

    public function openingBalanceHistory()
    {
        $this->ensurePermission();

        return view('mosque::finance.opening_balance.history');
    }

    public function openingBalanceData(Request $request)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            [$startDate, $endDate] = $this->resolveDateRange($request);
            $type = $request->input('type');

            $query = DB::table('mosque_finance_entries as mfe')
                ->leftJoin('mosque_finance_categories as mfc', 'mfc.id', '=', 'mfe.category_id')
                ->where('mfe.business_id', $businessId)
                ->whereNull('mfe.deleted_at')
                ->where('mfe.ref_module', self::OPENING_BALANCE_REF_MODULE)
                ->whereDate('mfe.entry_date', '>=', $startDate)
                ->whereDate('mfe.entry_date', '<=', $endDate)
                ->select([
                    'mfe.id',
                    'mfe.entry_date',
                    'mfe.type',
                    'mfe.amount',
                    'mfe.note',
                    'mfe.created_at',
                    'mfc.name as category',
                ]);

            if (! empty($type) && in_array($type, ['income', 'expense'], true)) {
                $query->where('mfe.type', $type);
            }

            return DataTables::of($query)
                ->addColumn('action', function ($row) {
                    $can = ($row->type === 'income' && auth()->user()->hasAnyPermission(['mosque.finance.income', 'mosque.finance.reports']))
                        || ($row->type === 'expense' && auth()->user()->hasAnyPermission(['mosque.finance.expense', 'mosque.finance.reports']));
                    if (! $can) {
                        return '';
                    }

                    $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FinanceController::class, 'entriesEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_finance_entry_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                    $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FinanceController::class, 'entriesDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_finance_entry"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                    return $edit.' '.$delete;
                })
                ->editColumn('amount', function ($row) {
                    $v = (float) ($row->amount ?? 0);
                    return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$v.'">'.$v.'</span>';
                })
                ->rawColumns(['amount', 'action'])
                ->make(true);
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return response()->json([
                'draw' => (int) $request->input('draw', 0),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => __('messages.something_went_wrong'),
            ]);
        }
    }

    private function getOrCreateCategory(int $businessId, string $type, string $name): MosqueFinanceCategory
    {
        $category = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->where('type', $type)
            ->where('name', $name)
            ->first();

        if ($category) {
            return $category;
        }

        return MosqueFinanceCategory::query()->create([
            'business_id' => $businessId,
            'type' => $type,
            'name' => $name,
            'active' => true,
            'sort_order' => 0,
        ]);
    }

    public function openingBalanceStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'as_of_date' => 'required|date',
            'opening_income' => 'nullable|numeric|min:0',
            'opening_expense' => 'nullable|numeric|min:0',
            'note' => 'nullable|string|max:2000',
        ]);

        try {
            $businessId = $this->businessId();

            $asOf = \Carbon\Carbon::parse($request->input('as_of_date'))->startOfDay();
            $entryDate = $asOf->copy();
            $refId = (int) $asOf->format('Ymd');

            $openingIncome = (float) ($request->input('opening_income') ?? 0);
            $openingExpense = (float) ($request->input('opening_expense') ?? 0);
            if ($openingIncome <= 0 && $openingExpense <= 0) {
                return ['success' => false, 'msg' => 'Please enter Opening Income or Opening Expense.'];
            }

            $note = trim('Opening balance (as of '.$asOf->format('Y-m-d').')'.($request->filled('note') ? ' - '.$request->input('note') : ''));

            DB::transaction(function () use ($businessId, $refId, $entryDate, $openingIncome, $openingExpense, $note) {
                if ($openingIncome > 0) {
                    $this->ensureTypePermission('income');
                    $cat = $this->getOrCreateCategory($businessId, 'income', 'Opening Balance');

                    $entry = MosqueFinanceEntry::query()
                        ->where('business_id', $businessId)
                        ->where('ref_module', self::OPENING_BALANCE_REF_MODULE)
                        ->where('ref_id', $refId)
                        ->where('type', 'income')
                        ->first();

                    $isNew = ! $entry;
                    if (! $entry) {
                        $entry = new MosqueFinanceEntry();
                        $entry->business_id = $businessId;
                        $entry->ref_module = self::OPENING_BALANCE_REF_MODULE;
                        $entry->ref_id = $refId;
                        $entry->type = 'income';
                        $entry->location_id = null;
                        $entry->created_by = auth()->id();
                    }

                    $entry->category_id = $cat->id;
                    $entry->amount = $openingIncome;
                    $entry->entry_date = $entryDate->format('Y-m-d');
                    $entry->fund_tag = null;
                    $entry->note = $note;
                    $entry->save();

                    MosqueAuditUtil::log($businessId, $isNew ? 'create' : 'update', 'finance_entry', (int) $entry->id, [
                        'ref_module' => self::OPENING_BALANCE_REF_MODULE,
                        'ref_id' => $refId,
                        'type' => $entry->type,
                        'amount' => $entry->amount,
                        'entry_date' => (string) $entry->entry_date,
                    ]);
                }

                if ($openingExpense > 0) {
                    $this->ensureTypePermission('expense');
                    $cat = $this->getOrCreateCategory($businessId, 'expense', 'Opening Balance');

                    $entry = MosqueFinanceEntry::query()
                        ->where('business_id', $businessId)
                        ->where('ref_module', self::OPENING_BALANCE_REF_MODULE)
                        ->where('ref_id', $refId)
                        ->where('type', 'expense')
                        ->first();

                    $isNew = ! $entry;
                    if (! $entry) {
                        $entry = new MosqueFinanceEntry();
                        $entry->business_id = $businessId;
                        $entry->ref_module = self::OPENING_BALANCE_REF_MODULE;
                        $entry->ref_id = $refId;
                        $entry->type = 'expense';
                        $entry->location_id = null;
                        $entry->created_by = auth()->id();
                    }

                    $entry->category_id = $cat->id;
                    $entry->amount = $openingExpense;
                    $entry->entry_date = $entryDate->format('Y-m-d');
                    $entry->fund_tag = null;
                    $entry->note = $note;
                    $entry->save();

                    MosqueAuditUtil::log($businessId, $isNew ? 'create' : 'update', 'finance_entry', (int) $entry->id, [
                        'ref_module' => self::OPENING_BALANCE_REF_MODULE,
                        'ref_id' => $refId,
                        'type' => $entry->type,
                        'amount' => $entry->amount,
                        'entry_date' => (string) $entry->entry_date,
                    ]);
                }
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function entriesEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $entry = MosqueFinanceEntry::query()
            ->where('business_id', $businessId)
            ->where(function ($q) {
                $q->whereNull('ref_module')
                    ->orWhere('ref_module', 'custom')
                    ->orWhere('ref_module', self::OPENING_BALANCE_REF_MODULE);
            })
            ->findOrFail($id);

        $this->ensureTypePermission($entry->type);

        $categories = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->where('type', $entry->type)
            ->where('active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::finance.entries.edit', compact('entry', 'categories'));
    }

    public function entriesUpdate(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'category_id' => 'required|integer',
            'amount' => 'required|numeric|min:0',
            'entry_date' => 'required|date',
            'note' => 'nullable|string|max:2000',
        ]);

        try {
            $businessId = $this->businessId();

            $entry = MosqueFinanceEntry::query()
                ->where('business_id', $businessId)
                ->where(function ($q) {
                    $q->whereNull('ref_module')
                        ->orWhere('ref_module', 'custom')
                        ->orWhere('ref_module', self::OPENING_BALANCE_REF_MODULE);
                })
                ->findOrFail($id);

            $this->ensureTypePermission($entry->type);

            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->where('id', $request->input('category_id'))
                ->where('type', $entry->type)
                ->firstOrFail();

            $entry->update([
                'category_id' => $category->id,
                'amount' => $request->input('amount'),
                'entry_date' => $request->input('entry_date'),
                'fund_tag' => null,
                'note' => $request->input('note'),
            ]);

            MosqueAuditUtil::log($businessId, 'update', 'finance_entry', (int) $entry->id, [
                'type' => $entry->type,
                'amount' => $entry->amount,
                'entry_date' => (string) $entry->entry_date,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function entriesDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();

            $entry = MosqueFinanceEntry::query()
                ->where('business_id', $businessId)
                ->where(function ($q) {
                    $q->whereNull('ref_module')
                        ->orWhere('ref_module', 'custom')
                        ->orWhere('ref_module', self::OPENING_BALANCE_REF_MODULE);
                })
                ->findOrFail($id);

            $this->ensureTypePermission($entry->type);

            $entry->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'finance_entry', (int) $entry->id, [
                'type' => $entry->type,
                'amount' => $entry->amount,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'finance entry', (int) $entry->id, [
                'type' => (string) $entry->type,
                'amount' => (float) $entry->amount,
                'entry_date' => (string) $entry->entry_date,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function reportsIndex(Request $request)
    {
        $this->ensurePermission();

        $businessId = $this->businessId();

        $categories = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->where('active', true)
            ->orderBy('type')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        return view('mosque::finance.reports', compact('categories'));
    }

    public function dayBookIndex(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $categories = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->where('active', true)
            ->orderBy('type')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $presetSource = null; // include module + core
        $title = 'Day Book';

        return view('mosque::finance.day_book', compact('categories', 'presetSource', 'title'));
    }

    public function cashBookIndex(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $categories = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->where('active', true)
            ->orderBy('type')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $presetSource = 'module'; // module only
        $title = 'Cash Book';

        return view('mosque::finance.day_book', compact('categories', 'presetSource', 'title'));
    }

    public function dayBookData(Request $request)
    {
        $this->ensurePermission();

        return $this->dailyBookData($request, null);
    }

    public function cashBookData(Request $request)
    {
        $this->ensurePermission();

        return $this->dailyBookData($request, 'module');
    }

    private function dailyBookData(Request $request, ?string $forcedSource)
    {
        $businessId = $this->businessId();

        [$startDate, $endDate] = $this->resolveDateRange($request);

        $locationId = $request->input('location_id');
        $source = $forcedSource ?: $request->input('source'); // module|core_sale|core_purchase
        $type = $request->input('type'); // income|expense
        $categoryId = $request->input('category_id');
        $fundTag = $request->input('fund_tag');

        $categoryName = null;
        $categoryType = null;
        if (! empty($categoryId)) {
            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->where('id', $categoryId)
                ->first();
            $categoryName = $category?->name;
            $categoryType = $category?->type;
        }

        $moduleDaily = DB::table('mosque_finance_entries as e')
            ->where('e.business_id', $businessId)
            ->whereNull('e.deleted_at')
            ->select([
                DB::raw('DATE(e.entry_date) as date'),
                DB::raw("SUM(CASE WHEN e.type='income' THEN e.amount ELSE 0 END) as income"),
                DB::raw("SUM(CASE WHEN e.type='expense' THEN e.amount ELSE 0 END) as expense"),
            ])
            ->groupBy(DB::raw('DATE(e.entry_date)'));

        $moduleOpening = DB::table('mosque_finance_entries as e')
            ->where('e.business_id', $businessId)
            ->whereNull('e.deleted_at')
            ->whereDate('e.entry_date', '<', $startDate);

        if (! empty($locationId)) {
            $moduleDaily->where('e.location_id', $locationId);
            $moduleOpening->where('e.location_id', $locationId);
        }
        if (! empty($type)) {
            $moduleDaily->where('e.type', $type);
            $moduleOpening->where('e.type', $type);
        }
        if (! empty($categoryId)) {
            $moduleDaily->where('e.category_id', $categoryId);
            $moduleOpening->where('e.category_id', $categoryId);
        }
        if (! empty($fundTag)) {
            $moduleDaily->where('e.fund_tag', $fundTag);
            $moduleOpening->where('e.fund_tag', $fundTag);
        }
        if (! empty($source) && $source !== 'module') {
            $moduleDaily->whereRaw('1=0');
            $moduleOpening->whereRaw('1=0');
        }

        $moduleDaily->whereDate('e.entry_date', '>=', $startDate)->whereDate('e.entry_date', '<=', $endDate);

        $openingBalance = 0.0;
        $moduleOpenIncome = (float) $moduleOpening->clone()->where('e.type', 'income')->sum('e.amount');
        $moduleOpenExpense = (float) $moduleOpening->clone()->where('e.type', 'expense')->sum('e.amount');
        $openingBalance += ($moduleOpenIncome - $moduleOpenExpense);

        $hasCorePayments = Schema::hasTable('transaction_payments');
        $salesPaidExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";
        $purchPaidExpr = "(SELECT COALESCE(SUM(tp.amount),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";

        $coreSalesDaily = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->select([
                DB::raw('DATE(t.transaction_date) as date'),
                DB::raw($hasCorePayments ? "SUM({$salesPaidExpr}) as income" : 'SUM(t.final_total) as income'),
                DB::raw('0 as expense'),
            ])
            ->groupBy(DB::raw('DATE(t.transaction_date)'));

        $coreSalesOpening = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereDate('t.transaction_date', '<', $startDate);

        if (! empty($locationId)) {
            $coreSalesDaily->where('t.location_id', $locationId);
            $coreSalesOpening->where('t.location_id', $locationId);
        }
        if (! empty($type) && $type !== 'income') {
            $coreSalesDaily->whereRaw('1=0');
            $coreSalesOpening->whereRaw('1=0');
        }
        if (! empty($source) && $source !== 'core_sale') {
            $coreSalesDaily->whereRaw('1=0');
            $coreSalesOpening->whereRaw('1=0');
        }
        if (! empty($categoryId) && ! ($categoryType === 'income' && $categoryName === 'Sales')) {
            $coreSalesDaily->whereRaw('1=0');
            $coreSalesOpening->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $coreSalesDaily->whereRaw('1=0');
            $coreSalesOpening->whereRaw('1=0');
        }

        $coreSalesDaily->whereDate('t.transaction_date', '>=', $startDate)->whereDate('t.transaction_date', '<=', $endDate);
        $openingBalance += (float) ($hasCorePayments ? $coreSalesOpening->sum(DB::raw($salesPaidExpr)) : $coreSalesOpening->sum('t.final_total'));

        $corePurchDaily = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->select([
                DB::raw('DATE(t.transaction_date) as date'),
                DB::raw('0 as income'),
                DB::raw($hasCorePayments ? "SUM({$purchPaidExpr}) as expense" : 'SUM(t.final_total) as expense'),
            ])
            ->groupBy(DB::raw('DATE(t.transaction_date)'));

        $corePurchOpening = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '<', $startDate);

        if (! empty($locationId)) {
            $corePurchDaily->where('t.location_id', $locationId);
            $corePurchOpening->where('t.location_id', $locationId);
        }
        if (! empty($type) && $type !== 'expense') {
            $corePurchDaily->whereRaw('1=0');
            $corePurchOpening->whereRaw('1=0');
        }
        if (! empty($source) && $source !== 'core_purchase') {
            $corePurchDaily->whereRaw('1=0');
            $corePurchOpening->whereRaw('1=0');
        }
        if (! empty($categoryId) && ! ($categoryType === 'expense' && $categoryName === 'Purchases')) {
            $corePurchDaily->whereRaw('1=0');
            $corePurchOpening->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $corePurchDaily->whereRaw('1=0');
            $corePurchOpening->whereRaw('1=0');
        }

        $corePurchDaily->whereDate('t.transaction_date', '>=', $startDate)->whereDate('t.transaction_date', '<=', $endDate);
        $openingBalance -= (float) ($hasCorePayments ? $corePurchOpening->sum(DB::raw($purchPaidExpr)) : $corePurchOpening->sum('t.final_total'));

        $union = $moduleDaily->unionAll($coreSalesDaily)->unionAll($corePurchDaily);

        $rows = DB::query()
            ->fromSub($union, 'u')
            ->select([
                'date',
                DB::raw('SUM(income) as income'),
                DB::raw('SUM(expense) as expense'),
            ])
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $running = (float) $openingBalance;
        $outRows = [];
        foreach ($rows as $row) {
            $incomeVal = (float) ($row->income ?? 0);
            $expenseVal = (float) ($row->expense ?? 0);
            $net = $incomeVal - $expenseVal;
            $running += $net;
            $outRows[] = [
                'date' => $row->date,
                'income' => $incomeVal,
                'expense' => $expenseVal,
                'net' => $net,
                'balance' => $running,
            ];
        }

        return [
            'success' => true,
            'opening_balance' => (float) $openingBalance,
            'rows' => $outRows,
        ];
    }

    public function reportsCategoriesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        [$startDate, $endDate] = $this->resolveDateRange($request);

        $locationId = $request->input('location_id');
        $source = $request->input('source'); // module|core_sale|core_purchase
        $type = $request->input('type'); // income|expense
        $fundTag = $request->input('fund_tag'); // zakat|...

        $module = DB::table('mosque_finance_entries as e')
            ->join('mosque_finance_categories as c', function ($join) use ($businessId) {
                $join->on('c.id', '=', 'e.category_id')
                    ->where('c.business_id', '=', $businessId);
            })
            ->where('e.business_id', $businessId)
            ->whereNull('e.deleted_at')
            ->whereDate('e.entry_date', '>=', $startDate)
            ->whereDate('e.entry_date', '<=', $endDate)
            ->select([
                DB::raw("'module' as source"),
                'e.type as type',
                'c.name as category',
                'e.fund_tag as fund_tag',
                DB::raw('SUM(e.amount) as total'),
            ])
            ->groupBy('e.type', 'c.name', 'e.fund_tag');

        if (! empty($locationId)) {
            $module->where('e.location_id', $locationId);
        }
        if (! empty($type)) {
            $module->where('e.type', $type);
        }
        if (! empty($source) && $source !== 'module') {
            $module->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $module->where('e.fund_tag', $fundTag);
        }

        $hasCorePayments = Schema::hasTable('transaction_payments');
        $salesPaidExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";
        $purchPaidExpr = "(SELECT COALESCE(SUM(tp.amount),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";

        $coreSales = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereDate('t.transaction_date', '>=', $startDate)
            ->whereDate('t.transaction_date', '<=', $endDate)
            ->select([
                DB::raw("'core_sale' as source"),
                DB::raw("'income' as type"),
                DB::raw("'Sales' as category"),
                DB::raw('NULL as fund_tag'),
                DB::raw($hasCorePayments ? "SUM({$salesPaidExpr}) as total" : 'SUM(t.final_total) as total'),
            ])
            ->groupBy(DB::raw("'core_sale'"), DB::raw("'income'"), DB::raw("'Sales'"), DB::raw('NULL'));

        if (! empty($locationId)) {
            $coreSales->where('t.location_id', $locationId);
        }
        if (! empty($type) && $type !== 'income') {
            $coreSales->whereRaw('1=0');
        }
        if (! empty($source) && $source !== 'core_sale') {
            $coreSales->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $coreSales->whereRaw('1=0');
        }

        $corePurchases = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '>=', $startDate)
            ->whereDate('t.transaction_date', '<=', $endDate)
            ->select([
                DB::raw("'core_purchase' as source"),
                DB::raw("'expense' as type"),
                DB::raw("'Purchases' as category"),
                DB::raw('NULL as fund_tag'),
                DB::raw($hasCorePayments ? "SUM({$purchPaidExpr}) as total" : 'SUM(t.final_total) as total'),
            ])
            ->groupBy(DB::raw("'core_purchase'"), DB::raw("'expense'"), DB::raw("'Purchases'"), DB::raw('NULL'));

        if (! empty($locationId)) {
            $corePurchases->where('t.location_id', $locationId);
        }
        if (! empty($type) && $type !== 'expense') {
            $corePurchases->whereRaw('1=0');
        }
        if (! empty($source) && $source !== 'core_purchase') {
            $corePurchases->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $corePurchases->whereRaw('1=0');
        }

        $union = $module->unionAll($coreSales)->unionAll($corePurchases);
        $query = DB::query()->fromSub($union, 'u');

        return DataTables::of($query)
            ->editColumn('total', function ($row) {
                $v = (float) ($row->total ?? 0);
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$v.'">'.$v.'</span>';
            })
            ->rawColumns(['total'])
            ->make(true);
    }

    public function reportsMonthlyData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        [$startDate, $endDate] = $this->resolveDateRange($request);

        $locationId = $request->input('location_id');
        $source = $request->input('source'); // module|core_sale|core_purchase
        $fundTag = $request->input('fund_tag'); // zakat|...

        $module = MosqueFinanceEntry::query()
            ->where('business_id', $businessId)
            ->whereDate('entry_date', '>=', $startDate)
            ->whereDate('entry_date', '<=', $endDate)
            ->select([
                DB::raw("DATE_FORMAT(entry_date, '%Y-%m') as ym"),
                'type',
                DB::raw('SUM(amount) as total'),
            ])
            ->groupBy(DB::raw("DATE_FORMAT(entry_date, '%Y-%m')"), 'type');

        if (! empty($locationId)) {
            $module->where('location_id', $locationId);
        }
        if (! empty($source) && $source !== 'module') {
            $module->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $module->where('fund_tag', $fundTag);
        }

        $hasCorePayments = Schema::hasTable('transaction_payments');
        $salesPaidExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";
        $purchPaidExpr = "(SELECT COALESCE(SUM(tp.amount),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";

        $coreSales = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereDate('t.transaction_date', '>=', $startDate)
            ->whereDate('t.transaction_date', '<=', $endDate)
            ->select([
                DB::raw("DATE_FORMAT(t.transaction_date, '%Y-%m') as ym"),
                DB::raw("'income' as type"),
                DB::raw($hasCorePayments ? "SUM({$salesPaidExpr}) as total" : 'SUM(t.final_total) as total'),
            ])
            ->groupBy(DB::raw("DATE_FORMAT(t.transaction_date, '%Y-%m')"));

        if (! empty($locationId)) {
            $coreSales->where('t.location_id', $locationId);
        }
        if (! empty($source) && $source !== 'core_sale') {
            $coreSales->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $coreSales->whereRaw('1=0');
        }

        $corePurchases = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '>=', $startDate)
            ->whereDate('t.transaction_date', '<=', $endDate)
            ->select([
                DB::raw("DATE_FORMAT(t.transaction_date, '%Y-%m') as ym"),
                DB::raw("'expense' as type"),
                DB::raw($hasCorePayments ? "SUM({$purchPaidExpr}) as total" : 'SUM(t.final_total) as total'),
            ])
            ->groupBy(DB::raw("DATE_FORMAT(t.transaction_date, '%Y-%m')"));

        if (! empty($locationId)) {
            $corePurchases->where('t.location_id', $locationId);
        }
        if (! empty($source) && $source !== 'core_purchase') {
            $corePurchases->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $corePurchases->whereRaw('1=0');
        }

        $union = $module->unionAll($coreSales)->unionAll($corePurchases);
        $rows = DB::query()
            ->fromSub($union, 'u')
            ->select([
                'ym',
                DB::raw("SUM(CASE WHEN type='income' THEN total ELSE 0 END) as income"),
                DB::raw("SUM(CASE WHEN type='expense' THEN total ELSE 0 END) as expense"),
            ])
            ->groupBy('ym')
            ->orderBy('ym')
            ->get();

        return ['success' => true, 'rows' => $rows];
    }

    public function reportsWeeklyData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        [$startDate, $endDate] = $this->resolveDateRange($request);

        $locationId = $request->input('location_id');
        $source = $request->input('source'); // module|core_sale|core_purchase
        $type = $request->input('type'); // income|expense
        $fundTag = $request->input('fund_tag'); // zakat|...

        $module = MosqueFinanceEntry::query()
            ->where('business_id', $businessId)
            ->whereDate('entry_date', '>=', $startDate)
            ->whereDate('entry_date', '<=', $endDate)
            ->select([
                DB::raw('YEARWEEK(entry_date, 1) as yw'),
                'type',
                DB::raw('SUM(amount) as total'),
            ])
            ->groupBy(DB::raw('YEARWEEK(entry_date, 1)'), 'type');

        if (! empty($locationId)) {
            $module->where('location_id', $locationId);
        }
        if (! empty($source) && $source !== 'module') {
            $module->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $module->where('fund_tag', $fundTag);
        }

        $hasCorePayments = Schema::hasTable('transaction_payments');
        $salesPaidExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";
        $purchPaidExpr = "(SELECT COALESCE(SUM(tp.amount),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";

        $coreSales = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereDate('t.transaction_date', '>=', $startDate)
            ->whereDate('t.transaction_date', '<=', $endDate)
            ->select([
                DB::raw('YEARWEEK(t.transaction_date, 1) as yw'),
                DB::raw("'income' as type"),
                DB::raw($hasCorePayments ? "SUM({$salesPaidExpr}) as total" : 'SUM(t.final_total) as total'),
            ])
            ->groupBy(DB::raw('YEARWEEK(t.transaction_date, 1)'));

        if (! empty($locationId)) {
            $coreSales->where('t.location_id', $locationId);
        }
        if (! empty($source) && $source !== 'core_sale') {
            $coreSales->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $coreSales->whereRaw('1=0');
        }

        $corePurchases = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '>=', $startDate)
            ->whereDate('t.transaction_date', '<=', $endDate)
            ->select([
                DB::raw('YEARWEEK(t.transaction_date, 1) as yw'),
                DB::raw("'expense' as type"),
                DB::raw($hasCorePayments ? "SUM({$purchPaidExpr}) as total" : 'SUM(t.final_total) as total'),
            ])
            ->groupBy(DB::raw('YEARWEEK(t.transaction_date, 1)'));

        if (! empty($locationId)) {
            $corePurchases->where('t.location_id', $locationId);
        }
        if (! empty($source) && $source !== 'core_purchase') {
            $corePurchases->whereRaw('1=0');
        }
        if (! empty($fundTag)) {
            $corePurchases->whereRaw('1=0');
        }

        $union = $module->unionAll($coreSales)->unionAll($corePurchases);
        $weekStartExpr = "STR_TO_DATE(CONCAT(yw, ' Monday'), '%X%V %W')";
        $rows = DB::query()
            ->fromSub($union, 'u')
            ->select([
                'yw',
                DB::raw($weekStartExpr.' as week_start'),
                DB::raw("DATE_ADD({$weekStartExpr}, INTERVAL 6 DAY) as week_end"),
                DB::raw("SUM(CASE WHEN type='income' THEN total ELSE 0 END) as income"),
                DB::raw("SUM(CASE WHEN type='expense' THEN total ELSE 0 END) as expense"),
            ])
            ->groupBy('yw')
            ->orderBy('week_start')
            ->get();

        if (! empty($type)) {
            foreach ($rows as $row) {
                if ($type === 'income') {
                    $row->expense = 0;
                } elseif ($type === 'expense') {
                    $row->income = 0;
                }
            }
        }

        return ['success' => true, 'rows' => $rows];
    }

    public function reportsZakatSummary(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        [$startDate, $endDate] = $this->resolveDateRange($request);

        $locationId = $request->input('location_id');

        $query = MosqueFinanceEntry::query()
            ->where('business_id', $businessId)
            ->where('fund_tag', 'zakat')
            ->whereDate('entry_date', '>=', $startDate)
            ->whereDate('entry_date', '<=', $endDate);

        if (! empty($locationId)) {
            $query->where('location_id', $locationId);
        }

        $byType = $query
            ->select('type', DB::raw('SUM(amount) as total'))
            ->groupBy('type')
            ->pluck('total', 'type')
            ->toArray();

        $income = (float) ($byType['income'] ?? 0);
        $expense = (float) ($byType['expense'] ?? 0);

        return [
            'success' => true,
            'income' => $income,
            'expense' => $expense,
            'balance' => $income - $expense,
        ];
    }

    public function summary(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        [$startDate, $endDate] = $this->resolveDateRange($request);

        $locationId = $request->input('location_id');
        $source = $request->input('source'); // module|core_sale|core_purchase
        $type = $request->input('type'); // income|expense
        $categoryId = $request->input('category_id');

        $totals = [
            'income' => 0,
            'expense' => 0,
        ];

        $moduleQuery = MosqueFinanceEntry::query()
            ->where('business_id', $businessId)
            ->whereDate('entry_date', '>=', $startDate)
            ->whereDate('entry_date', '<=', $endDate);

        if (! empty($locationId)) {
            $moduleQuery->where('location_id', $locationId);
        }
        if (! empty($type)) {
            $moduleQuery->where('type', $type);
        }
        if (! empty($categoryId)) {
            $moduleQuery->where('category_id', $categoryId);
        }
        if (! empty($source) && $source !== 'module') {
            $moduleQuery->whereRaw('1=0');
        }

        $moduleSums = $moduleQuery
            ->select('type', DB::raw('SUM(amount) as total'))
            ->groupBy('type')
            ->pluck('total', 'type')
            ->toArray();

        $totals['income'] += (float) ($moduleSums['income'] ?? 0);
        $totals['expense'] += (float) ($moduleSums['expense'] ?? 0);

        $categoryName = null;
        $categoryType = null;
        if (! empty($categoryId)) {
            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->where('id', $categoryId)
                ->first();
            $categoryName = $category?->name;
            $categoryType = $category?->type;
        }

        if (empty($source) || $source === 'core_sale') {
            if ((empty($type) || $type === 'income') && (empty($categoryId) || ($categoryType === 'income' && $categoryName === 'Sales'))) {
                $salesQuery = DB::table('transaction_payments as tp')
                    ->join('transactions as t', 't.id', '=', 'tp.transaction_id')
                    ->where('t.business_id', $businessId)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('tp.paid_on', '>=', $startDate)
                    ->whereDate('tp.paid_on', '<=', $endDate);

                if (! empty($locationId)) {
                    $salesQuery->where('t.location_id', $locationId);
                }

                // Wallet uses received/paid amount only (not invoice total).
                if (Schema::hasTable('transaction_payments')) {
                    $totals['income'] += (float) $salesQuery->sum(DB::raw('IF(tp.is_return = 1, -1*tp.amount, tp.amount)'));
                }
            }
        }

        if (empty($source) || $source === 'core_purchase') {
            if ((empty($type) || $type === 'expense') && (empty($categoryId) || ($categoryType === 'expense' && $categoryName === 'Purchases'))) {
                $purchasesQuery = DB::table('transaction_payments as tp')
                    ->join('transactions as t', 't.id', '=', 'tp.transaction_id')
                    ->where('t.business_id', $businessId)
                    ->where('t.type', 'purchase')
                    ->where('t.status', 'received')
                    ->whereDate('tp.paid_on', '>=', $startDate)
                    ->whereDate('tp.paid_on', '<=', $endDate);

                if (! empty($locationId)) {
                    $purchasesQuery->where('t.location_id', $locationId);
                }

                // Wallet uses paid amount only (not invoice total).
                if (Schema::hasTable('transaction_payments')) {
                    $totals['expense'] += (float) $purchasesQuery->sum(DB::raw('IF(tp.is_return = 1, -1*tp.amount, tp.amount)'));
                }
            }
        }

        return [
            'success' => true,
            'totals' => $totals,
            'net' => $totals['income'] - $totals['expense'],
        ];
    }

    public function entriesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        [$startDate, $endDate] = $this->resolveDateRange($request);

        $locationId = $request->input('location_id');
        $source = $request->input('source'); // module|core_sale|core_purchase
        $type = $request->input('type'); // income|expense
        $categoryId = $request->input('category_id');

        $categoryName = null;
        $categoryType = null;
        if (! empty($categoryId)) {
            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->where('id', $categoryId)
                ->first();
            $categoryName = $category?->name;
            $categoryType = $category?->type;
        }

        $moduleEntries = DB::table('mosque_finance_entries as mfe')
            ->leftJoin('mosque_finance_categories as mfc', function ($join) {
                $join->on('mfc.id', '=', 'mfe.category_id');
            })
            ->where('mfe.business_id', $businessId)
            ->whereNull('mfe.deleted_at')
            ->whereDate('mfe.entry_date', '>=', $startDate)
            ->whereDate('mfe.entry_date', '<=', $endDate)
            ->select([
                DB::raw("CONCAT('module-', mfe.id) as row_id"),
                'mfe.id as source_id',
                'mfe.entry_date as date',
                'mfe.type as type',
                DB::raw('mfe.amount as total_amount'),
                DB::raw('mfe.amount as paid_amount'),
                DB::raw('0 as due_amount'),
                'mfc.name as category',
                 'mfe.fund_tag as fund_tag',
                 'mfe.note as note',
                 'mfe.location_id as location_id',
                 DB::raw("'module' as source"),
                 DB::raw('NULL as ref_no'),
                'mfe.ref_module as ref_module',
                'mfe.ref_id as ref_id',
                DB::raw("CASE WHEN mfe.ref_module IS NULL OR mfe.ref_module = 'custom' OR mfe.ref_module = '".self::OPENING_BALANCE_REF_MODULE."' THEN 1 ELSE 0 END as editable"),
             ]);

        if (! empty($locationId)) {
            $moduleEntries->where('mfe.location_id', $locationId);
        }
        if (! empty($type)) {
            $moduleEntries->where('mfe.type', $type);
        }
        if (! empty($categoryId)) {
            $moduleEntries->where('mfe.category_id', $categoryId);
        }
        if (! empty($source) && $source !== 'module') {
            $moduleEntries->whereRaw('1=0');
        }

        $salesPaidExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments as tp WHERE tp.transaction_id = t.id)";
        $salesPaidInRangeExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments as tp WHERE tp.transaction_id = t.id AND DATE(tp.paid_on) >= '{$startDate}' AND DATE(tp.paid_on) <= '{$endDate}')";
        $salesPaidInRangeDateExpr = "(SELECT DATE(MAX(tp.paid_on)) FROM transaction_payments as tp WHERE tp.transaction_id = t.id AND DATE(tp.paid_on) >= '{$startDate}' AND DATE(tp.paid_on) <= '{$endDate}')";
        $coreSales = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereExists(function ($q) use ($startDate, $endDate) {
                $q->select(DB::raw(1))
                    ->from('transaction_payments as tp')
                    ->whereColumn('tp.transaction_id', 't.id')
                    ->whereDate('tp.paid_on', '>=', $startDate)
                    ->whereDate('tp.paid_on', '<=', $endDate);
            })
            ->select([
                DB::raw("CONCAT('core_sale-', t.id) as row_id"),
                't.id as source_id',
                DB::raw($salesPaidInRangeDateExpr.' as date'),
                DB::raw("'income' as type"),
                DB::raw('t.final_total as total_amount'),
                DB::raw($salesPaidInRangeExpr.' as paid_amount'),
                DB::raw("GREATEST(t.final_total - ({$salesPaidExpr}), 0) as due_amount"),
                DB::raw("'Sales' as category"),
                DB::raw('NULL as fund_tag'),
                DB::raw("CONCAT('Invoice: ', COALESCE(t.invoice_no, t.ref_no), ' | Paid (range): ', {$salesPaidInRangeExpr}, ' | Total Paid: ', {$salesPaidExpr}, ' | Due: ', GREATEST(t.final_total - ({$salesPaidExpr}), 0)) as note"),
                 't.location_id as location_id',
                 DB::raw("'core_sale' as source"),
                 DB::raw("COALESCE(t.invoice_no, t.ref_no) as ref_no"),
                 DB::raw('NULL as ref_module'),
                 DB::raw('NULL as ref_id'),
                 DB::raw('0 as editable'),
             ]);

        if (! empty($locationId)) {
            $coreSales->where('t.location_id', $locationId);
        }
        if (! empty($type) && $type !== 'income') {
            $coreSales->whereRaw('1=0');
        }
        if (! empty($source) && $source !== 'core_sale') {
            $coreSales->whereRaw('1=0');
        }
        if (! empty($categoryId) && ! ($categoryType === 'income' && $categoryName === 'Sales')) {
            $coreSales->whereRaw('1=0');
        }

        $purchPaidExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments as tp WHERE tp.transaction_id = t.id)";
        $purchPaidInRangeExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments as tp WHERE tp.transaction_id = t.id AND DATE(tp.paid_on) >= '{$startDate}' AND DATE(tp.paid_on) <= '{$endDate}')";
        $purchPaidInRangeDateExpr = "(SELECT DATE(MAX(tp.paid_on)) FROM transaction_payments as tp WHERE tp.transaction_id = t.id AND DATE(tp.paid_on) >= '{$startDate}' AND DATE(tp.paid_on) <= '{$endDate}')";
        $corePurchases = DB::table('transactions as t')
            ->where('t.business_id', $businessId)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereExists(function ($q) use ($startDate, $endDate) {
                $q->select(DB::raw(1))
                    ->from('transaction_payments as tp')
                    ->whereColumn('tp.transaction_id', 't.id')
                    ->whereDate('tp.paid_on', '>=', $startDate)
                    ->whereDate('tp.paid_on', '<=', $endDate);
            })
            ->select([
                DB::raw("CONCAT('core_purchase-', t.id) as row_id"),
                't.id as source_id',
                DB::raw($purchPaidInRangeDateExpr.' as date'),
                DB::raw("'expense' as type"),
                DB::raw('t.final_total as total_amount'),
                DB::raw($purchPaidInRangeExpr.' as paid_amount'),
                DB::raw("GREATEST(t.final_total - ({$purchPaidExpr}), 0) as due_amount"),
                DB::raw("'Purchases' as category"),
                DB::raw('NULL as fund_tag'),
                DB::raw("CONCAT('Ref: ', COALESCE(t.ref_no, t.invoice_no), ' | Paid (range): ', {$purchPaidInRangeExpr}, ' | Total Paid: ', {$purchPaidExpr}, ' | Due: ', GREATEST(t.final_total - ({$purchPaidExpr}), 0)) as note"),
                 't.location_id as location_id',
                 DB::raw("'core_purchase' as source"),
                 DB::raw("COALESCE(t.ref_no, t.invoice_no) as ref_no"),
                 DB::raw('NULL as ref_module'),
                 DB::raw('NULL as ref_id'),
                 DB::raw('0 as editable'),
             ]);

        if (! empty($locationId)) {
            $corePurchases->where('t.location_id', $locationId);
        }
        if (! empty($type) && $type !== 'expense') {
            $corePurchases->whereRaw('1=0');
        }
        if (! empty($source) && $source !== 'core_purchase') {
            $corePurchases->whereRaw('1=0');
        }
        if (! empty($categoryId) && ! ($categoryType === 'expense' && $categoryName === 'Purchases')) {
            $corePurchases->whereRaw('1=0');
        }

        $union = $moduleEntries
            ->unionAll($coreSales)
            ->unionAll($corePurchases);

        $query = DB::query()->fromSub($union, 'u');

        return DataTables::of($query)
            ->addColumn('source_label', function ($row) {
                return match ($row->source) {
                    'module' => 'Module',
                    'core_sale' => 'Core Sale',
                    'core_purchase' => 'Core Purchase',
                    default => $row->source,
                };
            })
            ->addColumn('action', function ($row) {
                if ($row->source === 'core_sale') {
                    $sellUrl = url('sells');
                    $title = "Core sales can't be deleted from Mosque Finance. Delete from core Sales list.";
                    return '<a href="'.$sellUrl.'" class="btn btn-xs btn-default" data-toggle="tooltip" title="'.$title.'"><i class="fa fa-info-circle"></i></a>';
                }
                if ($row->source === 'core_purchase') {
                    $purchaseUrl = url('purchases');
                    $title = "Core purchases can't be deleted from Mosque Finance. Delete from core Purchases list.";
                    return '<a href="'.$purchaseUrl.'" class="btn btn-xs btn-default" data-toggle="tooltip" title="'.$title.'"><i class="fa fa-info-circle"></i></a>';
                }
                if ($row->source !== 'module') {
                    return '';
                }

                $can = ($row->type === 'income' && auth()->user()->hasAnyPermission(['mosque.finance.income', 'mosque.finance.reports']))
                    || ($row->type === 'expense' && auth()->user()->hasAnyPermission(['mosque.finance.expense', 'mosque.finance.reports']));

                if (! $can) {
                    return '';
                }

                // For non-custom entries, delete must happen in the owning module to keep data consistent.
                if (empty($row->editable)) {
                    $refModule = (string) ($row->ref_module ?? '');
                    $refId = (int) ($row->ref_id ?? 0);

                    $url = null;
                    $title = 'This entry was generated by a module. Delete it from the module screen so Finance/Dashboard/Reports stay consistent.';

                    if ($refModule === 'donation') {
                        $url = route('mosque.donations', ['highlight_id' => $refId]);
                        $title = 'Delete this from Donations (it will be removed from Finance automatically).';
                    } elseif ($refModule === 'event') {
                        $url = route('mosque.events', ['highlight_id' => $refId]);
                        $title = 'Delete this from Event entries (it will be removed from Finance automatically).';
                    } elseif ($refModule === 'rental') {
                        $url = route('mosque.assets', ['highlight_id' => $refId]);
                        $title = 'Delete this from Rentals (it will be removed from Finance automatically).';
                    } elseif ($refModule === 'asset_maintenance') {
                        $url = route('mosque.assets', ['highlight_id' => $refId]);
                        $title = 'Delete this from Assets → Maintenance (it will be removed from Finance automatically).';
                    } elseif ($refModule === 'pledge') {
                        $url = route('mosque.pledges', ['highlight_id' => $refId]);
                        $title = 'Delete this from Pledges (it will be removed from Finance automatically).';
                    } elseif ($refModule === 'membership') {
                        $url = route('mosque.subscriptions.fees', ['highlight_id' => $refId]);
                        $title = 'Cancel/Delete this from Membership Fees/Payments (it will be removed from Finance automatically).';
                    } elseif ($refModule === 'payslip') {
                        $url = route('mosque.staff', ['highlight_id' => $refId]);
                        $title = 'Delete this from Staff → Payslips (it will be removed from Finance automatically).';
                    }

                    if ($url) {
                        return '<a href="'.$url.'" class="btn btn-xs btn-info" data-toggle="tooltip" title="'.$title.'"><i class="fa fa-external-link-alt"></i> Open</a>';
                    }

                    return '<button type="button" class="btn btn-xs btn-default" data-toggle="tooltip" title="'.$title.'"><i class="fa fa-info-circle"></i></button>';
                }

                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FinanceController::class, 'entriesEdit'], [$row->source_id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_finance_entry_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FinanceController::class, 'entriesDestroy'], [$row->source_id]).'" class="btn btn-xs btn-danger delete_mosque_finance_entry"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->editColumn('paid_amount', function ($row) {
                $v = (float) ($row->paid_amount ?? 0);
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$v.'">'.$v.'</span>';
            })
            ->editColumn('due_amount', function ($row) {
                $v = (float) ($row->due_amount ?? 0);
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$v.'">'.$v.'</span>';
            })
            ->rawColumns(['paid_amount', 'due_amount', 'action'])
            ->make(true);
    }

    public function categoriesIndex()
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }

        return view('mosque::finance.categories.index');
    }

    public function categoriesData(Request $request)
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = $this->businessId();

        $query = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->select(['id', 'type', 'name', 'active', 'sort_order']);

        if (! empty($request->input('type'))) {
            $query->where('type', $request->input('type'));
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                return '<button data-href="'.action([\Modules\Mosque\Http\Controllers\FinanceController::class, 'categoriesEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_category_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>
                        &nbsp;
                        <button data-href="'.action([\Modules\Mosque\Http\Controllers\FinanceController::class, 'categoriesDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_category"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
            })
            ->editColumn('active', function ($row) {
                return $row->active ? __('lang_v1.yes') : __('lang_v1.no');
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function categoriesCreate()
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }

        return view('mosque::finance.categories.create');
    }

    public function categoriesStore(Request $request)
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'type' => 'required|in:income,expense',
            'name' => 'required|string|max:255',
            'active' => 'nullable|boolean',
            'sort_order' => 'nullable|integer',
        ]);

        try {
            $businessId = $this->businessId();

            MosqueFinanceCategory::query()->create([
                'business_id' => $businessId,
                'type' => $request->input('type'),
                'name' => $request->input('name'),
                'active' => (bool) $request->input('active', true),
                'sort_order' => (int) $request->input('sort_order', 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function categoriesEdit($id)
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = $this->businessId();
        $category = MosqueFinanceCategory::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        return view('mosque::finance.categories.edit', compact('category'));
    }

    public function categoriesUpdate(Request $request, $id)
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'type' => 'required|in:income,expense',
            'name' => 'required|string|max:255',
            'active' => 'nullable|boolean',
            'sort_order' => 'nullable|integer',
        ]);

        try {
            $businessId = $this->businessId();
            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $category->update([
                'type' => $request->input('type'),
                'name' => $request->input('name'),
                'active' => (bool) $request->input('active', true),
                'sort_order' => (int) $request->input('sort_order', 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function categoriesDestroy($id)
    {
        if (! auth()->user()->can('mosque.manage')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = $this->businessId();
            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $category->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'finance_category', (int) $category->id, [
                'type' => (string) $category->type,
                'name' => (string) $category->name,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'finance category', (int) $category->id, [
                'type' => (string) $category->type,
                'name' => (string) $category->name,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    private function resolveDateRange(Request $request): array
    {
        $start = $request->input('start_date');
        $end = $request->input('end_date');

        if (empty($start) || empty($end)) {
            $start = now()->startOfMonth()->format('Y-m-d');
            $end = now()->endOfMonth()->format('Y-m-d');
        }

        try {
            $startDate = \Carbon\Carbon::parse($start)->format('Y-m-d');
        } catch (\Exception $e) {
            $startDate = now()->startOfMonth()->format('Y-m-d');
        }
        try {
            $endDate = \Carbon\Carbon::parse($end)->format('Y-m-d');
        } catch (\Exception $e) {
            $endDate = now()->endOfMonth()->format('Y-m-d');
        }

        if ($endDate < $startDate) {
            [$startDate, $endDate] = [$endDate, $startDate];
        }

        return [$startDate, $endDate];
    }
}
