<?php

namespace Modules\Mosque\Http\Controllers;

use App\System;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;

class InstallController extends Controller
{
    public function __construct()
    {
        $this->module_name = 'mosque';
        $this->appVersion = config('mosque.module_version');
        $this->module_display_name = 'Mosque';
    }

    public function index()
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        $is_installed = System::getProperty($this->module_name.'_version');
        if (! empty($is_installed)) {
            abort(404);
        }

        return view('mosque::install.index', [
            'action_url' => action([\Modules\Mosque\Http\Controllers\InstallController::class, 'install']),
            'module_display_name' => $this->module_display_name,
        ]);
    }

    public function install()
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $is_installed = System::getProperty($this->module_name.'_version');
            if (! empty($is_installed)) {
                abort(404);
            }

            DB::statement('SET default_storage_engine=INNODB;');
            Artisan::call('module:migrate', ['module' => 'Mosque', '--force' => true]);

            try {
                Artisan::call('module:seed', ['module' => 'Mosque', '--force' => true]);
            } catch (\Exception $e) {
                // Fallback if module:seed is not available.
                Artisan::call('db:seed', ['--class' => \Modules\Mosque\Database\Seeders\MosqueDatabaseSeeder::class, '--force' => true]);
            }

            System::addProperty($this->module_name.'_version', $this->appVersion);
            DB::commit();

            $output = ['success' => 1, 'msg' => 'Mosque module installed successfully'];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            $output = ['success' => 0, 'msg' => $e->getMessage()];
        }

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    public function update()
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            DB::statement('SET default_storage_engine=INNODB;');
            Artisan::call('module:migrate', ['module' => 'Mosque', '--force' => true]);
            try {
                Artisan::call('module:seed', ['module' => 'Mosque', '--force' => true]);
            } catch (\Exception $e) {
                Artisan::call('db:seed', ['--class' => \Modules\Mosque\Database\Seeders\MosqueDatabaseSeeder::class, '--force' => true]);
            }

            System::setProperty($this->module_name.'_version', $this->appVersion);
            DB::commit();

            $output = ['success' => 1, 'msg' => 'Mosque module updated successfully to version '.$this->appVersion];
        } catch (\Exception $e) {
            DB::rollBack();
            $output = ['success' => 0, 'msg' => $e->getMessage()];
        }

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    public function uninstall()
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            System::removeProperty($this->module_name.'_version');
            $output = ['success' => 1, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            $output = ['success' => 0, 'msg' => $e->getMessage()];
        }

        return redirect()->back()->with(['status' => $output]);
    }
}
