<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Barryvdh\DomPDF\Facade\Pdf;
use Modules\Mosque\Entities\MosqueMarriage;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class MarriageController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.marriage.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        return view('mosque::marriage.index');
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueMarriage::query()
            ->where('business_id', $businessId)
            ->select(['id', 'certificate_no', 'date', 'bride', 'groom', 'wali', 'mahr']);

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('date', '>=', $request->input('start_date'))
                ->whereDate('date', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\MarriageController::class, 'print'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $pdf = '<a href="'.action([\Modules\Mosque\Http\Controllers\MarriageController::class, 'pdf'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MarriageController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_marriage_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MarriageController::class, 'destroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_marriage"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $print.' '.$pdf.' '.$edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();
        return view('mosque::marriage.create');
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'date' => 'required|date',
            'certificate_no' => 'nullable|string|max:100',
            'bride' => 'required|string|max:255',
            'groom' => 'required|string|max:255',
            'wali' => 'nullable|string|max:255',
            'mahr' => 'nullable|string|max:255',
            'witness_1' => 'nullable|string|max:255',
            'witness_2' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $certificateNo = $request->input('certificate_no');
            if (empty($certificateNo)) {
                $certificateNo = 'NIKAH-'.now()->format('YmdHis').'-'.$businessId;
            }

            MosqueMarriage::query()->create([
                'business_id' => $businessId,
                'date' => $request->input('date'),
                'certificate_no' => $certificateNo,
                'bride' => $request->input('bride'),
                'groom' => $request->input('groom'),
                'wali' => $request->input('wali'),
                'mahr' => $request->input('mahr'),
                'witnesses' => array_values(array_filter([
                    $request->input('witness_1'),
                    $request->input('witness_2'),
                ])),
                'notes' => $request->input('notes'),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $marriage = MosqueMarriage::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::marriage.edit', compact('marriage'));
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'date' => 'required|date',
            'certificate_no' => 'nullable|string|max:100',
            'bride' => 'required|string|max:255',
            'groom' => 'required|string|max:255',
            'wali' => 'nullable|string|max:255',
            'mahr' => 'nullable|string|max:255',
            'witness_1' => 'nullable|string|max:255',
            'witness_2' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $marriage = MosqueMarriage::query()->where('business_id', $businessId)->findOrFail($id);

            $certificateNo = $request->input('certificate_no');
            if (empty($certificateNo)) {
                $certificateNo = $marriage->certificate_no;
            }

            $marriage->update([
                'date' => $request->input('date'),
                'certificate_no' => $certificateNo,
                'bride' => $request->input('bride'),
                'groom' => $request->input('groom'),
                'wali' => $request->input('wali'),
                'mahr' => $request->input('mahr'),
                'witnesses' => array_values(array_filter([
                    $request->input('witness_1'),
                    $request->input('witness_2'),
                ])),
                'notes' => $request->input('notes'),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $marriage = MosqueMarriage::query()->where('business_id', $businessId)->findOrFail($id);
            $marriage->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'marriage', (int) $marriage->id, [
                'certificate_no' => (string) ($marriage->certificate_no ?? ''),
                'date' => (string) ($marriage->date ?? ''),
                'bride' => (string) ($marriage->bride ?? ''),
                'groom' => (string) ($marriage->groom ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'marriage', (int) $marriage->id, [
                'certificate_no' => (string) ($marriage->certificate_no ?? ''),
                'bride' => (string) ($marriage->bride ?? ''),
                'groom' => (string) ($marriage->groom ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function print($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $marriage = MosqueMarriage::query()->where('business_id', $businessId)->findOrFail($id);

        MosqueAuditUtil::log($businessId, 'print', 'marriage_certificate', (int) $marriage->id, [
            'certificate_no' => $marriage->certificate_no,
        ]);

        return view('mosque::marriage.print', compact('profile', 'logoDataUri', 'marriage'));
    }

    public function pdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $marriage = MosqueMarriage::query()->where('business_id', $businessId)->findOrFail($id);

        MosqueAuditUtil::log($businessId, 'pdf', 'marriage_certificate', (int) $marriage->id, [
            'certificate_no' => $marriage->certificate_no,
        ]);

        $pdf = Pdf::loadView('mosque::marriage.print', compact('profile', 'logoDataUri', 'marriage'))
            ->setPaper('a4');

        $filename = 'marriage_certificate_'.$marriage->certificate_no.'.pdf';

        return $pdf->download($filename);
    }
}
