<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Mosque\Services\MemberFeeService;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Symfony\Component\HttpKernel\Exception\HttpException;

class MemberFeeController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId) && auth()->check()) {
            $businessId = (int) (auth()->user()->business_id ?? 0);
        }
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    public function cancel(Request $request, $feeId, MemberFeeService $service)
    {
        if (! auth()->user()->can('mosque.membership.cancel')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'reason' => 'required|string|min:3|max:2000',
        ]);

        $businessId = $this->businessId();
        $userId = (int) (auth()->id() ?? 0);

        try {
            $fee = $service->cancelFee($businessId, (int) $feeId, $userId, (string) $request->input('reason'));
        } catch (HttpException $e) {
            return response()->json(['ok' => false, 'message' => $e->getMessage()], $e->getStatusCode());
        }

        MosqueAuditUtil::log($businessId, 'cancel', 'member_fee', (int) $fee->id, [
            'cancel_reason' => (string) ($fee->cancel_reason ?? ''),
        ]);

        return response()->json([
            'ok' => true,
            'toast' => __('lang_v1.success'),
            'success' => true,
            'msg' => __('lang_v1.success'),
        ]);
    }

    public function reactivate(Request $request, $feeId, MemberFeeService $service)
    {
        if (! auth()->user()->can('mosque.membership.reactivate')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = $this->businessId();
        $userId = (int) (auth()->id() ?? 0);

        try {
            $fee = $service->reactivateFee($businessId, (int) $feeId, $userId);
        } catch (HttpException $e) {
            return response()->json(['ok' => false, 'message' => $e->getMessage()], $e->getStatusCode());
        }

        MosqueAuditUtil::log($businessId, 'reactivate', 'member_fee', (int) $fee->id, [
            'lifecycle_status' => (string) ($fee->lifecycle_status ?? ''),
        ]);

        return response()->json([
            'ok' => true,
            'toast' => __('lang_v1.success'),
            'success' => true,
            'msg' => __('lang_v1.success'),
        ]);
    }

    public function destroy(Request $request, $feeId, MemberFeeService $service)
    {
        if (! auth()->user()->can('mosque.membership.delete')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = $this->businessId();
        try {
            $service->deleteCancelledFee($businessId, (int) $feeId);
        } catch (HttpException $e) {
            return response()->json(['ok' => false, 'message' => $e->getMessage()], $e->getStatusCode());
        }

        MosqueAuditUtil::log($businessId, 'delete', 'member_fee', (int) $feeId, [
            'deleted_after_cancel' => true,
        ]);

        return response()->json([
            'ok' => true,
            'toast' => __('lang_v1.success'),
            'success' => true,
            'msg' => __('lang_v1.success'),
        ]);
    }
}
