<?php

namespace Modules\Mosque\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueMember;
use Modules\Mosque\Entities\MosqueMemberPlanAssignment;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class MembersController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId) && auth()->check()) {
            $businessId = (int) auth()->user()->business_id;
        }
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.members.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        $businessId = $this->businessId();

        $planOptions = [];
        if (Schema::hasTable('mosque_membership_plans')) {
            $planOptions = DB::table('mosque_membership_plans')
                ->where('business_id', $businessId)
                ->where('active', 1)
                ->orderBy('name')
                ->get(['id', 'name', 'type'])
                ->mapWithKeys(function ($p) {
                    $label = (string) $p->name;
                    if (! empty($p->type)) {
                        $label .= ' ('.ucfirst((string) $p->type).')';
                    }
                    return [(int) $p->id => $label];
                })
                ->all();
        }

        return view('mosque::members.index', compact('planOptions'));
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $hasWhatsappColumns = Schema::hasColumn('mosque_members', 'whatsapp_number')
            && Schema::hasColumn('mosque_members', 'whatsapp_same_as_phone');

        $select = [
            'mosque_members.id as id',
            'mosque_members.name as name',
            'mosque_members.phone as phone',
            'mosque_members.email as email',
            'mosque_members.status as status',
            'mosque_members.created_at as created_at',
        ];
        if ($hasWhatsappColumns) {
            $select[] = 'mosque_members.whatsapp_number as whatsapp_number';
            $select[] = 'mosque_members.whatsapp_same_as_phone as whatsapp_same_as_phone';
        }

        $query = MosqueMember::query()
            ->where('mosque_members.business_id', $businessId)
            ->select($select);

        if (Schema::hasColumn('mosque_members', 'deleted_at')) {
            $query->whereNull('mosque_members.deleted_at');
        }

        if (Schema::hasTable('mosque_family_members') && Schema::hasTable('mosque_families')) {
            $familyDeletedFilter = Schema::hasColumn('mosque_families', 'deleted_at') ? ' AND f.deleted_at IS NULL' : '';
            $query->addSelect(DB::raw(
                "(SELECT GROUP_CONCAT(DISTINCT f.family_code ORDER BY f.family_code SEPARATOR ', ') ".
                "FROM mosque_family_members fm JOIN mosque_families f ON f.id = fm.family_id AND f.business_id = {$businessId}{$familyDeletedFilter} ".
                "WHERE fm.business_id = {$businessId} AND fm.member_id = mosque_members.id) as family_codes"
            ));
        } else {
            $query->addSelect(DB::raw("'' as family_codes"));
        }

        if (! empty($request->input('status'))) {
            $query->where('status', $request->input('status'));
        }

        $familyCode = trim((string) $request->input('family_code', ''));
        if ($familyCode !== '' && Schema::hasTable('mosque_family_members') && Schema::hasTable('mosque_families')) {
            $query->whereExists(function ($q) use ($businessId, $familyCode) {
                $q->select(DB::raw(1))
                    ->from('mosque_family_members as fm')
                    ->join('mosque_families as f', function ($join) use ($businessId) {
                        $join->on('f.id', '=', 'fm.family_id')
                            ->where('f.business_id', '=', $businessId);

                        if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                            $join->whereNull('f.deleted_at');
                        }
                    })
                    ->where('fm.business_id', $businessId)
                    ->whereColumn('fm.member_id', 'mosque_members.id')
                    ->where('f.family_code', $familyCode);
            });
        }

        if (Schema::hasTable('mosque_member_plan_assignments') && Schema::hasTable('mosque_membership_plans')) {
            $query->leftJoin('mosque_member_plan_assignments as mpa', function ($join) use ($businessId) {
                $join->on('mpa.member_id', '=', 'mosque_members.id')
                    ->where('mpa.business_id', '=', $businessId)
                    ->where('mpa.active', '=', 1);
            });

            $query->leftJoin('mosque_membership_plans as mp', function ($join) use ($businessId) {
                $join->on('mp.id', '=', 'mpa.plan_id')
                    ->where('mp.business_id', '=', $businessId);
            });

            $query->addSelect([
                DB::raw('mp.name as fees_plan_name'),
                DB::raw('mp.type as billing_cycle'),
                DB::raw('mpa.start_ym as billing_start_ym'),
                DB::raw('mpa.plan_id as fees_plan_id'),
            ]);

            $planId = (int) $request->input('plan_id', 0);
            if ($planId > 0) {
                $query->where('mpa.plan_id', $planId);
            }

            $billingCycle = trim((string) $request->input('billing_cycle', ''));
            if (in_array($billingCycle, ['monthly', 'yearly'], true)) {
                $query->where('mp.type', $billingCycle);
            }

            $billingEnabled = trim((string) $request->input('billing_enabled', ''));
            if ($billingEnabled === 'yes') {
                $query->whereNotNull('mpa.id');
            } elseif ($billingEnabled === 'no') {
                $query->whereNull('mpa.id');
            }

            $billingStartYm = trim((string) $request->input('billing_start_ym', ''));
            if ($billingStartYm !== '') {
                $query->where('mpa.start_ym', $billingStartYm);
            }
        } else {
            $query->addSelect([
                DB::raw("'' as fees_plan_name"),
                DB::raw("'' as billing_cycle"),
                DB::raw("'' as billing_start_ym"),
                DB::raw("NULL as fees_plan_id"),
            ]);
        }

        return DataTables::of($query)
            ->editColumn('family_codes', function ($row) {
                $val = trim((string) ($row->family_codes ?? ''));
                return $val !== '' ? e($val) : '-';
            })
            ->editColumn('fees_plan_name', function ($row) {
                $val = trim((string) ($row->fees_plan_name ?? ''));
                return $val !== '' ? e($val) : '-';
            })
            ->editColumn('billing_cycle', function ($row) {
                $val = trim((string) ($row->billing_cycle ?? ''));
                return $val !== '' ? e(ucfirst($val)) : '-';
            })
            ->editColumn('billing_start_ym', function ($row) {
                $val = trim((string) ($row->billing_start_ym ?? ''));
                return $val !== '' ? e($val) : '-';
            })
            ->addColumn('whatsapp_number', function ($row) use ($hasWhatsappColumns) {
                $displayNumber = (string) ($row->phone ?? '');
                if ($hasWhatsappColumns) {
                    if (! empty($row->whatsapp_same_as_phone)) {
                        $displayNumber = (string) ($row->phone ?? '');
                    } else {
                        $displayNumber = (string) ($row->whatsapp_number ?? '');
                    }
                }

                $displayNumber = trim($displayNumber);
                if ($displayNumber === '') {
                    return '-';
                }

                $digits = preg_replace('/\D+/', '', $displayNumber);
                if (empty($digits)) {
                    return e($displayNumber);
                }

                return '<a href="https://wa.me/'.$digits.'" target="_blank" rel="noopener"><i class="fa fa-whatsapp"></i> '.e($displayNumber).'</a>';
            })
            ->addColumn('action', function ($row) {
                $card = '<a href="'.route('mosque.members.card.print', [$row->id]).'" class="btn btn-xs btn-default" target="_blank"><i class="fa fa-id-card"></i> Card</a>';
                $pdf = '<a href="'.route('mosque.members.card.pdf', [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $export = '<a href="'.route('mosque.members.export', [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-download"></i> Export</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MembersController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_member_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MembersController::class, 'destroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_member"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $card.' '.$pdf.' '.$export.' '.$edit.' '.$delete;
            })
            ->rawColumns(['action', 'whatsapp_number'])
            ->make(true);
    }

    public function search(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $term = trim((string) $request->input('q', ''));
        $familyCode = trim((string) $request->input('family_code', ''));

        $query = MosqueMember::query()
            ->where('business_id', $businessId);

        if (Schema::hasColumn('mosque_members', 'deleted_at')) {
            $query->whereNull('deleted_at');
        }

        if ($familyCode !== '' && Schema::hasTable('mosque_family_members') && Schema::hasTable('mosque_families')) {
            $query->whereExists(function ($q) use ($businessId, $familyCode) {
                $q->select(DB::raw(1))
                    ->from('mosque_family_members as fm')
                    ->join('mosque_families as f', function ($join) use ($businessId, $familyCode) {
                        $join->on('f.id', '=', 'fm.family_id')
                            ->where('f.business_id', '=', $businessId)
                            ->where('f.family_code', '=', $familyCode);

                        if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                            $join->whereNull('f.deleted_at');
                        }
                    })
                    ->where('fm.business_id', $businessId)
                    ->whereColumn('fm.member_id', 'mosque_members.id');
            });
        }

        if ($term !== '') {
            $query->where(function ($q) use ($term, $businessId) {
                $q->where('name', 'like', '%'.$term.'%')
                    ->orWhere('phone', 'like', '%'.$term.'%')
                    ->orWhere('email', 'like', '%'.$term.'%');

                if (Schema::hasTable('mosque_family_members') && Schema::hasTable('mosque_families')) {
                    $q->orWhereExists(function ($sub) use ($businessId, $term) {
                        $sub->select(DB::raw(1))
                            ->from('mosque_family_members as fm')
                            ->join('mosque_families as f', function ($join) use ($businessId, $term) {
                                $join->on('f.id', '=', 'fm.family_id')
                                    ->where('f.business_id', '=', $businessId)
                                    ->where('f.family_code', 'like', '%'.$term.'%');

                                if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                                    $join->whereNull('f.deleted_at');
                                }
                            })
                            ->where('fm.business_id', $businessId)
                            ->whereColumn('fm.member_id', 'mosque_members.id');
                    });
                }
            });
        }

        $results = $query
            ->orderBy('name')
            ->limit(20)
            ->get(['id', 'name', 'phone'])
            ->map(function ($m) {
                $label = (string) $m->name;
                $label .= ' (ID: '.$m->id.')';
                if (! empty($m->phone)) {
                    $label .= ' - '.$m->phone;
                }

                return [
                    'id' => (int) $m->id,
                    'text' => $label,
                ];
            })
            ->values()
            ->all();

        return response()->json(['results' => $results]);
    }

    public function create()
    {
        $this->ensurePermission();

        $businessId = $this->businessId();

        $plans = [];
        if (Schema::hasTable('mosque_membership_plans')) {
            $plans = DB::table('mosque_membership_plans')
                ->where('business_id', $businessId)
                ->where('active', 1)
                ->orderBy('name')
                ->get(['id', 'name', 'type', 'amount'])
                ->mapWithKeys(function ($p) {
                    $label = (string) $p->name;
                    if (! empty($p->type)) {
                        $label .= ' ('.ucfirst((string) $p->type).')';
                    }
                    return [(int) $p->id => $label];
                })
                ->all();
        }

        return view('mosque::members.create', compact('plans'));
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'whatsapp_number' => 'nullable|string|max:50',
            'whatsapp_same_as_phone' => 'sometimes|boolean',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'status' => 'required|in:Active,Inactive,Deceased,Migrated',
            'notes' => 'nullable|string',
            'billing_enabled' => 'sometimes|boolean',
            'membership_plan_id' => 'nullable|integer',
            'billing_start_ym' => 'nullable|date_format:Y-m',
        ]);

        try {
            $businessId = $this->businessId();

            $data = [
                'business_id' => $businessId,
                'location_id' => null,
                'name' => $request->input('name'),
                'phone' => $request->input('phone'),
                'email' => $request->input('email'),
                'address' => $request->input('address'),
                'status' => $request->input('status'),
                'notes' => $request->input('notes'),
            ];

            if (Schema::hasColumn('mosque_members', 'whatsapp_number') && Schema::hasColumn('mosque_members', 'whatsapp_same_as_phone')) {
                $same = $request->boolean('whatsapp_same_as_phone', true);
                $data['whatsapp_same_as_phone'] = $same ? 1 : 0;
                $data['whatsapp_number'] = $same ? ($request->input('phone') ?: null) : ($request->input('whatsapp_number') ?: null);
            }

            $member = MosqueMember::query()->create($data);

            if (Schema::hasTable('mosque_member_plan_assignments')) {
                $billingEnabled = $request->boolean('billing_enabled', false);
                $planId = (int) $request->input('membership_plan_id', 0);
                $startYm = trim((string) $request->input('billing_start_ym', ''));

                if ($billingEnabled && $planId > 0) {
                    $planExists = DB::table('mosque_membership_plans')
                        ->where('business_id', $businessId)
                        ->where('id', $planId)
                        ->where('active', 1)
                        ->exists();

                    if ($planExists) {
                        MosqueMemberPlanAssignment::query()->updateOrCreate(
                            ['business_id' => $businessId, 'member_id' => (int) $member->id],
                            [
                                'plan_id' => $planId,
                                'start_ym' => $startYm !== '' ? $startYm : null,
                                'active' => true,
                                'created_by' => (int) (auth()->id() ?? 0) ?: null,
                                'updated_by' => (int) (auth()->id() ?? 0) ?: null,
                            ]
                        );
                    }
                } else {
                    MosqueMemberPlanAssignment::query()
                        ->where('business_id', $businessId)
                        ->where('member_id', (int) $member->id)
                        ->update([
                            'active' => false,
                            'updated_by' => (int) (auth()->id() ?? 0) ?: null,
                        ]);
                }
            }

            MosqueAuditUtil::log($businessId, 'create', 'member', (int) $member->id, [
                'name' => $member->name,
                'status' => $member->status,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $member = MosqueMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $plans = [];
        if (Schema::hasTable('mosque_membership_plans')) {
            $plans = DB::table('mosque_membership_plans')
                ->where('business_id', $businessId)
                ->where('active', 1)
                ->orderBy('name')
                ->get(['id', 'name', 'type', 'amount'])
                ->mapWithKeys(function ($p) {
                    $label = (string) $p->name;
                    if (! empty($p->type)) {
                        $label .= ' ('.ucfirst((string) $p->type).')';
                    }
                    return [(int) $p->id => $label];
                })
                ->all();
        }

        $billing = null;
        if (Schema::hasTable('mosque_member_plan_assignments')) {
            $billing = MosqueMemberPlanAssignment::query()
                ->where('business_id', $businessId)
                ->where('member_id', (int) $member->id)
                ->first();
        }

        return view('mosque::members.edit', compact('member', 'plans', 'billing'));
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'whatsapp_number' => 'nullable|string|max:50',
            'whatsapp_same_as_phone' => 'sometimes|boolean',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'status' => 'required|in:Active,Inactive,Deceased,Migrated',
            'notes' => 'nullable|string',
            'billing_enabled' => 'sometimes|boolean',
            'membership_plan_id' => 'nullable|integer',
            'billing_start_ym' => 'nullable|date_format:Y-m',
        ]);

        try {
            $businessId = $this->businessId();
            $member = MosqueMember::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $data = [
                'name' => $request->input('name'),
                'phone' => $request->input('phone'),
                'email' => $request->input('email'),
                'address' => $request->input('address'),
                'status' => $request->input('status'),
                'notes' => $request->input('notes'),
            ];

            if (Schema::hasColumn('mosque_members', 'whatsapp_number') && Schema::hasColumn('mosque_members', 'whatsapp_same_as_phone')) {
                $same = $request->boolean('whatsapp_same_as_phone', true);
                $data['whatsapp_same_as_phone'] = $same ? 1 : 0;
                $data['whatsapp_number'] = $same ? ($request->input('phone') ?: null) : ($request->input('whatsapp_number') ?: null);
            }

            $member->update($data);

            if (Schema::hasTable('mosque_member_plan_assignments')) {
                $billingEnabled = $request->boolean('billing_enabled', false);
                $planId = (int) $request->input('membership_plan_id', 0);
                $startYm = trim((string) $request->input('billing_start_ym', ''));

                if ($billingEnabled && $planId > 0) {
                    $planExists = DB::table('mosque_membership_plans')
                        ->where('business_id', $businessId)
                        ->where('id', $planId)
                        ->where('active', 1)
                        ->exists();

                    if ($planExists) {
                        MosqueMemberPlanAssignment::query()->updateOrCreate(
                            ['business_id' => $businessId, 'member_id' => (int) $member->id],
                            [
                                'plan_id' => $planId,
                                'start_ym' => $startYm !== '' ? $startYm : null,
                                'active' => true,
                                'updated_by' => (int) (auth()->id() ?? 0) ?: null,
                            ]
                        );
                    }
                } else {
                    MosqueMemberPlanAssignment::query()
                        ->where('business_id', $businessId)
                        ->where('member_id', (int) $member->id)
                        ->update([
                            'active' => false,
                            'updated_by' => (int) (auth()->id() ?? 0) ?: null,
                        ]);
                }
            }

            MosqueAuditUtil::log($businessId, 'update', 'member', (int) $member->id, [
                'name' => $member->name,
                'status' => $member->status,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $member = MosqueMember::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $member->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'member', (int) $member->id, [
                'name' => $member->name,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'member', (int) $member->id, [
                'name' => (string) $member->name,
                'phone' => (string) ($member->phone ?? ''),
                'status' => (string) ($member->status ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function export($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $member = MosqueMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $families = DB::table('mosque_family_members as fm')
            ->join('mosque_families as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'fm.family_id')
                    ->where('f.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                    $join->whereNull('f.deleted_at');
                }
            })
            ->where('fm.business_id', $businessId)
            ->where('fm.member_id', $member->id)
            ->select(['f.id', 'f.family_code', 'f.address', 'fm.relation'])
            ->get();

        $fees = DB::table('mosque_member_fees as fee')
            ->where('fee.business_id', $businessId)
            ->where('fee.member_id', $member->id)
            ->orderBy('fee.period_ym')
            ->get();

        $payments = DB::table('mosque_member_payments as p')
            ->join('mosque_member_fees as fee', function ($join) use ($businessId) {
                $join->on('fee.id', '=', 'p.member_fee_id')
                    ->where('fee.business_id', '=', $businessId);
            })
            ->where('p.business_id', $businessId)
            ->where('fee.member_id', $member->id)
            ->select([
                'p.id',
                'p.paid_on',
                'p.amount',
                'p.method',
                'p.ref_no',
                'fee.period_ym',
                'p.created_at',
            ])
            ->orderBy('p.paid_on')
            ->get();

        MosqueAuditUtil::log($businessId, 'export', 'member', (int) $member->id, [
            'member_name' => $member->name,
        ]);

        $payload = [
            'exported_at' => now()->toIso8601String(),
            'business_id' => $businessId,
            'member' => [
                'id' => $member->id,
                'name' => $member->name,
                'phone' => $member->phone,
                'email' => $member->email,
                'address' => $member->address,
                'status' => $member->status,
                'notes' => $member->notes,
                'created_at' => optional($member->created_at)->toIso8601String(),
                'updated_at' => optional($member->updated_at)->toIso8601String(),
            ],
            'families' => $families,
            'membership' => [
                'fees' => $fees,
                'payments' => $payments,
            ],
        ];

        $filename = 'mosque_member_'.$member->id.'_export_'.now()->format('Ymd_His').'.json';

        return response()->streamDownload(function () use ($payload) {
            echo json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }, $filename, [
            'Content-Type' => 'application/json',
        ]);
    }

    public function cardPrint($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $member = MosqueMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $billing = null;
        if (Schema::hasTable('mosque_member_plan_assignments') && Schema::hasTable('mosque_membership_plans')) {
            $billing = DB::table('mosque_member_plan_assignments as a')
                ->join('mosque_membership_plans as p', function ($join) use ($businessId) {
                    $join->on('p.id', '=', 'a.plan_id')
                        ->where('p.business_id', '=', $businessId);
                })
                ->where('a.business_id', $businessId)
                ->where('a.member_id', (int) $member->id)
                ->where('a.active', 1)
                ->select(['a.start_ym', 'p.name as plan_name', 'p.type as plan_type'])
                ->first();
        }

        $families = DB::table('mosque_family_members as fm')
            ->join('mosque_families as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'fm.family_id')
                    ->where('f.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                    $join->whereNull('f.deleted_at');
                }
            })
            ->where('fm.business_id', $businessId)
            ->where('fm.member_id', $member->id)
            ->select(['f.id', 'f.family_code', 'f.address', 'fm.relation'])
            ->orderBy('f.family_code')
            ->get();

        MosqueAuditUtil::log($businessId, 'print', 'member_card', (int) $member->id, [
            'member_name' => $member->name,
        ]);

        return view('mosque::members.card', compact('member', 'families', 'profile', 'logoDataUri', 'billing'));
    }

    public function cardPdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $member = MosqueMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $billing = null;
        if (Schema::hasTable('mosque_member_plan_assignments') && Schema::hasTable('mosque_membership_plans')) {
            $billing = DB::table('mosque_member_plan_assignments as a')
                ->join('mosque_membership_plans as p', function ($join) use ($businessId) {
                    $join->on('p.id', '=', 'a.plan_id')
                        ->where('p.business_id', '=', $businessId);
                })
                ->where('a.business_id', $businessId)
                ->where('a.member_id', (int) $member->id)
                ->where('a.active', 1)
                ->select(['a.start_ym', 'p.name as plan_name', 'p.type as plan_type'])
                ->first();
        }

        $families = DB::table('mosque_family_members as fm')
            ->join('mosque_families as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'fm.family_id')
                    ->where('f.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                    $join->whereNull('f.deleted_at');
                }
            })
            ->where('fm.business_id', $businessId)
            ->where('fm.member_id', $member->id)
            ->select(['f.id', 'f.family_code', 'f.address', 'fm.relation'])
            ->orderBy('f.family_code')
            ->get();

        MosqueAuditUtil::log($businessId, 'pdf', 'member_card', (int) $member->id, [
            'member_name' => $member->name,
        ]);

        $pdf = Pdf::loadView('mosque::members.card', compact('member', 'families', 'profile', 'logoDataUri', 'billing'))
            ->setPaper('a4');

        $filename = 'member_card_'.$member->id.'.pdf';

        return $pdf->download($filename);
    }

    public function importForm()
    {
        $this->ensurePermission();

        return view('mosque::members.import');
    }

    public function importStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:2048',
        ]);

        $businessId = $this->businessId();

        $file = $request->file('file');
        $path = $file?->getRealPath();
        if (empty($path) || ! is_file($path)) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $created = 0;
        $updated = 0;
        $skipped = 0;

        $handle = fopen($path, 'r');
        if ($handle === false) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        try {
            $header = fgetcsv($handle);
            if (empty($header) || ! is_array($header)) {
                return ['success' => false, 'msg' => 'Invalid CSV header.'];
            }

            $map = [];
            foreach ($header as $index => $col) {
                $key = Str::of((string) $col)->trim()->lower()->replace(' ', '_')->replace('-', '_')->toString();
                $map[$index] = $key;
            }

            $allowedStatuses = ['Active', 'Inactive', 'Deceased', 'Migrated'];

            DB::beginTransaction();

            while (($row = fgetcsv($handle)) !== false) {
                if (! is_array($row) || count(array_filter($row, fn ($v) => trim((string) $v) !== '')) === 0) {
                    continue;
                }

                $data = [];
                foreach ($map as $i => $key) {
                    $data[$key] = array_key_exists($i, $row) ? trim((string) $row[$i]) : null;
                }

                $name = (string) ($data['name'] ?? '');
                if ($name === '') {
                    $skipped++;
                    continue;
                }

                $status = (string) ($data['status'] ?? 'Active');
                if (! in_array($status, $allowedStatuses, true)) {
                    $status = 'Active';
                }

                $payload = [
                    'name' => $name,
                    'phone' => $data['phone'] ?? null,
                    'email' => $data['email'] ?? null,
                    'address' => $data['address'] ?? null,
                    'status' => $status,
                    'notes' => $data['notes'] ?? null,
                ];

                $email = trim((string) ($payload['email'] ?? ''));
                $phone = trim((string) ($payload['phone'] ?? ''));

                $existing = null;
                if ($email !== '') {
                    $existing = MosqueMember::query()
                        ->where('business_id', $businessId)
                        ->where('email', $email)
                        ->first();
                } elseif ($phone !== '') {
                    $existing = MosqueMember::query()
                        ->where('business_id', $businessId)
                        ->where('phone', $phone)
                        ->first();
                }

                if (! empty($existing)) {
                    $existing->update($payload);
                    $updated++;
                } else {
                    MosqueMember::query()->create(array_merge($payload, [
                        'business_id' => $businessId,
                        'location_id' => null,
                    ]));
                    $created++;
                }
            }

            DB::commit();

            MosqueAuditUtil::log($businessId, 'import', 'members', null, [
                'created' => $created,
                'updated' => $updated,
                'skipped' => $skipped,
            ]);

            $msg = "Import complete. Created: {$created}, Updated: {$updated}, Skipped: {$skipped}.";

            return ['success' => true, 'msg' => $msg];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        } finally {
            fclose($handle);
        }
    }
}
