<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class MembersReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.members.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        return view('mosque::members.reports');
    }

    public function statusData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_members as m')
            ->where('m.business_id', $businessId)
            ->whereNull('m.deleted_at')
            ->select([
                'm.status',
                DB::raw('COUNT(m.id) as members_count'),
                DB::raw('MAX(m.created_at) as last_added_at'),
            ])
            ->groupBy('m.status');

        return DataTables::of($query)
            ->editColumn('status', function ($row) {
                return (string) ($row->status ?? '');
            })
            ->editColumn('last_added_at', function ($row) {
                return ! empty($row->last_added_at) ? \Carbon\Carbon::parse($row->last_added_at)->format('Y-m-d') : '';
            })
            ->make(true);
    }

    public function inactiveData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_members as m')
            ->where('m.business_id', $businessId)
            ->where('m.status', 'Inactive')
            ->whereNull('m.deleted_at')
            ->select(['m.id', 'm.name', 'm.phone', 'm.email', 'm.created_at']);

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('m.created_at', '>=', $request->input('start_date'))
                ->whereDate('m.created_at', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $card = '<a href="'.route('mosque.members.card.print', [$row->id]).'" class="btn btn-xs btn-default" target="_blank"><i class="fa fa-id-card"></i> Card</a>';
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MembersController::class, 'edit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_member_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                return $card.' '.$edit;
            })
            ->editColumn('created_at', function ($row) {
                return ! empty($row->created_at) ? \Carbon\Carbon::parse($row->created_at)->format('Y-m-d') : '';
            })
            ->rawColumns(['action'])
            ->make(true);
    }
}
