<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Utils\Util;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueMember;
use Modules\Mosque\Entities\MosqueMemberFee;
use Modules\Mosque\Entities\MosqueMemberPayment;
use Modules\Mosque\Entities\MosqueMemberPlanAssignment;
use Modules\Mosque\Entities\MosqueMembershipPlan;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Services\MemberFeeService;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class MembershipController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId) && auth()->check()) {
            $businessId = (int) (auth()->user()->business_id ?? 0);
        }
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    public function feesMembersSearch(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $term = trim((string) $request->input('q', ''));
        $familyCode = trim((string) $request->input('family_code', ''));

        $query = MosqueMember::query()
            ->where('business_id', $businessId);

        if (Schema::hasColumn('mosque_members', 'deleted_at')) {
            $query->whereNull('deleted_at');
        }

        if ($familyCode !== '' && Schema::hasTable('mosque_family_members') && Schema::hasTable('mosque_families')) {
            $query->whereExists(function ($q) use ($businessId, $familyCode) {
                $q->select(DB::raw(1))
                    ->from('mosque_family_members as fm')
                    ->join('mosque_families as f', function ($join) use ($businessId, $familyCode) {
                        $join->on('f.id', '=', 'fm.family_id')
                            ->where('f.business_id', '=', $businessId)
                            ->where('f.family_code', '=', $familyCode);

                        if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                            $join->whereNull('f.deleted_at');
                        }
                    })
                    ->where('fm.business_id', $businessId)
                    ->whereColumn('fm.member_id', 'mosque_members.id');
            });
        }

        if ($term !== '') {
            $query->where(function ($q) use ($term, $businessId) {
                $q->where('name', 'like', '%'.$term.'%')
                    ->orWhere('phone', 'like', '%'.$term.'%')
                    ->orWhere('email', 'like', '%'.$term.'%');

                if (Schema::hasTable('mosque_family_members') && Schema::hasTable('mosque_families')) {
                    $q->orWhereExists(function ($sub) use ($businessId, $term) {
                        $sub->select(DB::raw(1))
                            ->from('mosque_family_members as fm')
                            ->join('mosque_families as f', function ($join) use ($businessId, $term) {
                                $join->on('f.id', '=', 'fm.family_id')
                                    ->where('f.business_id', '=', $businessId)
                                    ->where('f.family_code', 'like', '%'.$term.'%');

                                if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                                    $join->whereNull('f.deleted_at');
                                }
                            })
                            ->where('fm.business_id', $businessId)
                            ->whereColumn('fm.member_id', 'mosque_members.id');
                    });
                }
            });
        }

        $results = $query
            ->orderBy('name')
            ->limit(20)
            ->get(['id', 'name', 'phone'])
            ->map(function ($m) {
                $label = (string) $m->name;
                $label .= ' (ID: '.$m->id.')';
                if (! empty($m->phone)) {
                    $label .= ' - '.$m->phone;
                }

                return [
                    'id' => (int) $m->id,
                    'text' => $label,
                ];
            })
            ->values()
            ->all();

        return response()->json(['results' => $results]);
    }

    public function feesFamilyCodesSearch(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $term = trim((string) $request->input('q', ''));

        if (Schema::hasTable('mosque_family_codes')) {
            $query = DB::table('mosque_family_codes')
                ->where('business_id', $businessId)
                ->where('active', 1)
                ->whereNull('deleted_at');

            if ($term !== '') {
                $query->where('code', 'like', '%'.$term.'%');
            }

            $codes = $query->orderBy('code')
                ->limit(20)
                ->pluck('code')
                ->all();
        } else {
            $query = DB::table('mosque_families')
                ->where('business_id', $businessId);

            if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                $query->whereNull('deleted_at');
            }

            if ($term !== '') {
                $query->where('family_code', 'like', '%'.$term.'%');
            }

            $codes = $query->whereNotNull('family_code')
                ->where('family_code', '!=', '')
                ->distinct()
                ->orderBy('family_code')
                ->limit(20)
                ->pluck('family_code')
                ->all();
        }

        $results = array_map(function ($c) {
            return ['id' => $c, 'text' => $c];
        }, $codes);

        return response()->json(['results' => $results]);
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.subscriptions.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        return view('mosque::membership.index');
    }

    // Plans
    public function plansIndex()
    {
        $this->ensurePermission();

        return view('mosque::membership.plans.index');
    }

    public function plansData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueMembershipPlan::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'type', 'amount', 'active']);

        return DataTables::of($query)
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->editColumn('active', function ($row) {
                return $row->active ? __('lang_v1.yes') : __('lang_v1.no');
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MembershipController::class, 'plansEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_plan_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MembershipController::class, 'plansDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_plan"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action', 'amount'])
            ->make(true);
    }

    public function plansCreate()
    {
        $this->ensurePermission();

        return view('mosque::membership.plans.create');
    }

    public function plansImportForm()
    {
        $this->ensurePermission();

        return view('mosque::membership.plans.import');
    }

    public function plansImportStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:2048',
        ]);

        $businessId = $this->businessId();

        $file = $request->file('file');
        $path = $file?->getRealPath();
        if (empty($path) || ! is_file($path)) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $created = 0;
        $updated = 0;
        $skipped = 0;

        $handle = fopen($path, 'r');
        if ($handle === false) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        try {
            $header = fgetcsv($handle);
            if (empty($header) || ! is_array($header)) {
                return ['success' => false, 'msg' => 'Invalid CSV header.'];
            }

            $map = [];
            foreach ($header as $index => $col) {
                $key = Str::of((string) $col)->trim()->lower()->replace(' ', '_')->replace('-', '_')->toString();
                $map[$index] = $key;
            }

            $allowedTypes = ['monthly', 'yearly'];

            DB::beginTransaction();

            while (($row = fgetcsv($handle)) !== false) {
                if (! is_array($row) || count(array_filter($row, fn ($v) => trim((string) $v) !== '')) === 0) {
                    continue;
                }

                $data = [];
                foreach ($map as $i => $key) {
                    $data[$key] = array_key_exists($i, $row) ? trim((string) $row[$i]) : null;
                }

                $name = (string) ($data['name'] ?? '');
                if ($name === '') {
                    $skipped++;
                    continue;
                }

                $type = (string) ($data['type'] ?? 'monthly');
                if (! in_array($type, $allowedTypes, true)) {
                    $type = 'monthly';
                }

                $amount = (float) ($data['amount'] ?? 0);
                if ($amount < 0) {
                    $amount = 0;
                }

                $activeRaw = strtolower(trim((string) ($data['active'] ?? '1')));
                $active = ! in_array($activeRaw, ['0', 'no', 'false', 'inactive'], true);

                $existing = MosqueMembershipPlan::query()
                    ->where('business_id', $businessId)
                    ->where('type', $type)
                    ->where('name', $name)
                    ->first();

                if (! empty($existing)) {
                    $existing->update([
                        'amount' => $amount,
                        'active' => $active,
                    ]);
                    $updated++;
                } else {
                    MosqueMembershipPlan::query()->create([
                        'business_id' => $businessId,
                        'name' => $name,
                        'type' => $type,
                        'amount' => $amount,
                        'active' => $active,
                    ]);
                    $created++;
                }
            }

            DB::commit();

            MosqueAuditUtil::log($businessId, 'import', 'membership_plans', null, [
                'created' => $created,
                'updated' => $updated,
                'skipped' => $skipped,
            ]);

            $msg = "Import complete. Created: {$created}, Updated: {$updated}, Skipped: {$skipped}.";
            return ['success' => true, 'msg' => $msg];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        } finally {
            fclose($handle);
        }
    }

    public function plansStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:monthly,yearly',
            'amount' => 'required|numeric|min:0',
            'active' => 'nullable|boolean',
        ]);

        try {
            $businessId = $this->businessId();
            MosqueMembershipPlan::query()->create([
                'business_id' => $businessId,
                'name' => $request->input('name'),
                'type' => $request->input('type'),
                'amount' => $request->input('amount'),
                'active' => (bool) $request->input('active', true),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function plansEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $plan = MosqueMembershipPlan::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        return view('mosque::membership.plans.edit', compact('plan'));
    }

    public function plansUpdate(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:monthly,yearly',
            'amount' => 'required|numeric|min:0',
            'active' => 'nullable|boolean',
        ]);

        try {
            $businessId = $this->businessId();
            $plan = MosqueMembershipPlan::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $plan->update([
                'name' => $request->input('name'),
                'type' => $request->input('type'),
                'amount' => $request->input('amount'),
                'active' => (bool) $request->input('active', true),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function plansDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $plan = MosqueMembershipPlan::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $plan->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'membership_plan', (int) $plan->id, [
                'name' => (string) $plan->name,
                'type' => (string) $plan->type,
                'amount' => (float) $plan->amount,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'membership plan', (int) $plan->id, [
                'name' => (string) $plan->name,
                'type' => (string) $plan->type,
                'amount' => (float) $plan->amount,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Fees
    public function feesIndex()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        // Do not preload all members; use searchable selector in UI.
        $members = collect();
        $plans = MosqueMembershipPlan::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        $payment_types = app(\App\Utils\Util::class)->payment_types(null, false, $businessId);

        return view('mosque::membership.fees.index', compact('members', 'plans', 'payment_types'));
    }

    public function feesCancelHistoryIndex()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $members = collect();
        $plans = MosqueMembershipPlan::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        $payment_types = app(\App\Utils\Util::class)->payment_types(null, false, $businessId);

        return view('mosque::membership.fees.cancel_history', compact('members', 'plans', 'payment_types'));
    }

    public function feesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_member_fees as f')
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'f.member_id')
                    ->where('m.business_id', '=', $businessId);
            })
            ->leftJoin('mosque_membership_plans as p', function ($join) use ($businessId) {
                $join->on('p.id', '=', 'f.plan_id')
                    ->where('p.business_id', '=', $businessId);
            })
            ->where('f.business_id', $businessId)
            ->select($this->feesSelectColumns())
            ->addSelect(DB::raw('(SELECT MAX(mp.id) FROM mosque_member_payments mp WHERE mp.business_id = '.$businessId.' AND mp.member_fee_id = f.id) as last_payment_id'));

        $query->addSelect(DB::raw('(SELECT mp2.ref_no FROM mosque_member_payments mp2 WHERE mp2.business_id = '.$businessId.' AND mp2.member_fee_id = f.id ORDER BY mp2.id DESC LIMIT 1) as receipt_no'));

        if (! empty($request->input('member_id'))) {
            $query->where('f.member_id', $request->input('member_id'));
        }
        if (! empty($request->input('plan_id'))) {
            $query->where('f.plan_id', $request->input('plan_id'));
        }
        if (! empty($request->input('family_code'))) {
            $query->where('m.family_code', trim((string) $request->input('family_code')));
        }
        $receiptNo = trim((string) $request->input('receipt_no', ''));
        if ($receiptNo !== '') {
            $query->whereExists(function ($q) use ($businessId, $receiptNo) {
                $q->select(DB::raw(1))
                    ->from('mosque_member_payments as mp3')
                    ->whereColumn('mp3.member_fee_id', 'f.id')
                    ->where('mp3.business_id', $businessId)
                    ->where('mp3.ref_no', 'like', '%'.$receiptNo.'%');
            });
        }

        $cycle = trim((string) $request->input('cycle', ''));
        if (in_array($cycle, ['monthly', 'yearly'], true)) {
            $hasPeriodTypeCol = Schema::hasColumn('mosque_member_fees', 'period_type');

            if ($cycle === 'monthly') {
                $query->where(function ($q) use ($hasPeriodTypeCol) {
                    if ($hasPeriodTypeCol) {
                        $q->where('f.period_type', 'monthly');
                    }
                    $q->orWhere(function ($qq) {
                        $qq->whereNull('f.period_type')
                            ->where(function ($qqq) {
                                $qqq->where('p.type', 'monthly')->orWhere('f.period_ym', 'like', '____-__');
                            });
                    });
                });
            } else {
                $query->where(function ($q) use ($hasPeriodTypeCol) {
                    if ($hasPeriodTypeCol) {
                        $q->where('f.period_type', 'yearly');
                    }
                    $q->orWhere(function ($qq) {
                        $qq->whereNull('f.period_type')
                            ->where(function ($qqq) {
                                $qqq->where('p.type', 'yearly')
                                    ->orWhereRaw("f.period_ym REGEXP '^[0-9]{4}(-01)?$'");
                            });
                    });
                });
            }
        }
        $statusFilter = trim((string) $request->input('status', ''));
        if ($statusFilter !== '') {
            if (in_array($statusFilter, ['due', 'paid', 'cancelled'], true) && Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                $query->where('f.lifecycle_status', $statusFilter);
            } elseif (in_array($statusFilter, ['pending', 'partial', 'paid'], true)) {
                $query->where('f.status', $statusFilter);
            }
        } else {
            // Default: hide cancelled unless explicitly filtered.
            if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                $query->where(function ($q) {
                    $q->whereNull('f.lifecycle_status')->orWhere('f.lifecycle_status', '!=', 'cancelled');
                });
            } elseif (Schema::hasColumn('mosque_member_fees', 'cancelled_at')) {
                $query->whereNull('f.cancelled_at');
            }
        }

        if (! empty($request->input('period_ym'))) {
            $periodYm = trim((string) $request->input('period_ym'));
            if (preg_match('/^\d{4}-\d{2}$/', $periodYm)) {
                // Monthly only (do not mix yearly into monthly filter).
                $query->where('f.period_ym', $periodYm);
            } elseif (preg_match('/^\d{4}$/', $periodYm)) {
                // Yearly: match both new YYYY and legacy YYYY-01.
                $legacy = $periodYm.'-01';
                $query->where(function ($q) use ($periodYm, $legacy) {
                    $q->where('f.period_ym', $periodYm)
                        ->orWhere('f.period_ym', $legacy);
                });
            }
        }

        $month = trim((string) $request->input('month', ''));
        if ($month !== '' && preg_match('/^\d{4}-\d{2}$/', $month)) {
            $query->where('f.period_ym', $month);
        }

        $dateStart = trim((string) $request->input('date_start', ''));
        $dateEnd = trim((string) $request->input('date_end', ''));
        if (Schema::hasColumn('mosque_member_fees', 'created_at') && $dateStart !== '' && $dateEnd !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateStart) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateEnd)) {
            $query->whereBetween('f.created_at', [$dateStart.' 00:00:00', $dateEnd.' 23:59:59']);
        }

        $totals = ['total_due' => 0, 'total_paid' => 0, 'total_outstanding' => 0];
        try {
            $totalsQuery = clone $query;
            $totalsQuery->getQuery()->orders = null;
            $totalsQuery->getQuery()->columns = null;
            $row = $totalsQuery
                ->selectRaw('COALESCE(SUM(f.due_amount),0) as total_due, COALESCE(SUM(f.paid_amount),0) as total_paid')
                ->first();
            if (! empty($row)) {
                $totals['total_due'] = (float) ($row->total_due ?? 0);
                $totals['total_paid'] = (float) ($row->total_paid ?? 0);
                $totals['total_outstanding'] = (float) $totals['total_due'] - (float) $totals['total_paid'];
            }
        } catch (\Throwable $e) {
            // Avoid breaking the table response if totals fail.
        }

        return DataTables::of($query)
            ->with(['totals' => $totals])
            ->addColumn('select', function ($row) {
                $lifecycle = (string) ($row->lifecycle_status ?? '');
                if ($lifecycle === '') {
                    $lifecycle = ((string) ($row->status ?? '') === 'paid') ? 'paid' : 'due';
                }
                $disabled = ($lifecycle === 'cancelled') ? 'disabled' : '';
                return '<input type="checkbox" class="mfee_row_select" value="'.(int) $row->id.'" '.$disabled.'>';
            })
            ->addColumn('action', function ($row) {
                $outstanding = max(0, (float) $row->due_amount - (float) $row->paid_amount);
                $lifecycle = (string) ($row->lifecycle_status ?? '');
                if ($lifecycle === '') {
                    $lifecycle = ((string) ($row->status ?? '') === 'paid') ? 'paid' : 'due';
                }
                $isCancelled = ($lifecycle === 'cancelled');

                $payFull = '';
                if (! $isCancelled && $outstanding > 0.00001) {
                    $payFull = '<button data-href="'.route('mosque.subscriptions.fees.pay_full', [$row->id]).'" class="btn btn-xs btn-primary mosque_fee_pay_full" data-toggle="tooltip" title="Records a payment and posts income to Finance."><i class="fa fa-check"></i> Pay full</button>';
                }
                $receipt = '';
                if (! empty($row->last_payment_id)) {
                    $receipt = '<a href="'.route('mosque.subscriptions.payments.print', [(int) $row->last_payment_id]).'" class="btn btn-xs btn-default" target="_blank"><i class="fa fa-print"></i> Receipt</a>';
                }

                $pay = '';
                if (! $isCancelled && $outstanding > 0.00001) {
                    $pay = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MembershipController::class, 'paymentsCreate'], [$row->id]).'" class="btn btn-xs btn-success btn-modal" data-container=".mosque_payment_modal" data-toggle="tooltip" title="Adds a payment and posts income to Finance."><i class="fa fa-money-bill"></i> '.__('purchase.add_payment').'</button>';
                }

                $cancel = '';
                if (! $isCancelled && auth()->user()->can('mosque.membership.cancel')) {
                    $cancel = '<button type="button" class="btn btn-xs btn-warning mosque_fee_cancel" data-href="'.route('mosque.membership_fees.cancel', [(int) $row->id]).'" data-fee-id="'.(int) $row->id.'"><i class="fa fa-ban"></i> Cancel</button>'
                        .' <i class="fa fa-info-circle text-muted" data-toggle="tooltip" title="Cancel will void payments and remove Finance income. You can restore from Cancel History using Reactivate."></i>';
                }

                $reactivate = '';
                if ($isCancelled && auth()->user()->can('mosque.membership.reactivate')) {
                    $reactivate = '<button type="button" class="btn btn-xs btn-success mosque_fee_reactivate" data-href="'.route('mosque.membership_fees.reactivate', [(int) $row->id]).'" data-toggle="tooltip" title="Restores the fee and re-posts Finance income from payments."><i class="fa fa-undo"></i> Reactivate</button>';
                }

                $delete = '';
                if ($isCancelled && auth()->user()->can('mosque.membership.delete')) {
                    $delete = '<button data-href="'.route('mosque.membership_fees.destroy', [(int) $row->id]).'" class="btn btn-xs btn-danger delete_mosque_fee"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>'
                        .' <i class="fa fa-info-circle text-muted" data-toggle="tooltip" title="Permanent delete: removes fee, payments, and Finance entries. Not recoverable (audit logs remain)."></i>';
                }

                return trim($payFull.' '.$receipt.' '.$pay.' '.$cancel.' '.$reactivate.' '.$delete);
            })
            ->editColumn('status', function ($row) {
                $lifecycle = (string) ($row->lifecycle_status ?? '');
                if ($lifecycle === '') {
                    $lifecycle = ((string) ($row->status ?? '') === 'paid') ? 'paid' : 'due';
                }

                if ($lifecycle === 'cancelled') {
                    return '<span class="label label-default">Cancelled</span>';
                }
                if ($lifecycle === 'paid') {
                    return '<span class="label label-success">Paid</span>';
                }
                return '<span class="label label-warning">Due</span>';
            })
            ->editColumn('cancelled_at', function ($row) {
                if (empty($row->cancelled_at)) {
                    return '';
                }
                try {
                    return e(\Carbon\Carbon::parse($row->cancelled_at)->format('Y-m-d H:i'));
                } catch (\Throwable $e) {
                    return e((string) $row->cancelled_at);
                }
            })
            ->editColumn('cancel_reason', function ($row) {
                return e((string) ($row->cancel_reason ?? ''));
            })
            ->editColumn('period_ym', function ($row) {
                $period = (string) ($row->period_ym ?? '');
                $type = (string) ($row->period_type ?? '') ?: (string) ($row->plan_type ?? '');
                if ($type === 'yearly' && strlen($period) >= 4) {
                    return e(substr($period, 0, 4));
                }
                return e($period);
            })
            ->editColumn('due_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->due_amount.'">'.$row->due_amount.'</span>';
            })
            ->editColumn('paid_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->paid_amount.'">'.$row->paid_amount.'</span>';
            })
            ->rawColumns(['select', 'action', 'status', 'due_amount', 'paid_amount'])
            ->make(true);
    }

    public function feesPayBulk(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $request->validate([
            'fee_ids' => 'required|array|min:1',
            'fee_ids.*' => 'integer|min:1',
            'paid_on' => 'nullable|date',
            'method' => 'nullable|string|max:50',
            'note' => 'nullable|string|max:1000',
        ]);

        try {
            $feeIds = array_values(array_unique(array_map('intval', (array) $request->input('fee_ids', []))));
            $paidOn = $request->input('paid_on') ? date('Y-m-d', strtotime((string) $request->input('paid_on'))) : now()->toDateString();
            $method = trim((string) $request->input('method', 'cash')) ?: 'cash';
            $note = trim((string) $request->input('note', '')) ?: null;

            $hasLifecycleStatusCol = Schema::hasColumn('mosque_member_fees', 'lifecycle_status');
            $receiptUrls = [];
            $paidCount = 0;
            $skippedCount = 0;

            DB::transaction(function () use ($businessId, $feeIds, $paidOn, $method, $note, $hasLifecycleStatusCol, &$paidCount, &$skippedCount, &$receiptUrls) {
                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'income', 'name' => 'Membership fees'],
                    ['active' => true, 'sort_order' => 2]
                );

                foreach ($feeIds as $feeId) {
                    $fee = MosqueMemberFee::query()
                        ->where('business_id', $businessId)
                        ->lockForUpdate()
                        ->find($feeId);

                    if (empty($fee)) {
                        $skippedCount++;
                        continue;
                    }

                    if ($hasLifecycleStatusCol && (string) ($fee->lifecycle_status ?? '') === 'cancelled') {
                        $skippedCount++;
                        continue;
                    }

                    $outstanding = max(0, (float) $fee->due_amount - (float) $fee->paid_amount);
                    if ($outstanding <= 0.00001) {
                        $skippedCount++;
                        continue;
                    }

                    $refNo = 'MF-'.now()->format('YmdHis').'-'.$businessId.'-'.$fee->id;

                    $payment = MosqueMemberPayment::query()->create([
                        'business_id' => $businessId,
                        'member_fee_id' => $fee->id,
                        'paid_on' => $paidOn,
                        'amount' => $outstanding,
                        'method' => $method,
                        'ref_no' => $refNo,
                        'note' => $note,
                    ]);

                    $fee->paid_amount = (float) $fee->paid_amount + (float) $payment->amount;
                    if ((float) $fee->paid_amount >= (float) $fee->due_amount) {
                        $fee->status = 'paid';
                        if ($hasLifecycleStatusCol) {
                            $fee->lifecycle_status = 'paid';
                        }
                    } elseif ((float) $fee->paid_amount > 0) {
                        $fee->status = 'partial';
                        if ($hasLifecycleStatusCol) {
                            $fee->lifecycle_status = 'due';
                        }
                    } else {
                        $fee->status = 'pending';
                        if ($hasLifecycleStatusCol) {
                            $fee->lifecycle_status = 'due';
                        }
                    }
                    $fee->save();

                    MosqueFinanceEntry::query()->create([
                        'business_id' => $businessId,
                        'location_id' => null,
                        'type' => 'income',
                        'category_id' => $category->id,
                        'amount' => $payment->amount,
                        'entry_date' => $payment->paid_on,
                        'ref_module' => 'membership',
                        'ref_id' => $payment->id,
                        'fund_tag' => null,
                        'note' => $payment->ref_no,
                        'created_by' => auth()->id(),
                    ]);

                    $paidCount++;
                    $receiptUrls[] = route('mosque.subscriptions.payments.print', [(int) $payment->id]);
                }
            });

            $msg = "Paid: {$paidCount}, Skipped: {$skippedCount}.";
            return response()->json(['success' => true, 'msg' => $msg, 'receipt_urls' => array_slice($receiptUrls, 0, 5)]);
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return response()->json(['success' => false, 'msg' => __('messages.something_went_wrong')], 500);
        }
    }

    private function feesSelectColumns(): array
    {
        $cols = [
            'f.id',
            'f.period_ym',
            'f.due_amount',
            'f.paid_amount',
            'f.status',
            'm.name as member_name',
            'p.name as plan_name',
            'p.type as plan_type',
        ];

        if (Schema::hasColumn('mosque_member_fees', 'period_type')) {
            $cols[] = 'f.period_type';
        }
        if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
            $cols[] = 'f.lifecycle_status';
        }
        if (Schema::hasColumn('mosque_member_fees', 'cancelled_at')) {
            $cols[] = 'f.cancelled_at';
        }
        if (Schema::hasColumn('mosque_member_fees', 'cancel_reason')) {
            $cols[] = 'f.cancel_reason';
        }
        if (Schema::hasColumn('mosque_member_fees', 'cancelled_by')) {
            $cols[] = 'f.cancelled_by';
        }

        return $cols;
    }

    public function feesPayFull(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        try {
            $feeId = (int) $id;

            $fee = MosqueMemberFee::query()
                ->where('business_id', $businessId)
                ->findOrFail($feeId);

            if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status') && (string) ($fee->lifecycle_status ?? '') === 'cancelled') {
                return ['success' => false, 'msg' => 'Fee is cancelled.'];
            }

            $outstanding = max(0, (float) $fee->due_amount - (float) $fee->paid_amount);
            if ($outstanding <= 0.00001) {
                return ['success' => true, 'msg' => 'Already paid.'];
            }

            $paidOn = now()->toDateString();
            $refNo = 'MF-'.now()->format('YmdHis').'-'.$businessId;
            $method = 'cash';

            $paymentId = null;

            DB::transaction(function () use ($businessId, $fee, $outstanding, $paidOn, $refNo, $method, &$paymentId) {
                $payment = MosqueMemberPayment::query()->create([
                    'business_id' => $businessId,
                    'member_fee_id' => $fee->id,
                    'paid_on' => $paidOn,
                    'amount' => $outstanding,
                    'method' => $method,
                    'ref_no' => $refNo,
                    'note' => null,
                ]);

                $paymentId = (int) $payment->id;

                $fee->paid_amount = (float) $fee->paid_amount + (float) $payment->amount;
                if ((float) $fee->paid_amount >= (float) $fee->due_amount) {
                    $fee->status = 'paid';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'paid';
                    }
                } elseif ((float) $fee->paid_amount > 0) {
                    $fee->status = 'partial';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'due';
                    }
                } else {
                    $fee->status = 'pending';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'due';
                    }
                }
                $fee->save();

                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'income', 'name' => 'Membership fees'],
                    ['active' => true, 'sort_order' => 2]
                );

                MosqueFinanceEntry::query()->create([
                    'business_id' => $businessId,
                    'location_id' => null,
                    'type' => 'income',
                    'category_id' => $category->id,
                    'amount' => $payment->amount,
                    'entry_date' => $payment->paid_on,
                    'ref_module' => 'membership',
                    'ref_id' => $payment->id,
                    'fund_tag' => null,
                    'note' => $payment->ref_no,
                    'created_by' => auth()->id(),
                ]);
            });

            $receiptUrl = $paymentId ? route('mosque.subscriptions.payments.print', [$paymentId]) : null;

            return ['success' => true, 'msg' => 'Payment recorded.', 'receipt_url' => $receiptUrl];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function feesGenerateForm()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        // Do not preload all members; use searchable selector in UI.
        $members = collect();

        $plans = MosqueMembershipPlan::query()
            ->where('business_id', $businessId)
            ->where('active', true)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::membership.fees.generate', compact('members', 'plans'));
    }

    public function feesMemberPlan(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $memberId = (int) $request->input('member_id', 0);
        if ($memberId <= 0) {
            return response()->json(['success' => false], 422);
        }

        $memberQuery = MosqueMember::query()
            ->where('business_id', $businessId)
            ->where('id', $memberId);
        if (Schema::hasColumn('mosque_members', 'deleted_at')) {
            $memberQuery->whereNull('deleted_at');
        }
        if (! $memberQuery->exists()) {
            return response()->json(['success' => false], 404);
        }

        $assignment = MosqueMemberPlanAssignment::query()
            ->where('business_id', $businessId)
            ->where('member_id', $memberId)
            ->where('active', true)
            ->first();

        return response()->json([
            'success' => true,
            'plan_id' => (int) ($assignment->plan_id ?? 0) ?: null,
            'start_ym' => (string) ($assignment->start_ym ?? ''),
        ]);
    }

    public function feesGenerate(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'mode' => 'required|in:single,bulk',
            'plan_source' => 'required|in:selected,assigned',
            'cycle' => 'nullable|in:monthly,yearly',
            'force_cycle' => 'sometimes|boolean',
            'exclude_members' => 'nullable|array',
            'exclude_members.*' => 'integer|min:1',
            'member_id' => 'nullable|integer',
            'members' => 'nullable|array',
            'members.*' => 'nullable|integer',
            'plan_id' => 'nullable|integer',
            'filter_plan_id' => 'nullable|integer',
            'start_ym' => 'nullable|string',
            'end_ym' => 'nullable|string',
            'start_period' => 'nullable|string',
            'end_period' => 'nullable|string',
            'family_code' => 'nullable|string|max:50',
            'family_codes' => 'nullable|array',
            'family_codes.*' => 'nullable|string|max:50',
            'save_default_plan' => 'sometimes|boolean',
            'fallback_plan_enabled' => 'sometimes|boolean',
            'preview' => 'sometimes|boolean',
        ]);

        try {
            $businessId = $this->businessId();
            $mode = (string) $request->input('mode', 'single');
            $planSource = (string) $request->input('plan_source', 'assigned');
            $cycle = (string) $request->input('cycle', 'monthly');
            $preview = (bool) $request->boolean('preview', false);
            $forceCycle = (bool) $request->boolean('force_cycle', false);

            $memberId = (int) $request->input('member_id', 0);
            $memberIds = array_values(array_filter(array_map('intval', (array) $request->input('members', []))));
            if ($memberId > 0 && empty($memberIds)) {
                $memberIds = [$memberId];
            }
            $excludeMembers = array_values(array_unique(array_filter(array_map('intval', (array) $request->input('exclude_members', [])))));

            $planId = (int) $request->input('plan_id', 0);
            $filterPlanId = (int) $request->input('filter_plan_id', 0);

            $familyCode = trim((string) $request->input('family_code', ''));
            $familyCodes = array_values(array_filter(array_map(function ($v) {
                return trim((string) $v);
            }, (array) $request->input('family_codes', []))));
            if ($familyCode !== '' && empty($familyCodes)) {
                $familyCodes = [$familyCode];
            }

            $saveDefaultPlan = $request->boolean('save_default_plan', false);
            $fallbackPlanEnabled = $request->boolean('fallback_plan_enabled', true);

            if ($mode === 'single' && empty($memberIds)) {
                return ['success' => false, 'msg' => 'Please select member(s)'];
            }
            if ($planSource === 'selected' && $planId <= 0) {
                return ['success' => false, 'msg' => 'Please select a plan'];
            }

            $startInput = trim((string) ($request->input('start_period') ?: $request->input('start_ym')));
            $endInput = trim((string) ($request->input('end_period') ?: $request->input('end_ym')));
            if ($startInput === '' || $endInput === '') {
                return ['success' => false, 'msg' => 'Please select start and end period'];
            }

            $startIsYm = preg_match('/^\d{4}-\d{2}$/', $startInput) === 1;
            $endIsYm = preg_match('/^\d{4}-\d{2}$/', $endInput) === 1;
            $startIsYear = preg_match('/^\d{4}$/', $startInput) === 1;
            $endIsYear = preg_match('/^\d{4}$/', $endInput) === 1;

            if ($cycle === 'monthly' && (! $startIsYm || ! $endIsYm)) {
                return ['success' => false, 'msg' => 'Monthly cycle requires YYYY-MM periods'];
            }
            if ($cycle === 'yearly' && (! ($startIsYm || $startIsYear) || ! ($endIsYm || $endIsYear))) {
                return ['success' => false, 'msg' => 'Yearly cycle requires YYYY or YYYY-MM periods'];
            }

            $startDate = $startIsYm
                ? \Carbon\Carbon::createFromFormat('Y-m', $startInput)->startOfMonth()
                : \Carbon\Carbon::createFromFormat('Y', $startInput)->startOfYear();
            $endDate = $endIsYm
                ? \Carbon\Carbon::createFromFormat('Y-m', $endInput)->startOfMonth()
                : \Carbon\Carbon::createFromFormat('Y', $endInput)->startOfYear();

            if ($endDate->lt($startDate)) {
                return ['success' => false, 'msg' => 'Invalid period range'];
            }

            // Default behavior expects YYYY-MM window for assigned-plan generation (legacy wizard).
            // Bill Generation mode can force monthly vs yearly and allow YYYY for yearly runs.
            if ($planSource === 'assigned') {
                if ($forceCycle) {
                    if ($cycle === 'yearly') {
                        if (! $startIsYear || ! $endIsYear) {
                            return ['success' => false, 'msg' => 'For yearly billing, please use YYYY for start/end.'];
                        }
                    } else {
                        if (! $startIsYm || ! $endIsYm) {
                            return ['success' => false, 'msg' => 'For monthly billing, please use YYYY-MM for start/end.'];
                        }
                    }
                } else {
                    if (! $startIsYm || ! $endIsYm) {
                        return ['success' => false, 'msg' => 'When using assigned plans, please use YYYY-MM for start/end. Yearly members will use the year portion automatically.'];
                    }
                }
            }

            $membersQuery = MosqueMember::query()
                ->where('business_id', $businessId);
            if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                $membersQuery->whereNull('deleted_at');
            }
            if (Schema::hasColumn('mosque_members', 'status')) {
                $membersQuery->where('status', 'Active');
            }

            if ($mode === 'single') {
                $membersQuery->whereIn('id', $memberIds);
            } elseif (! empty($familyCodes) && Schema::hasTable('mosque_family_members') && Schema::hasTable('mosque_families')) {
                $membersQuery->whereExists(function ($q) use ($businessId, $familyCodes) {
                    $q->select(DB::raw(1))
                        ->from('mosque_family_members as fm')
                        ->join('mosque_families as f', function ($join) use ($businessId, $familyCodes) {
                            $join->on('f.id', '=', 'fm.family_id')
                                ->where('f.business_id', '=', $businessId)
                                ->whereIn('f.family_code', $familyCodes);
                            if (Schema::hasColumn('mosque_families', 'deleted_at')) {
                                $join->whereNull('f.deleted_at');
                            }
                        })
                        ->where('fm.business_id', $businessId)
                        ->whereColumn('fm.member_id', 'mosque_members.id');
                });
            }
            if (! empty($excludeMembers)) {
                $membersQuery->whereNotIn('id', $excludeMembers);
            }

            if ($forceCycle && $planSource === 'assigned' && in_array($cycle, ['monthly', 'yearly'], true) && Schema::hasTable('mosque_member_plan_assignments')) {
                $membersQuery->whereExists(function ($q) use ($businessId, $cycle) {
                    $q->select(DB::raw(1))
                        ->from('mosque_member_plan_assignments as a')
                        ->join('mosque_membership_plans as p', function ($join) use ($businessId, $cycle) {
                            $join->on('p.id', '=', 'a.plan_id')
                                ->where('p.business_id', '=', $businessId)
                                ->where('p.active', true)
                                ->where('p.type', '=', $cycle);
                        })
                        ->where('a.business_id', $businessId)
                        ->where('a.active', true)
                        ->whereColumn('a.member_id', 'mosque_members.id');
                });
            }

            if ($filterPlanId > 0 && Schema::hasTable('mosque_member_plan_assignments')) {
                $membersQuery->whereExists(function ($q) use ($businessId, $filterPlanId) {
                    $q->select(DB::raw(1))
                        ->from('mosque_member_plan_assignments as a')
                        ->where('a.business_id', $businessId)
                        ->where('a.active', true)
                        ->where('a.plan_id', $filterPlanId)
                        ->whereColumn('a.member_id', 'mosque_members.id');
                });
            }

            $selectedPlan = null;
            if ($planId > 0) {
                $selectedPlan = MosqueMembershipPlan::query()
                    ->where('business_id', $businessId)
                    ->where('active', true)
                    ->findOrFail($planId);
            }

            $sessionKey = 'mosque.membership_fees.generate_last.'.$businessId;
            request()->session()->put($sessionKey, [
                'mode' => $mode,
                'plan_source' => $planSource,
                'cycle' => $cycle,
                'plan_id' => $planId,
                'filter_plan_id' => $filterPlanId,
                'start_period' => $startInput,
                'end_period' => $endInput,
                'family_codes' => $familyCodes,
            ]);

            $monthPeriods = [];
            if ($startIsYm && $endIsYm) {
                $cursor = $startDate->copy();
                while ($cursor->lte($endDate)) {
                    $monthPeriods[] = $cursor->format('Y-m');
                    $cursor->addMonth();
                }
            }

            $yearPeriods = [];
            $y = (int) $startDate->year;
            $yEnd = (int) $endDate->year;
            for ($yy = $y; $yy <= $yEnd; $yy++) {
                $yearPeriods[] = (string) $yy;
            }
            $yearPeriodsLegacy = array_map(fn ($yy) => $yy.'-01', $yearPeriods);

            $hasPeriodTypeCol = Schema::hasColumn('mosque_member_fees', 'period_type');
            $hasLifecycleStatusCol = Schema::hasColumn('mosque_member_fees', 'lifecycle_status');

            $created = 0;
            $skipped = 0;
            $restoredDue = 0;
            $noPlan = 0;
            $toCreateAmount = 0.0;
            $previewRows = [];
            $previewLimit = 250;

            $worker = function ($doWrite) use (
                $businessId,
                $membersQuery,
                $planSource,
                $cycle,
                $forceCycle,
                $selectedPlan,
                $saveDefaultPlan,
                $fallbackPlanEnabled,
                $startDate,
                $monthPeriods,
                $yearPeriods,
                $yearPeriodsLegacy,
                $hasPeriodTypeCol,
                $hasLifecycleStatusCol,
                $preview,
                &$created,
                &$skipped,
                &$restoredDue,
                &$noPlan,
                &$toCreateAmount,
                &$previewRows,
                $previewLimit
            ) {
                $membersQuery
                    ->select(['id', 'name'])
                    ->orderBy('id')
                    ->chunk(200, function ($chunk) use (
                        $doWrite,
                        $businessId,
                        $planSource,
                        $cycle,
                        $forceCycle,
                        $selectedPlan,
                        $saveDefaultPlan,
                        $fallbackPlanEnabled,
                        $startDate,
                        $monthPeriods,
                        $yearPeriods,
                        $yearPeriodsLegacy,
                        $hasPeriodTypeCol,
                        $hasLifecycleStatusCol,
                        $preview,
                        &$created,
                        &$skipped,
                        &$restoredDue,
                        &$noPlan,
                        &$toCreateAmount,
                        &$previewRows,
                        $previewLimit
                    ) {
                        $memberIds = $chunk->pluck('id')->map(fn ($id) => (int) $id)->all();

                        $assignments = collect();
                        if ($planSource === 'assigned' && Schema::hasTable('mosque_member_plan_assignments')) {
                            $assignments = MosqueMemberPlanAssignment::query()
                                ->where('business_id', $businessId)
                                ->whereIn('member_id', $memberIds)
                                ->where('active', true)
                                ->get(['member_id', 'plan_id', 'start_ym']);
                        }

                        $assignmentMap = [];
                        $assignmentStartMap = [];
                        $planIds = [];
                        foreach ($assignments as $a) {
                            $mid = (int) $a->member_id;
                            $pid = (int) $a->plan_id;
                            if ($mid > 0 && $pid > 0) {
                                $assignmentMap[$mid] = $pid;
                                $assignmentStartMap[$mid] = (string) ($a->start_ym ?? '');
                                $planIds[] = $pid;
                            }
                        }

                        if (! empty($selectedPlan)) {
                            $planIds[] = (int) $selectedPlan->id;
                        }

                        $plansById = [];
                        if (! empty($planIds)) {
                            $plansById = MosqueMembershipPlan::query()
                                ->where('business_id', $businessId)
                                ->where('active', true)
                                ->whereIn('id', array_values(array_unique($planIds)))
                                ->get()
                                ->keyBy('id')
                                ->all();
                        }

                        // Load existing fees for this chunk across the window (normalized by member+plan+period).
                        $existing = [];
                        $cancelledExisting = [];
                        $periodKeys = [];
                        if (! empty($monthPeriods)) {
                            $periodKeys = array_merge($periodKeys, $monthPeriods);
                        }
                        if (! empty($yearPeriods)) {
                            $periodKeys = array_merge($periodKeys, $yearPeriods, $yearPeriodsLegacy);
                        }
                        $periodKeys = array_values(array_unique($periodKeys));

                        if (! empty($periodKeys)) {
                            $rows = DB::table('mosque_member_fees as fee')
                                ->leftJoin('mosque_membership_plans as plan', function ($join) use ($businessId) {
                                    $join->on('plan.id', '=', 'fee.plan_id')
                                        ->where('plan.business_id', '=', $businessId);
                                })
                                ->where('fee.business_id', $businessId)
                                ->whereIn('fee.member_id', $memberIds)
                                ->whereIn('fee.period_ym', $periodKeys)
                                ->select(array_filter([
                                    'fee.id',
                                    'fee.member_id',
                                    'fee.plan_id',
                                    'fee.period_ym',
                                    'plan.type as plan_type',
                                    Schema::hasColumn('mosque_member_fees', 'period_type') ? 'fee.period_type' : null,
                                    Schema::hasColumn('mosque_member_fees', 'lifecycle_status') ? 'fee.lifecycle_status' : null,
                                ]))
                                ->get();

                            foreach ($rows as $r) {
                                $feeId = (int) ($r->id ?? 0);
                                $mid = (int) $r->member_id;
                                $pid = (int) ($r->plan_id ?? 0);
                                $p = (string) $r->period_ym;
                                $planType = (string) ($r->plan_type ?? '');
                                $periodType = (string) ($r->period_type ?? '');
                                $lifecycle = (string) ($r->lifecycle_status ?? '');
                                if (preg_match('/^\d{4}-01$/', $p) && ($periodType === 'yearly' || $planType === 'yearly')) {
                                    $p = substr($p, 0, 4);
                                }
                                if ($mid > 0 && $pid > 0) {
                                    if ($hasLifecycleStatusCol && $lifecycle === 'cancelled') {
                                        $cancelledExisting[$mid][$p][$pid] = $feeId;
                                    } else {
                                        $existing[$mid][$p][$pid] = true;
                                    }
                                }
                            }
                        }

                        $inserts = [];
                        $now = now();

                        foreach ($chunk as $m) {
                            $mid = (int) $m->id;
                            if ($mid <= 0) {
                                continue;
                            }

                            $plan = $selectedPlan;
                            $assignmentStartYm = '';
                            if ($planSource === 'assigned') {
                                $pid = (int) ($assignmentMap[$mid] ?? 0);
                                if ($pid > 0 && isset($plansById[$pid])) {
                                    $plan = $plansById[$pid];
                                    $assignmentStartYm = (string) ($assignmentStartMap[$mid] ?? '');
                                }

                                if (empty($plan) && $fallbackPlanEnabled && ! empty($selectedPlan)) {
                                    $plan = $selectedPlan;
                                    $assignmentStartYm = '';
                                }
                            }

                            if (empty($plan)) {
                                $noPlan++;
                                continue;
                            }

                            $effectiveCycle = (($planSource === 'assigned') && ! $forceCycle) ? (string) ($plan->type ?? 'monthly') : $cycle;
                            $periodList = ($effectiveCycle === 'yearly') ? $yearPeriods : $monthPeriods;

                            if (empty($periodList)) {
                                continue;
                            }

                            if ($doWrite && $saveDefaultPlan && ! empty($selectedPlan) && (int) $plan->id === (int) $selectedPlan->id) {
                                MosqueMemberPlanAssignment::query()->updateOrCreate(
                                    ['business_id' => $businessId, 'member_id' => $mid],
                                    [
                                        'plan_id' => (int) $plan->id,
                                        'start_ym' => (string) $startDate->format('Y-m'),
                                        'active' => true,
                                        'updated_by' => (int) (auth()->id() ?? 0) ?: null,
                                        'created_by' => (int) (auth()->id() ?? 0) ?: null,
                                    ]
                                );
                            }

                            foreach ($periodList as $period) {
                                // Respect assigned plan start (monthly uses YYYY-MM, yearly uses year portion).
                                if ($assignmentStartYm !== '') {
                                    if ($effectiveCycle === 'yearly') {
                                        $sy = substr($assignmentStartYm, 0, 4);
                                        if ($sy !== '' && $period < $sy) {
                                            continue;
                                        }
                                    } else {
                                        if ($period < $assignmentStartYm) {
                                            continue;
                                        }
                                    }
                                }

                                $planIdForKey = (int) ($plan->id ?? 0);
                                if ($planIdForKey > 0 && ! empty($cancelledExisting[$mid][$period][$planIdForKey])) {
                                    $feeId = (int) $cancelledExisting[$mid][$period][$planIdForKey];

                                    // Treat cancelled fee as "restore as due" for this run (Bill Generation should always produce Due rows).
                                    $created++;
                                    $restoredDue++;
                                    $toCreateAmount += (float) ($plan->amount ?? 0);

                                    if ($preview && count($previewRows) < $previewLimit) {
                                        $previewRows[] = [
                                            'member_id' => (int) $mid,
                                            'plan_id' => (int) $planIdForKey,
                                            'member' => (string) ($m->name ?? ''),
                                            'period' => $period,
                                            'plan' => (string) ($plan->name ?? ''),
                                            'amount' => (float) ($plan->amount ?? 0),
                                            'state' => 'restore_due',
                                        ];
                                    }

                                    if ($doWrite && $feeId > 0) {
                                        // Restore the fee as a clean Due bill (do not restore payments/finance; keep any past payments voided).
                                        $fee = MosqueMemberFee::query()
                                            ->where('business_id', $businessId)
                                            ->lockForUpdate()
                                            ->find($feeId);

                                        if (! empty($fee)) {
                                            $fee->paid_amount = 0;
                                            $fee->status = 'pending';
                                            if ($hasLifecycleStatusCol) {
                                                $fee->lifecycle_status = 'due';
                                            }
                                            if (Schema::hasColumn('mosque_member_fees', 'cancel_reason')) {
                                                $fee->cancel_reason = null;
                                            }
                                            if (Schema::hasColumn('mosque_member_fees', 'cancelled_by')) {
                                                $fee->cancelled_by = null;
                                            }
                                            if (Schema::hasColumn('mosque_member_fees', 'cancelled_at')) {
                                                $fee->cancelled_at = null;
                                            }
                                            $fee->save();
                                        }

                                        // Mark as existing in-memory.
                                        $existing[$mid][$period][$planIdForKey] = true;
                                    }

                                    continue;
                                }
                                if ($planIdForKey > 0 && ! empty($existing[$mid][$period][$planIdForKey])) {
                                    $skipped++;
                                    if ($preview && count($previewRows) < $previewLimit) {
                                        $previewRows[] = [
                                            'member_id' => (int) $mid,
                                            'plan_id' => (int) $planIdForKey,
                                            'member' => (string) ($m->name ?? ''),
                                            'period' => $period,
                                            'plan' => (string) ($plan->name ?? ''),
                                            'amount' => (float) ($plan->amount ?? 0),
                                            'state' => 'skipped',
                                        ];
                                    }
                                    continue;
                                }

                                $created++;
                                $toCreateAmount += (float) ($plan->amount ?? 0);

                                if ($preview && count($previewRows) < $previewLimit) {
                                    $previewRows[] = [
                                        'member_id' => (int) $mid,
                                        'plan_id' => (int) ($plan->id ?? 0),
                                        'member' => (string) ($m->name ?? ''),
                                        'period' => $period,
                                        'plan' => (string) ($plan->name ?? ''),
                                        'amount' => (float) ($plan->amount ?? 0),
                                        'state' => 'create',
                                    ];
                                }

                                if ($doWrite) {
                                    $row = [
                                        'business_id' => $businessId,
                                        'member_id' => $mid,
                                        'plan_id' => (int) $plan->id,
                                        'period_ym' => $period,
                                        'due_amount' => (float) ($plan->amount ?? 0),
                                        'paid_amount' => 0,
                                        'status' => 'pending',
                                        'created_at' => $now,
                                        'updated_at' => $now,
                                    ];

                                    if ($hasPeriodTypeCol) {
                                        $row['period_type'] = $effectiveCycle;
                                    }
                                    if ($hasLifecycleStatusCol) {
                                        $row['lifecycle_status'] = 'due';
                                    }

                                    $inserts[] = $row;

                                    // Mark as existing in-memory so we don't double insert within the same request.
                                    $planIdForKey = (int) ($plan->id ?? 0);
                                    if ($planIdForKey > 0) {
                                        $existing[$mid][$period][$planIdForKey] = true;
                                    }
                                }
                            }
                        }

                        if ($doWrite && ! empty($inserts)) {
                            foreach (array_chunk($inserts, 500) as $batch) {
                                DB::table('mosque_member_fees')->insert($batch);
                            }
                        }
                    });
            };

            if ($preview) {
                $worker(false);

                return [
                    'success' => true,
                    'msg' => 'Preview ready.',
                    'preview' => [
                        'rows' => $previewRows,
                        'created' => $created,
                        'skipped' => $skipped,
                        'restored_due' => $restoredDue,
                        'no_plan' => $noPlan,
                        'total_amount' => $toCreateAmount,
                    ],
                ];
            }

            DB::transaction(function () use ($worker) {
                $worker(true);
            });

            $msg = "Generated {$created} fee(s)";
            if ($restoredDue > 0) {
                $msg .= ", restored {$restoredDue} as due";
            }
            if ($skipped > 0) {
                $msg .= ", skipped {$skipped} existing";
            }
            if ($noPlan > 0) {
                $msg .= ", {$noPlan} member(s) without plan";
            }

            return ['success' => true, 'msg' => $msg];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function feesDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $fee = MosqueMemberFee::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status') && (string) ($fee->lifecycle_status ?? '') !== 'cancelled') {
                return ['success' => false, 'msg' => 'Cancel first.'];
            }

            $hasPayments = MosqueMemberPayment::query()
                ->where('business_id', $businessId)
                ->where('member_fee_id', $fee->id)
                ->exists();

            if ($hasPayments) {
                return ['success' => false, 'msg' => 'Cannot delete a fee with payments.'];
            }

            $fee->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'member_fee', (int) $fee->id, [
                'member_id' => (int) $fee->member_id,
                'plan_id' => (int) $fee->plan_id,
                'period_ym' => (string) $fee->period_ym,
                'due_amount' => (float) $fee->due_amount,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'membership fee', (int) $fee->id, [
                'member_id' => (int) $fee->member_id,
                'plan_id' => (int) $fee->plan_id,
                'period_ym' => (string) $fee->period_ym,
                'due_amount' => (float) $fee->due_amount,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Payments
    public function paymentsCreate($feeId)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $fee = MosqueMemberFee::query()
            ->where('business_id', $businessId)
            ->findOrFail($feeId);

        $payment_types = app(Util::class)->payment_types(null, false, $businessId);

        return view('mosque::membership.payments.create', compact('fee', 'payment_types'));
    }

    public function paymentsStore(Request $request, $feeId)
    {
        $this->ensurePermission();

        $request->validate([
            'paid_on' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'method' => 'nullable|string|max:50',
            'ref_no' => 'nullable|string|max:100',
            'note' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $refNo = $request->input('ref_no');
            if (empty($refNo)) {
                $refNo = 'MF-'.now()->format('YmdHis').'-'.$businessId;
            }

            DB::transaction(function () use ($businessId, $feeId, $request, $refNo) {
                $fee = MosqueMemberFee::query()
                    ->where('business_id', $businessId)
                    ->findOrFail($feeId);

                if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status') && (string) ($fee->lifecycle_status ?? '') === 'cancelled') {
                    abort(422, 'Fee is cancelled.');
                }

                $payment = MosqueMemberPayment::query()->create([
                    'business_id' => $businessId,
                    'member_fee_id' => $fee->id,
                    'paid_on' => $request->input('paid_on'),
                    'amount' => $request->input('amount'),
                    'method' => $request->input('method'),
                    'ref_no' => $refNo,
                    'note' => $request->input('note'),
                ]);

                $fee->paid_amount = (float) $fee->paid_amount + (float) $payment->amount;
                if ((float) $fee->paid_amount >= (float) $fee->due_amount) {
                    $fee->status = 'paid';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'paid';
                    }
                } elseif ((float) $fee->paid_amount > 0) {
                    $fee->status = 'partial';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'due';
                    }
                } else {
                    $fee->status = 'pending';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'due';
                    }
                }
                $fee->save();

                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'income', 'name' => 'Membership fees'],
                    ['active' => true, 'sort_order' => 2]
                );

                MosqueFinanceEntry::query()->create([
                    'business_id' => $businessId,
                    'location_id' => null,
                    'type' => 'income',
                    'category_id' => $category->id,
                    'amount' => $payment->amount,
                    'entry_date' => $payment->paid_on,
                    'ref_module' => 'membership',
                    'ref_id' => $payment->id,
                    'fund_tag' => null,
                    'note' => $payment->ref_no,
                    'created_by' => auth()->id(),
                ]);
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function paymentsIndex()
    {
        $this->ensurePermission();

        return view('mosque::membership.payments.index');
    }

    public function paymentsData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_member_payments as mp')
            ->join('mosque_member_fees as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'mp.member_fee_id')
                    ->where('f.business_id', '=', $businessId);
            })
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'f.member_id')
                    ->where('m.business_id', '=', $businessId);
            })
            ->where('mp.business_id', $businessId)
            ->select([
                'mp.id',
                'mp.paid_on',
                'mp.amount',
                'mp.method',
                'mp.ref_no',
                'm.name as member_name',
                'f.period_ym',
            ]);

        if (Schema::hasColumn('mosque_member_payments', 'voided_at')) {
            $query->whereNull('mp.voided_at');
        }

        return DataTables::of($query)
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\MembershipController::class, 'paymentPrint'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $pdf = '<a href="'.action([\Modules\Mosque\Http\Controllers\MembershipController::class, 'paymentPdf'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\MembershipController::class, 'paymentsDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_payment"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $print.' '.$pdf.' '.$delete;
            })
            ->rawColumns(['amount', 'action'])
            ->make(true);
    }

    public function paymentsDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();

            $payment = MosqueMemberPayment::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            if (Schema::hasColumn('mosque_member_payments', 'voided_at') && ! empty($payment->voided_at)) {
                return ['success' => false, 'msg' => 'Cannot delete a voided payment. Reactivate the fee first.'];
            }

            $meta = [
                'member_fee_id' => (int) $payment->member_fee_id,
                'paid_on' => (string) $payment->paid_on,
                'amount' => (float) $payment->amount,
                'method' => (string) ($payment->method ?? ''),
                'ref_no' => (string) ($payment->ref_no ?? ''),
            ];

            DB::transaction(function () use ($businessId, $id) {
                $payment = MosqueMemberPayment::query()
                    ->where('business_id', $businessId)
                    ->findOrFail($id);

                $fee = MosqueMemberFee::query()
                    ->where('business_id', $businessId)
                    ->findOrFail($payment->member_fee_id);

                $fee->paid_amount = max((float) $fee->paid_amount - (float) $payment->amount, 0);
                if ((float) $fee->paid_amount >= (float) $fee->due_amount) {
                    $fee->status = 'paid';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'paid';
                    }
                } elseif ((float) $fee->paid_amount > 0) {
                    $fee->status = 'partial';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'due';
                    }
                } else {
                    $fee->status = 'pending';
                    if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                        $fee->lifecycle_status = 'due';
                    }
                }
                $fee->save();

                $payment->delete();

                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'membership')
                    ->where('ref_id', $id)
                    ->delete();
            });

            MosqueAuditUtil::log($businessId, 'delete', 'member_payment', (int) $id, $meta);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'membership payment', (int) $id, $meta);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function paymentPrint($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $payment = MosqueMemberPayment::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $fee = MosqueMemberFee::query()
            ->where('business_id', $businessId)
            ->findOrFail($payment->member_fee_id);

        $member = MosqueMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($fee->member_id);

        $plan = MosqueMembershipPlan::query()
            ->where('business_id', $businessId)
            ->find($fee->plan_id);

        $planType = (string) (($fee->period_type ?? '') ?: ($plan->type ?? 'monthly'));
        $periodLabel = (string) ($fee->period_ym ?? '');
        if ($planType === 'monthly' && preg_match('/^\d{4}-\d{2}$/', $periodLabel)) {
            try {
                $periodLabel = \Carbon\Carbon::createFromFormat('Y-m', $periodLabel)->format('F Y');
            } catch (\Exception $e) {
                // keep raw period_ym
            }
        } elseif ($planType === 'yearly' && strlen($periodLabel) >= 4) {
            $periodLabel = substr($periodLabel, 0, 4);
        }

        $isVoided = false;
        if (Schema::hasColumn('mosque_member_payments', 'voided_at') && ! empty($payment->voided_at)) {
            $isVoided = true;
        }
        if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status') && (string) ($fee->lifecycle_status ?? '') === 'cancelled') {
            $isVoided = true;
        }

        if ($isVoided) {
            $statusText = 'Cancelled';
            $statusClass = 'danger';
        } else {
            $statusText = ((string) ($fee->status ?? '') === 'paid') ? 'Paid' : (((string) ($fee->status ?? '') === 'partial') ? 'Partial' : 'Due');
            $statusClass = ((string) ($fee->status ?? '') === 'paid') ? 'success' : (((string) ($fee->status ?? '') === 'partial') ? 'warning' : 'danger');
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        MosqueAuditUtil::log($businessId, 'print', 'membership_receipt', (int) $payment->id, [
            'ref_no' => $payment->ref_no,
            'period_ym' => $fee->period_ym,
        ]);

        return view('mosque::membership.payments.print', compact('profile', 'logoDataUri', 'payment', 'fee', 'member', 'plan', 'planType', 'periodLabel', 'statusText', 'statusClass', 'settings', 'isVoided'));
    }

    public function paymentPdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $payment = MosqueMemberPayment::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $fee = MosqueMemberFee::query()
            ->where('business_id', $businessId)
            ->findOrFail($payment->member_fee_id);

        $member = MosqueMember::query()
            ->where('business_id', $businessId)
            ->findOrFail($fee->member_id);

        $plan = MosqueMembershipPlan::query()
            ->where('business_id', $businessId)
            ->find($fee->plan_id);

        $planType = (string) (($fee->period_type ?? '') ?: ($plan->type ?? 'monthly'));
        $periodLabel = (string) ($fee->period_ym ?? '');
        if ($planType === 'monthly' && preg_match('/^\d{4}-\d{2}$/', $periodLabel)) {
            try {
                $periodLabel = \Carbon\Carbon::createFromFormat('Y-m', $periodLabel)->format('F Y');
            } catch (\Exception $e) {
                // keep raw period_ym
            }
        } elseif ($planType === 'yearly' && strlen($periodLabel) >= 4) {
            $periodLabel = substr($periodLabel, 0, 4);
        }

        $isVoided = false;
        if (Schema::hasColumn('mosque_member_payments', 'voided_at') && ! empty($payment->voided_at)) {
            $isVoided = true;
        }
        if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status') && (string) ($fee->lifecycle_status ?? '') === 'cancelled') {
            $isVoided = true;
        }

        if ($isVoided) {
            $statusText = 'Cancelled';
            $statusClass = 'danger';
        } else {
            $statusText = ((string) ($fee->status ?? '') === 'paid') ? 'Paid' : (((string) ($fee->status ?? '') === 'partial') ? 'Partial' : 'Due');
            $statusClass = ((string) ($fee->status ?? '') === 'paid') ? 'success' : (((string) ($fee->status ?? '') === 'partial') ? 'warning' : 'danger');
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $row = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $row?->settings ?: [];
        }

        MosqueAuditUtil::log($businessId, 'pdf', 'membership_receipt', (int) $payment->id, [
            'ref_no' => $payment->ref_no,
            'period_ym' => $fee->period_ym,
        ]);

        $pdf = Pdf::loadView('mosque::membership.payments.print', compact('profile', 'logoDataUri', 'payment', 'fee', 'member', 'plan', 'planType', 'periodLabel', 'statusText', 'statusClass', 'settings', 'isVoided'))
            ->setPaper('a4');

        $filename = 'membership_receipt_'.$payment->ref_no.'.pdf';

        return $pdf->download($filename);
    }
}
