<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Utils\MosqueQueryUtil;
use Yajra\DataTables\Facades\DataTables;

class MembershipReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.subscriptions.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();

        return view('mosque::membership.reports');
    }

    public function monthlyCollectionsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_member_payments as p')
            ->join('mosque_member_fees as fee', function ($join) use ($businessId) {
                $join->on('fee.id', '=', 'p.member_fee_id')
                    ->where('fee.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_member_fees', 'deleted_at')) {
                    $join->whereNull('fee.deleted_at');
                }
                if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                    $join->where(function ($q) {
                        $q->whereNull('fee.lifecycle_status')->orWhere('fee.lifecycle_status', '!=', 'cancelled');
                    });
                } elseif (Schema::hasColumn('mosque_member_fees', 'cancelled_at')) {
                    $join->whereNull('fee.cancelled_at');
                }
            })
            ->where('p.business_id', $businessId)
            ->select([
                DB::raw("DATE_FORMAT(p.paid_on, '%Y-%m') as month_ym"),
                DB::raw('COUNT(p.id) as payments_count'),
                DB::raw('COUNT(DISTINCT fee.member_id) as members_count'),
                DB::raw('SUM(p.amount) as total_amount'),
            ])
            ->groupBy(DB::raw("DATE_FORMAT(p.paid_on, '%Y-%m')"));

        $query = MosqueQueryUtil::whereNotDeleted($query, 'mosque_member_payments', 'p');
        if (Schema::hasColumn('mosque_member_payments', 'voided_at')) {
            $query->whereNull('p.voided_at');
        }

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('p.paid_on', '>=', $request->input('start_date'))
                ->whereDate('p.paid_on', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->editColumn('total_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->total_amount.'">'.$row->total_amount.'</span>';
            })
            ->rawColumns(['total_amount'])
            ->make(true);
    }

    public function outstandingDuesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_member_fees as fee')
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'fee.member_id')
                    ->where('m.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $join->whereNull('m.deleted_at');
                }
            })
            ->leftJoin('mosque_membership_plans as plan', function ($join) use ($businessId) {
                $join->on('plan.id', '=', 'fee.plan_id')
                    ->where('plan.business_id', '=', $businessId);
            })
            ->where('fee.business_id', $businessId)
            ->whereIn('fee.status', ['pending', 'partial'])
            ->select([
                'fee.id',
                'fee.period_ym',
                'fee.status',
                'fee.due_amount',
                'fee.paid_amount',
                DB::raw('(fee.due_amount - fee.paid_amount) as balance_amount'),
                'm.name as member_name',
                'm.phone as member_phone',
                'plan.name as plan_name',
            ]);

        $query = MosqueQueryUtil::whereNotDeleted($query, 'mosque_member_fees', 'fee');
        if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
            $query->where(function ($q) {
                $q->whereNull('fee.lifecycle_status')->orWhere('fee.lifecycle_status', '!=', 'cancelled');
            });
        } elseif (Schema::hasColumn('mosque_member_fees', 'cancelled_at')) {
            $query->whereNull('fee.cancelled_at');
        }

        if (! empty($request->input('period_ym'))) {
            $query->where('fee.period_ym', $request->input('period_ym'));
        }
        if (! empty($request->input('status')) && in_array($request->input('status'), ['pending', 'partial'], true)) {
            $query->where('fee.status', $request->input('status'));
        }

        return DataTables::of($query)
            ->editColumn('due_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->due_amount.'">'.$row->due_amount.'</span>';
            })
            ->editColumn('paid_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->paid_amount.'">'.$row->paid_amount.'</span>';
            })
            ->editColumn('balance_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->balance_amount.'">'.$row->balance_amount.'</span>';
            })
            ->rawColumns(['due_amount', 'paid_amount', 'balance_amount'])
            ->make(true);
    }

    public function paymentHistoryData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_member_payments as p')
            ->join('mosque_member_fees as fee', function ($join) use ($businessId) {
                $join->on('fee.id', '=', 'p.member_fee_id')
                    ->where('fee.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_member_fees', 'deleted_at')) {
                    $join->whereNull('fee.deleted_at');
                }
                if (Schema::hasColumn('mosque_member_fees', 'lifecycle_status')) {
                    $join->where(function ($q) {
                        $q->whereNull('fee.lifecycle_status')->orWhere('fee.lifecycle_status', '!=', 'cancelled');
                    });
                } elseif (Schema::hasColumn('mosque_member_fees', 'cancelled_at')) {
                    $join->whereNull('fee.cancelled_at');
                }
            })
            ->join('mosque_members as m', function ($join) use ($businessId) {
                $join->on('m.id', '=', 'fee.member_id')
                    ->where('m.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_members', 'deleted_at')) {
                    $join->whereNull('m.deleted_at');
                }
            })
            ->where('p.business_id', $businessId)
            ->select([
                'p.id',
                'p.paid_on',
                'p.amount',
                'p.method',
                'p.ref_no',
                'fee.period_ym',
                'm.name as member_name',
                'm.phone as member_phone',
            ]);

        $query = MosqueQueryUtil::whereNotDeleted($query, 'mosque_member_payments', 'p');
        if (Schema::hasColumn('mosque_member_payments', 'voided_at')) {
            $query->whereNull('p.voided_at');
        }

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('p.paid_on', '>=', $request->input('start_date'))
                ->whereDate('p.paid_on', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $print = '<a href="'.route('mosque.subscriptions.payments.print', [$row->id]).'" class="btn btn-xs btn-default" target="_blank"><i class="fa fa-print"></i> Print</a>';
                $pdf = '<a href="'.route('mosque.subscriptions.payments.pdf', [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';
                return $print.' '.$pdf;
            })
            ->editColumn('paid_on', function ($row) {
                return ! empty($row->paid_on) ? \Carbon\Carbon::parse($row->paid_on)->format('Y-m-d') : '';
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->rawColumns(['amount', 'action'])
            ->make(true);
    }
}
