<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueBankBalanceUtil;
use Modules\Mosque\Utils\MosqueQueryUtil;
use Modules\Mosque\Services\PledgeStatusService;
use Carbon\Carbon;

class MosqueController extends Controller
{
    private function ensureAnyMosquePermission(): void
    {
        $permissions = [
            'mosque.members.manage',
            'mosque.subscriptions.manage',
            'mosque.donations.manage',
            'mosque.pledges.view',
            'mosque.finance.income',
            'mosque.finance.expense',
            'mosque.finance.reports',
            'mosque.finance.bank.manage',
            'mosque.finance.bank.transfer',
            'mosque.finance.cashout.manage',
            'mosque.committee.manage',
            'mosque.staff.manage',
            'mosque.marriage.manage',
            'mosque.death.manage',
            'mosque.assets.manage',
            'mosque.events.manage',
            'mosque.audit.view',
            'mosque.manage',
            'mosque.license.manage',
        ];

        if (! auth()->user()->hasAnyPermission($permissions)) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensurePermission(string $permission): void
    {
        if (! auth()->user()->can($permission)) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function dashboard()
    {
        $this->ensureAnyMosquePermission();

        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settingsRow = MosqueSetting::query()->where('business_id', $businessId)->first();
            $settings = $settingsRow?->settings ?: [];
        }

        $timezone = (string) ($profile?->timezone ?: request()->session()->get('business.time_zone') ?: config('app.timezone'));
        $nowTz = now()->copy()->timezone($timezone);

        $hijriDate = null;
        if (! empty($settings['hijri_display_enabled'])) {
            try {
                if (class_exists(\IntlDateFormatter::class)) {
                    $fmt = new \IntlDateFormatter(
                        app()->getLocale() ?: 'en',
                        \IntlDateFormatter::FULL,
                        \IntlDateFormatter::NONE,
                        $timezone,
                        \IntlDateFormatter::TRADITIONAL,
                        'EEEE, d MMMM y G'
                    );
                    $fmt->setCalendar(\IntlDateFormatter::TRADITIONAL);
                    $hijriDate = $fmt->format($nowTz);
                }
            } catch (\Exception $e) {
                $hijriDate = null;
            }
        }

        // Default to current year for "Selected" dashboards.
        $defaultStart = $nowTz->copy()->startOfYear()->format('Y-m-d');
        $defaultEnd = $nowTz->copy()->endOfYear()->format('Y-m-d');

        $startDate = $this->parseDateParam((string) request()->query('start_date', $defaultStart), $timezone, $defaultStart);
        $endDate = $this->parseDateParam((string) request()->query('end_date', $defaultEnd), $timezone, $defaultEnd);
        if ($startDate > $endDate) {
            [$startDate, $endDate] = [$endDate, $startDate];
        }

        $counts = [
            'members_total' => Schema::hasTable('mosque_members')
                ? (int) MosqueQueryUtil::whereNotDeleted(DB::table('mosque_members')->where('business_id', $businessId), 'mosque_members')->count()
                : 0,
            'members_active' => Schema::hasTable('mosque_members')
                ? (int) MosqueQueryUtil::whereNotDeleted(DB::table('mosque_members')->where('business_id', $businessId)->where('status', 'Active'), 'mosque_members')->count()
                : 0,
            'donations_month' => Schema::hasTable('mosque_donations')
                ? (float) MosqueQueryUtil::whereNotDeleted(DB::table('mosque_donations')->where('business_id', $businessId)->whereDate('date', '>=', $startDate)->whereDate('date', '<=', $endDate), 'mosque_donations')->sum('amount')
                : 0.0,
            'membership_month' => Schema::hasTable('mosque_member_payments')
                ? (function () use ($businessId, $startDate, $endDate) {
                    $q = MosqueQueryUtil::whereNotDeleted(
                        DB::table('mosque_member_payments')->where('business_id', $businessId)->whereDate('paid_on', '>=', $startDate)->whereDate('paid_on', '<=', $endDate),
                        'mosque_member_payments'
                    );
                    if (Schema::hasColumn('mosque_member_payments', 'voided_at')) {
                        $q->whereNull('voided_at');
                    }
                    return (float) $q->sum('amount');
                })()
                : 0.0,
        ];

        $finance = [
            'module_income' => 0.0,
            'module_expense' => 0.0,
            'core_sales' => 0.0,
            'core_purchases' => 0.0,
            'income_total' => 0.0,
            'expense_total' => 0.0,
            'net_total' => 0.0,
        ];

        if (Schema::hasTable('mosque_finance_entries')) {
            $sums = MosqueQueryUtil::whereNotDeleted(DB::table('mosque_finance_entries'), 'mosque_finance_entries')
                ->where('business_id', $businessId)
                ->whereDate('entry_date', '>=', $startDate)
                ->whereDate('entry_date', '<=', $endDate)
                ->select('type', DB::raw('SUM(amount) as total'))
                ->groupBy('type')
                ->pluck('total', 'type')
                ->toArray();

            $finance['module_income'] = (float) ($sums['income'] ?? 0);
            $finance['module_expense'] = (float) ($sums['expense'] ?? 0);
        }

        // Core POS linkage: enabled by default; can be disabled by setting mosque_settings.settings.include_core_pos = false.
        $includeCorePos = array_key_exists('include_core_pos', $settings) ? (bool) $settings['include_core_pos'] : true;
        $finance['core_sales'] = 0.0;
        $finance['core_purchases'] = 0.0;
        $finance['core_sales_due'] = 0.0;
        $finance['core_purchases_due'] = 0.0;

        if ($includeCorePos && Schema::hasTable('transactions') && Schema::hasTable('transaction_payments')) {
            $salesPaidExpr = "(SELECT COALESCE(SUM(IF(tp.is_return = 1, -1*tp.amount, tp.amount)),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";
            $purchPaidExpr = "(SELECT COALESCE(SUM(tp.amount),0) FROM transaction_payments tp WHERE tp.transaction_id = t.id)";

            $salesBase = DB::table('transactions as t')
                ->where('t.business_id', $businessId)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereDate('t.transaction_date', '>=', $startDate)
                ->whereDate('t.transaction_date', '<=', $endDate);

            $finance['core_sales'] = (float) $salesBase->sum(DB::raw($salesPaidExpr));
            $finance['core_sales_due'] = (float) $salesBase->sum(DB::raw("GREATEST(t.final_total - ({$salesPaidExpr}), 0)"));

            $purchBase = DB::table('transactions as t')
                ->where('t.business_id', $businessId)
                ->where('t.type', 'purchase')
                ->where('t.status', 'received')
                ->whereDate('t.transaction_date', '>=', $startDate)
                ->whereDate('t.transaction_date', '<=', $endDate);

            $finance['core_purchases'] = (float) $purchBase->sum(DB::raw($purchPaidExpr));
            $finance['core_purchases_due'] = (float) $purchBase->sum(DB::raw("GREATEST(t.final_total - ({$purchPaidExpr}), 0)"));
        }

        // Wallet totals use paid/received amounts only.
        $finance['income_total'] = $finance['module_income'] + $finance['core_sales'];
        $finance['expense_total'] = $finance['module_expense'] + $finance['core_purchases'];
        $finance['net_total'] = $finance['income_total'] - $finance['expense_total'];

        $financeBreakdown = [
            'income_categories' => [],
            'expense_categories' => [],
            'income_sources' => [],
            'expense_sources' => [],
            'donation_types' => [],
        ];

        if (Schema::hasTable('mosque_finance_categories') && Schema::hasTable('mosque_finance_entries')) {
            $hasFinanceDeletedAt = Schema::hasColumn('mosque_finance_entries', 'deleted_at');
            $incomeCats = DB::table('mosque_finance_categories as c')
                ->leftJoin('mosque_finance_entries as e', function ($join) use ($businessId, $startDate, $endDate, $hasFinanceDeletedAt) {
                    $join->on('e.category_id', '=', 'c.id')
                        ->where('e.business_id', '=', $businessId)
                        ->where('e.type', '=', 'income')
                        ->when($hasFinanceDeletedAt, fn ($j) => $j->whereNull('e.deleted_at'))
                        ->whereDate('e.entry_date', '>=', $startDate)
                        ->whereDate('e.entry_date', '<=', $endDate);
                })
                ->where('c.business_id', $businessId)
                ->where('c.type', 'income')
                ->select('c.name', DB::raw('COALESCE(SUM(e.amount),0) as total'))
                ->groupBy('c.id', 'c.name')
                ->orderBy('c.sort_order')
                ->orderBy('c.name')
                ->get();
            $financeBreakdown['income_categories'] = $incomeCats->map(fn ($r) => ['name' => (string) $r->name, 'total' => (float) $r->total])->all();

            $expenseCats = DB::table('mosque_finance_categories as c')
                ->leftJoin('mosque_finance_entries as e', function ($join) use ($businessId, $startDate, $endDate, $hasFinanceDeletedAt) {
                    $join->on('e.category_id', '=', 'c.id')
                        ->where('e.business_id', '=', $businessId)
                        ->where('e.type', '=', 'expense')
                        ->when($hasFinanceDeletedAt, fn ($j) => $j->whereNull('e.deleted_at'))
                        ->whereDate('e.entry_date', '>=', $startDate)
                        ->whereDate('e.entry_date', '<=', $endDate);
                })
                ->where('c.business_id', $businessId)
                ->where('c.type', 'expense')
                ->select('c.name', DB::raw('COALESCE(SUM(e.amount),0) as total'))
                ->groupBy('c.id', 'c.name')
                ->orderBy('c.sort_order')
                ->orderBy('c.name')
                ->get();
            $financeBreakdown['expense_categories'] = $expenseCats->map(fn ($r) => ['name' => (string) $r->name, 'total' => (float) $r->total])->all();
        }

        $financeBreakdown['income_sources'] = [
            ['name' => 'Module finance entries', 'total' => (float) $finance['module_income']],
        ];
        $financeBreakdown['expense_sources'] = [
            ['name' => 'Module finance entries', 'total' => (float) $finance['module_expense']],
        ];
        if ($includeCorePos) {
            $financeBreakdown['income_sources'][] = ['name' => 'POS Sales Received (core)', 'total' => (float) $finance['core_sales']];
            $financeBreakdown['income_sources'][] = ['name' => 'POS Sales Due (core)', 'total' => (float) $finance['core_sales_due']];
            $financeBreakdown['expense_sources'][] = ['name' => 'POS Purchases Paid (core)', 'total' => (float) $finance['core_purchases']];
            $financeBreakdown['expense_sources'][] = ['name' => 'POS Purchases Due (core)', 'total' => (float) $finance['core_purchases_due']];
        }

        if (Schema::hasTable('mosque_donations')) {
            $donTypes = MosqueQueryUtil::whereNotDeleted(DB::table('mosque_donations'), 'mosque_donations')
                ->where('business_id', $businessId)
                ->whereDate('date', '>=', $startDate)
                ->whereDate('date', '<=', $endDate)
                ->select('type', DB::raw('SUM(amount) as total'))
                ->groupBy('type')
                ->orderBy('type')
                ->get();
            $financeBreakdown['donation_types'] = $donTypes->map(fn ($r) => ['name' => strtoupper((string) $r->type), 'total' => (float) $r->total])->all();
        }

        $pledges = [
            'enabled' => empty($settings) ? true : (bool) ($settings['pledges_enabled'] ?? true),
            'due_soon_count' => 0,
            'due_30_count' => 0,
            'overdue_count' => 0,
            'fulfilled_month' => 0.0,
            'fulfillment_pct' => 0.0,
        ];

        if ($pledges['enabled'] && Schema::hasTable('mosq_pledges') && Schema::hasTable('mosq_pledge_fulfillments')) {
            try {
                PledgeStatusService::recalcBusiness($businessId, $settings);
            } catch (\Exception $e) {
                // ignore
            }

            $pledges['due_soon_count'] = (int) DB::table('mosq_pledges')->where('business_id', $businessId)->where('status', 'due_soon')->count();
            $pledges['overdue_count'] = (int) DB::table('mosq_pledges')->where('business_id', $businessId)->where('status', 'overdue')->count();
            $pledges['due_30_count'] = (int) DB::table('mosq_pledges')
                ->where('business_id', $businessId)
                ->whereNotIn('status', ['fulfilled', 'cancelled'])
                ->whereNotNull('due_date')
                ->whereBetween('due_date', [now()->toDateString(), now()->addDays(30)->toDateString()])
                ->count();

            $pledges['fulfilled_month'] = (float) DB::table('mosq_pledge_fulfillments as f')
                ->join('mosq_pledges as p', 'p.id', '=', 'f.pledge_id')
                ->where('p.business_id', $businessId)
                ->whereDate('f.date', '>=', $startDate)
                ->whereDate('f.date', '<=', $endDate)
                ->sum(DB::raw("COALESCE(CASE WHEN f.channel IN ('cash','bank','card','online','mfs') THEN f.amount_cash ELSE f.est_value END,0)"));

            $totalEst = (float) DB::table('mosq_pledges')->where('business_id', $businessId)->whereNotIn('status', ['cancelled'])->sum('est_total_value');
            $totalFul = (float) DB::table('mosq_pledge_fulfillments as f')
                ->join('mosq_pledges as p', 'p.id', '=', 'f.pledge_id')
                ->where('p.business_id', $businessId)
                ->sum(DB::raw("COALESCE(CASE WHEN f.channel IN ('cash','bank','card','online','mfs') THEN f.amount_cash ELSE f.est_value END,0)"));
            $pledges['fulfillment_pct'] = $totalEst > 0 ? round(($totalFul / $totalEst) * 100, 2) : 0.0;
        }

        $dateRange = ['start' => $startDate, 'end' => $endDate];

        $bankSummary = [
            // As-of end date, so dashboard date range impacts balances consistently.
            'cash_net_balance' => MosqueBankBalanceUtil::cashNetBalance($businessId, $endDate),
            'bank_wallet_total' => MosqueBankBalanceUtil::bankWalletTotal($businessId, $endDate),
            'net_including_bank' => MosqueBankBalanceUtil::overallNetBalance($businessId, $endDate),
        ];

        return view('mosque::dashboard', compact('profile', 'counts', 'finance', 'financeBreakdown', 'pledges', 'settings', 'timezone', 'hijriDate', 'dateRange', 'includeCorePos', 'bankSummary'));
    }

    private function parseDateParam(string $raw, string $timezone, string $fallbackYmd): string
    {
        $raw = trim($raw);
        if ($raw === '') {
            return $fallbackYmd;
        }

        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $raw)) {
            try {
                return Carbon::createFromFormat('Y-m-d', $raw, $timezone)->format('Y-m-d');
            } catch (\Throwable $e) {
                return $fallbackYmd;
            }
        }

        $businessFormat = (string) request()->session()->get('business.date_format', '');
        $candidateFormats = array_values(array_unique(array_filter([
            $businessFormat,
            'm/d/Y',
            'd/m/Y',
            'm-d-Y',
            'd-m-Y',
            'Y/m/d',
        ])));

        // Heuristic for dd/mm vs mm/dd when both are plausible.
        if (preg_match('/^(\d{1,2})\/(\d{1,2})\/(\d{4})$/', $raw, $m)) {
            $a = (int) $m[1];
            $b = (int) $m[2];
            if ($a > 12 && $b <= 12) {
                array_unshift($candidateFormats, 'd/m/Y');
            } elseif ($b > 12 && $a <= 12) {
                array_unshift($candidateFormats, 'm/d/Y');
            }
        }

        foreach ($candidateFormats as $fmt) {
            try {
                $dt = Carbon::createFromFormat($fmt, $raw, $timezone);
                return $dt->format('Y-m-d');
            } catch (\Throwable $e) {
                // continue
            }
        }

        // Last resort: Carbon parser.
        try {
            return Carbon::parse($raw, $timezone)->format('Y-m-d');
        } catch (\Throwable $e) {
            return $fallbackYmd;
        }
    }

    public function members()
    {
        $this->ensurePermission('mosque.members.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_members')]);
    }

    public function subscriptions()
    {
        $this->ensurePermission('mosque.subscriptions.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_membership_fees')]);
    }

    public function donations()
    {
        $this->ensurePermission('mosque.donations.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_donations')]);
    }

    public function finance()
    {
        if (! auth()->user()->hasAnyPermission(['mosque.finance.income', 'mosque.finance.expense', 'mosque.finance.reports'])) {
            abort(403, 'Unauthorized action.');
        }

        return app(\Modules\Mosque\Http\Controllers\FinanceController::class)->index();
    }

    public function committee()
    {
        $this->ensurePermission('mosque.committee.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_committee')]);
    }

    public function staff()
    {
        $this->ensurePermission('mosque.staff.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_staff')]);
    }

    public function marriage()
    {
        $this->ensurePermission('mosque.marriage.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_marriage')]);
    }

    public function death()
    {
        $this->ensurePermission('mosque.death.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_death')]);
    }

    public function assets()
    {
        $this->ensurePermission('mosque.assets.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_assets')]);
    }

    public function events()
    {
        $this->ensurePermission('mosque.events.manage');

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_events')]);
    }

    public function reports()
    {
        if (! auth()->user()->hasAnyPermission(['mosque.finance.reports', 'mosque.audit.view'])) {
            abort(403, 'Unauthorized action.');
        }

        return view('mosque::placeholders.page', ['title' => __('mosque::mosque.menu_reports')]);
    }
}
