<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueBankAccount;
use Modules\Mosque\Entities\MosqueBankWalletTransaction;
use Modules\Mosque\Entities\MosqueNetBankTransfer;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueBankBalanceUtil;
use Yajra\DataTables\Facades\DataTables;

class NetToBankTransferController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->hasAnyPermission([
            'mosque.finance.expense',
            'mosque.finance.reports',
            'mosque.finance.bank.transfer',
        ])) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $banks = [];
        if (Schema::hasTable('mosque_bank_accounts')) {
            $banks = MosqueBankAccount::query()
                ->where('business_id', $businessId)
                ->where('is_active', true)
                ->orderBy('bank_name')
                ->select(['id'])
                ->selectRaw("CONCAT(bank_name,' - ',account_no) as label")
                ->pluck('label', 'id')
                ->toArray();
        }

        return view('mosque::finance.transfers.index', compact('banks'));
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_net_bank_transfers as t')
            ->join('mosque_bank_accounts as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 't.bank_account_id')
                    ->where('b.business_id', '=', $businessId);
            })
            ->where('t.business_id', $businessId)
            ->select([
                't.id',
                't.amount',
                't.transfer_date',
                't.status',
                't.note',
                'b.bank_name',
                'b.account_no',
                'b.account_holder',
            ])
            ->orderByDesc('t.transfer_date')
            ->orderByDesc('t.id');

        if (! empty($request->input('bank_account_id'))) {
            $query->where('t.bank_account_id', $request->input('bank_account_id'));
        }
        if (! empty($request->input('status'))) {
            $query->where('t.status', $request->input('status'));
        }
        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('t.transfer_date', '>=', $request->input('start_date'))
                ->whereDate('t.transfer_date', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->amount.'">'.$row->amount.'</span>';
            })
            ->addColumn('bank', function ($row) {
                $bank = e((string) $row->bank_name).' - '.e((string) $row->account_no);
                $holder = trim((string) ($row->account_holder ?? ''));
                if ($holder !== '') {
                    return $bank.'<br><small class="text-muted">'.e($holder).'</small>';
                }
                return $bank;
            })
            ->addColumn('action', function ($row) {
                $buttons = '';

                if ((string) $row->status === 'completed') {
                    $voidUrl = action([self::class, 'void'], [$row->id], false);
                    $buttons .= '<button data-href="'.$voidUrl.'" class="btn btn-xs btn-warning delete_mosque_net_bank_transfer"><i class="fa fa-ban"></i> Void</button> ';
                }

                $deleteUrl = route('mosque.finance.transfers.destroy', [(int) $row->id]);
                $buttons .= '<button data-href="'.$deleteUrl.'" class="btn btn-xs btn-danger delete_mosque_net_bank_transfer_permanent"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';

                return trim($buttons) !== '' ? trim($buttons) : '-';
            })
            ->rawColumns(['amount', 'bank', 'action'])
            ->make(true);
    }

    public function store(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'bank_account_id' => 'required|integer',
            'amount' => 'required|numeric|min:0.01',
            'transfer_date' => 'required|date',
            'note' => 'nullable|string|max:2000',
        ]);

        try {
            $businessId = $this->businessId();
            if (! Schema::hasTable('mosque_bank_accounts') || ! Schema::hasTable('mosque_bank_wallet_transactions') || ! Schema::hasTable('mosque_net_bank_transfers')) {
                return ['success' => false, 'msg' => __('messages.something_went_wrong')];
            }

            $bank = MosqueBankAccount::query()
                ->where('business_id', $businessId)
                ->where('id', $request->input('bank_account_id'))
                ->where('is_active', true)
                ->firstOrFail();

            $amount = (float) $request->input('amount');
            $cashBalance = MosqueBankBalanceUtil::cashNetBalance($businessId);
            if ($amount - $cashBalance > 0.0001) {
                return ['success' => false, 'msg' => __('mosque::mosque.bank_err_insufficient_cash_balance')];
            }

            $transfer = null;
            DB::transaction(function () use ($businessId, $bank, $amount, $request, &$transfer) {
                $transfer = MosqueNetBankTransfer::query()->create([
                    'business_id' => $businessId,
                    'bank_account_id' => (int) $bank->id,
                    'amount' => $amount,
                    'transfer_date' => $request->input('transfer_date'),
                    'note' => $request->input('note'),
                    'status' => 'completed',
                    'created_by' => auth()->id(),
                ]);

                MosqueBankWalletTransaction::query()->create([
                    'business_id' => $businessId,
                    'bank_account_id' => (int) $bank->id,
                    'direction' => 'credit',
                    'amount' => $amount,
                    'txn_date' => $request->input('transfer_date'),
                    'ref_module' => 'net_to_bank',
                    'ref_id' => (int) $transfer->id,
                    'note' => $request->input('note'),
                    'created_by' => auth()->id(),
                ]);
            });

            if ($transfer) {
                MosqueAuditUtil::log($businessId, 'create', 'net_to_bank_transfer', (int) $transfer->id, [
                    'bank_account_id' => (int) $bank->id,
                    'amount' => $amount,
                ]);
            }

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function void($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $transfer = MosqueNetBankTransfer::query()->where('business_id', $businessId)->find($id);
            if (! $transfer) {
                return ['success' => true, 'msg' => __('lang_v1.success')];
            }
            if ((string) $transfer->status !== 'completed') {
                return ['success' => false, 'msg' => __('messages.something_went_wrong')];
            }

            DB::transaction(function () use ($businessId, $transfer) {
                $transfer->update([
                    'status' => 'voided',
                    'voided_by' => auth()->id(),
                    'voided_at' => now(),
                ]);

                MosqueBankWalletTransaction::query()->create([
                    'business_id' => $businessId,
                    'bank_account_id' => (int) $transfer->bank_account_id,
                    'direction' => 'debit',
                    'amount' => (float) $transfer->amount,
                    'txn_date' => (string) ($transfer->transfer_date ?? now()->format('Y-m-d')),
                    'ref_module' => 'net_to_bank_void',
                    'ref_id' => (int) $transfer->id,
                    'note' => 'Voided transfer',
                    'created_by' => auth()->id(),
                ]);
            });

            MosqueAuditUtil::log($businessId, 'void', 'net_to_bank_transfer', (int) $transfer->id, [
                'amount' => (float) $transfer->amount,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    /**
     * Permanently delete a transfer record (and its wallet ledger rows).
     * This is destructive and removes the transfer from history.
     */
    public function destroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();
            $transfer = MosqueNetBankTransfer::query()->where('business_id', $businessId)->find($id);
            if (! $transfer) {
                MosqueBankWalletTransaction::query()
                    ->where('business_id', $businessId)
                    ->where('ref_id', (int) $id)
                    ->whereIn('ref_module', ['net_to_bank', 'net_to_bank_void'])
                    ->delete();

                return ['success' => true, 'msg' => __('lang_v1.success')];
            }

            DB::transaction(function () use ($businessId, $transfer) {
                MosqueBankWalletTransaction::query()
                    ->where('business_id', $businessId)
                    ->where('ref_id', (int) $transfer->id)
                    ->whereIn('ref_module', ['net_to_bank', 'net_to_bank_void'])
                    ->delete();

                $transfer->delete();
            });

            MosqueAuditUtil::log($businessId, 'delete', 'net_to_bank_transfer', (int) $id, [
                'amount' => (float) ($transfer->amount ?? 0),
                'bank_account_id' => (int) ($transfer->bank_account_id ?? 0),
                'status' => (string) ($transfer->status ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
