<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqPledge;
use Modules\Mosque\Entities\MosqPledgeFulfillment;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Services\FinancePostingService;
use Modules\Mosque\Services\PledgeStatusService;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;

class PledgeFulfillmentController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function settings(int $businessId): array
    {
        if (! Schema::hasTable('mosque_settings')) {
            return [];
        }
        $row = MosqueSetting::query()->where('business_id', $businessId)->first();
        return $row?->settings ?: [];
    }

    private function ensureEnabled(int $businessId): void
    {
        $settings = $this->settings($businessId);
        if (array_key_exists('pledges_enabled', $settings) && ! (bool) $settings['pledges_enabled']) {
            abort(403, 'Pledges is disabled.');
        }
    }

    private function ensurePermission(string $perm): void
    {
        if (! auth()->user()->can($perm)) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function form($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.fulfill');

        $pledge = MosqPledge::query()->where('business_id', $businessId)->findOrFail($id);
        $settings = $this->settings($businessId);
        $calc = PledgeStatusService::recalcPledge($businessId, (int) $pledge->id, $settings);

        return view('mosque::pledges.fulfill', compact('pledge', 'calc'));
    }

    public function store(Request $request, $id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.fulfill');

        $request->validate([
            'date' => 'required|date',
            'channel' => 'required|in:cash,bank,card,online,mfs,in_kind',
            'amount_cash' => 'nullable|numeric|min:0',
            'goods_name' => 'nullable|string|max:255',
            'unit' => 'nullable|string|max:50',
            'qty' => 'nullable|numeric|min:0',
            'est_value' => 'nullable|numeric|min:0',
            'remarks' => 'nullable|string',
        ]);

        try {
            $settings = $this->settings($businessId);
            $pledge = MosqPledge::query()->where('business_id', $businessId)->findOrFail($id);

            if ($pledge->type === 'zakat' && $request->input('channel') === 'in_kind') {
                return ['success' => false, 'msg' => 'Zakat pledges are cash only.'];
            }

            $calc = PledgeStatusService::recalcPledge($businessId, (int) $pledge->id, $settings);
            $balance = (float) ($calc['balance'] ?? 0);

            $channel = (string) $request->input('channel');
            $isCashLike = in_array($channel, ['cash', 'bank', 'card', 'online', 'mfs'], true);

            $amountCash = $isCashLike ? (float) $request->input('amount_cash', 0) : 0;
            $estValue = ! $isCashLike ? (float) $request->input('est_value', 0) : 0;

            if ($isCashLike && $amountCash <= 0) {
                return ['success' => false, 'msg' => 'Amount is required.'];
            }
            if (! $isCashLike) {
                if (trim((string) $request->input('goods_name', '')) === '') {
                    return ['success' => false, 'msg' => 'Goods name is required.'];
                }
                if ((float) $request->input('qty', 0) <= 0) {
                    return ['success' => false, 'msg' => 'Quantity is required.'];
                }
            }

            $applyAmount = $isCashLike ? $amountCash : $estValue;
            if ($applyAmount > ($balance + 0.01)) {
                return ['success' => false, 'msg' => 'Over-payment is not allowed.'];
            }

            $fulfillmentId = null;
            DB::transaction(function () use ($businessId, $pledge, $request, $channel, $isCashLike, $amountCash, $estValue, &$fulfillmentId) {
                $fulfillment = MosqPledgeFulfillment::query()->create([
                    'pledge_id' => $pledge->id,
                    'date' => $request->input('date'),
                    'channel' => $channel,
                    'amount_cash' => $isCashLike ? $amountCash : null,
                    'goods_name' => $isCashLike ? null : $request->input('goods_name'),
                    'unit' => $isCashLike ? null : $request->input('unit'),
                    'qty' => $isCashLike ? null : $request->input('qty'),
                    'est_value' => $isCashLike ? null : ($estValue > 0 ? $estValue : null),
                    'remarks' => $request->input('remarks'),
                    'created_by' => auth()->id(),
                ]);

                $fulfillmentId = (int) $fulfillment->id;

                if ($isCashLike) {
                    FinancePostingService::postCashFulfillment(
                        $businessId,
                        (int) $fulfillment->id,
                        (float) $amountCash,
                        (string) $pledge->type,
                        (string) $fulfillment->date,
                        $pledge->location_id ? (int) $pledge->location_id : null,
                        auth()->id(),
                        (string) $pledge->pledge_code
                    );
                }
            });

            PledgeStatusService::recalcPledge($businessId, (int) $pledge->id, $settings);

            MosqueAuditUtil::log($businessId, 'create', 'pledge_fulfillment', (int) $fulfillmentId, [
                'pledge_id' => (int) $pledge->id,
                'channel' => $channel,
                'amount_cash' => $amountCash,
                'est_value' => $estValue,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.edit');

        try {
            $fulfillment = MosqPledgeFulfillment::query()
                ->whereHas('pledge', function ($q) use ($businessId) {
                    $q->where('business_id', $businessId);
                })
                ->findOrFail($id);

            $pledgeId = (int) $fulfillment->pledge_id;

            $meta = [
                'pledge_id' => $pledgeId,
                'channel' => (string) $fulfillment->channel,
            ];

            DB::transaction(function () use ($businessId, $fulfillment) {
                $fulfillment->delete();
                DB::table('mosque_finance_entries')
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'pledge')
                    ->where('ref_id', (int) $fulfillment->id)
                    ->delete();
            });

            $settings = $this->settings($businessId);
            PledgeStatusService::recalcPledge($businessId, $pledgeId, $settings);

            MosqueAuditUtil::log($businessId, 'delete', 'pledge_fulfillment', (int) $id, $meta);
            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'pledge fulfillment', (int) $id, $meta);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
