<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqDonationGood;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Yajra\DataTables\Facades\DataTables;

class PledgeGoodsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function settings(int $businessId): array
    {
        if (! Schema::hasTable('mosque_settings')) {
            return [];
        }
        $row = MosqueSetting::query()->where('business_id', $businessId)->first();
        return $row?->settings ?: [];
    }

    private function ensureEnabled(int $businessId): void
    {
        $settings = $this->settings($businessId);
        if (array_key_exists('pledges_enabled', $settings) && ! (bool) $settings['pledges_enabled']) {
            abort(403, 'Pledges is disabled.');
        }
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.pledges.edit')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        return view('mosque::pledges.goods.index');
    }

    public function data()
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        $query = MosqDonationGood::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'unit', 'default_est_value', 'is_active']);

        return DataTables::of($query)
            ->editColumn('default_est_value', function ($row) {
                $val = $row->default_est_value ?? '';
                if ($val === null || $val === '') {
                    return '-';
                }
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$val.'">'.$val.'</span>';
            })
            ->editColumn('is_active', function ($row) {
                return ! empty($row->is_active) ? '<span class="label label-success">Active</span>' : '<span class="label label-default">Inactive</span>';
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.route('mosque.pledges.goods.edit', [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_goods_modal"><i class="fa fa-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.route('mosque.pledges.goods.destroy', [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_good"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['default_est_value', 'is_active', 'action'])
            ->make(true);
    }

    public function create()
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        return view('mosque::pledges.goods.create');
    }

    public function store(Request $request)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'unit' => 'nullable|string|max:50',
            'default_est_value' => 'nullable|numeric|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        try {
            $good = MosqDonationGood::query()->create([
                'business_id' => $businessId,
                'name' => $request->input('name'),
                'unit' => $request->input('unit'),
                'default_est_value' => $request->input('default_est_value'),
                'is_active' => (bool) $request->input('is_active', true),
            ]);

            MosqueAuditUtil::log($businessId, 'create', 'pledge_good', (int) $good->id, ['name' => $good->name]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        $good = MosqDonationGood::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::pledges.goods.edit', compact('good'));
    }

    public function update(Request $request, $id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'unit' => 'nullable|string|max:50',
            'default_est_value' => 'nullable|numeric|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        try {
            $good = MosqDonationGood::query()->where('business_id', $businessId)->findOrFail($id);
            $good->name = $request->input('name');
            $good->unit = $request->input('unit');
            $good->default_est_value = $request->input('default_est_value');
            $good->is_active = (bool) $request->input('is_active', true);
            $good->save();

            MosqueAuditUtil::log($businessId, 'update', 'pledge_good', (int) $good->id, ['name' => $good->name]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        try {
            $good = MosqDonationGood::query()->where('business_id', $businessId)->findOrFail($id);
            $meta = ['name' => (string) $good->name];

            $good->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'pledge_good', (int) $id, $meta);
            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'pledge good', (int) $id, $meta);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}

