<?php

namespace Modules\Mosque\Http\Controllers;

use App\Business;
use App\Contact;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqDonationGood;
use Modules\Mosque\Entities\MosqPledge;
use Modules\Mosque\Entities\MosqPledgeFulfillment;
use Modules\Mosque\Entities\MosqPledgeItem;
use Modules\Mosque\Entities\MosqueMember;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Services\PledgeNumberService;
use Modules\Mosque\Services\PledgeStatusService;
use Modules\Mosque\Services\ReminderService;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class PledgesController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function settings(int $businessId): array
    {
        if (! Schema::hasTable('mosque_settings')) {
            return [];
        }
        $row = MosqueSetting::query()->where('business_id', $businessId)->first();
        return $row?->settings ?: [];
    }

    private function ensureEnabled(int $businessId): void
    {
        $settings = $this->settings($businessId);
        if (array_key_exists('pledges_enabled', $settings) && ! (bool) $settings['pledges_enabled']) {
            abort(403, 'Pledges is disabled.');
        }
    }

    private function ensurePermission(string $perm): void
    {
        if (! auth()->user()->can($perm)) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.view');

        return view('mosque::pledges.index');
    }

    public function data(Request $request)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.view');

        $settings = $this->settings($businessId);
        if (Schema::hasTable('mosq_pledges')) {
            PledgeStatusService::recalcBusiness($businessId, $settings);
        }

        $commitmentDateSelect = Schema::hasColumn('mosq_pledges', 'commitment_date')
            ? DB::raw('COALESCE(p.commitment_date, DATE(p.created_at)) as commitment_date')
            : DB::raw('DATE(p.created_at) as commitment_date');

        $query = DB::table('mosq_pledges as p')
            ->leftJoin('contacts as c', function ($join) use ($businessId) {
                $join->on('c.id', '=', 'p.donor_contact_id')
                    ->where('c.business_id', '=', $businessId);
            })
            ->where('p.business_id', $businessId)
            ->select([
                'p.id',
                'p.pledge_code',
                'p.type',
                $commitmentDateSelect,
                'p.due_date',
                'p.recurrence',
                'p.est_total_value',
                'p.currency',
                'p.status',
                DB::raw("COALESCE(c.name, p.donor_name, '') as donor_name"),
                DB::raw("COALESCE(c.mobile, p.donor_phone, '') as donor_phone"),
                DB::raw("(SELECT COALESCE(SUM(CASE WHEN f.channel IN ('cash','bank','card','online','mfs') THEN f.amount_cash ELSE f.est_value END),0) FROM mosq_pledge_fulfillments f WHERE f.pledge_id = p.id) as fulfilled_value"),
            ]);

        if (! empty($request->input('type'))) {
            $query->where('p.type', $request->input('type'));
        }
        if (! empty($request->input('status'))) {
            $query->where('p.status', $request->input('status'));
        }
        if (! empty($request->input('location_id'))) {
            $query->where('p.location_id', (int) $request->input('location_id'));
        }
        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereBetween('p.due_date', [$request->input('start_date'), $request->input('end_date')]);
        }
        $chip = (string) $request->input('chip', '');
        if ($chip === 'overdue') {
            $query->where('p.status', 'overdue');
        } elseif ($chip === 'due7') {
            $query->whereBetween('p.due_date', [now()->toDateString(), now()->addDays(7)->toDateString()]);
        } elseif ($chip === 'due30') {
            $query->whereBetween('p.due_date', [now()->toDateString(), now()->addDays(30)->toDateString()]);
        }

        $donorSearch = trim((string) $request->input('donor_q', ''));
        if ($donorSearch !== '') {
            $query->where(function ($q) use ($donorSearch) {
                $q->where('p.pledge_code', 'like', '%'.$donorSearch.'%')
                    ->orWhere('p.donor_name', 'like', '%'.$donorSearch.'%')
                    ->orWhere('p.donor_phone', 'like', '%'.$donorSearch.'%')
                    ->orWhere('p.donor_email', 'like', '%'.$donorSearch.'%')
                    ->orWhere('c.name', 'like', '%'.$donorSearch.'%')
                    ->orWhere('c.mobile', 'like', '%'.$donorSearch.'%');
            });
        }

        return DataTables::of($query)
            ->addColumn('balance', function ($row) {
                $est = (float) ($row->est_total_value ?? 0);
                $ful = (float) ($row->fulfilled_value ?? 0);
                $bal = max($est - $ful, 0);
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$bal.'">'.$bal.'</span>';
            })
            ->editColumn('donor_name', function ($row) {
                $name = trim((string) ($row->donor_name ?? ''));
                $phone = trim((string) ($row->donor_phone ?? ''));
                $out = e($name !== '' ? $name : '-');
                if ($phone !== '') {
                    $out .= '<br><small>'.e($phone).'</small>';
                }
                return $out;
            })
            ->editColumn('type', function ($row) {
                return e(strtoupper((string) ($row->type ?? '')));
            })
            ->editColumn('est_total_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->est_total_value.'">'.$row->est_total_value.'</span>';
            })
            ->editColumn('fulfilled_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->fulfilled_value.'">'.$row->fulfilled_value.'</span>';
            })
            ->editColumn('due_date', function ($row) {
                return ! empty($row->due_date) ? e($row->due_date) : '-';
            })
            ->editColumn('commitment_date', function ($row) {
                return ! empty($row->commitment_date) ? e($row->commitment_date) : '-';
            })
            ->editColumn('status', function ($row) {
                $status = (string) ($row->status ?? '');
                $map = [
                    'draft' => 'default',
                    'active' => 'primary',
                    'due_soon' => 'warning',
                    'overdue' => 'danger',
                    'partially_fulfilled' => 'warning',
                    'fulfilled' => 'success',
                    'cancelled' => 'default',
                ];
                $cls = $map[$status] ?? 'default';
                return '<span class="label label-'.$cls.'">'.e(strtoupper(str_replace('_', ' ', $status))).'</span>';
            })
            ->addColumn('action', function ($row) {
                $actions = [];
                if (auth()->user()->can('mosque.pledges.edit')) {
                    $actions[] = '<button data-href="'.route('mosque.pledges.edit', [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_pledge_modal"><i class="fa fa-edit"></i> '.__('messages.edit').'</button>';
                }
                $actions[] = '<a href="'.route('mosque.pledges.receipt', [$row->id]).'" target="_blank" class="btn btn-xs btn-default"><i class="fa fa-print"></i> Receipt</a>';
                if (auth()->user()->can('mosque.pledges.fulfill')) {
                    $actions[] = '<button data-href="'.route('mosque.pledges.fulfill.form', [$row->id]).'" class="btn btn-xs btn-success btn-modal" data-container=".mosque_pledge_fulfill_modal"><i class="fa fa-check"></i> Fulfill</button>';
                }
                if (auth()->user()->can('mosque.pledges.remind')) {
                    $actions[] = '<button data-href="'.route('mosque.pledges.remind', [$row->id]).'" class="btn btn-xs btn-default mosque_pledge_remind"><i class="fa fa-bell"></i> Remind</button>';
                }
                if (auth()->user()->can('mosque.pledges.edit')) {
                    $actions[] = '<button data-href="'.route('mosque.pledges.cancel', [$row->id]).'" class="btn btn-xs btn-warning cancel_mosque_pledge"><i class="fa fa-ban"></i> Cancel</button>';
                }
                if (auth()->user()->can('mosque.pledges.delete')) {
                    $actions[] = '<button data-href="'.route('mosque.pledges.destroy', [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_pledge"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                }
                return implode(' ', $actions);
            })
            ->rawColumns(['donor_name', 'est_total_value', 'fulfilled_value', 'balance', 'status', 'action'])
            ->make(true);
    }

    public function create()
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.create');

        $goods = Schema::hasTable('mosq_donation_goods')
            ? MosqDonationGood::query()->where('business_id', $businessId)->where('is_active', true)->orderBy('name')->get()
            : collect();

        return view('mosque::pledges.create', compact('goods'));
    }

    public function store(Request $request)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.create');

        $request->validate([
            'donor_contact_id' => 'nullable|string|max:32',
            'donor_name' => 'nullable|string|max:255',
            'donor_phone' => 'nullable|string|max:50',
            'donor_email' => 'nullable|email|max:255',
            'type' => 'required|in:zakat,sadaqah,fitrah,general,event,waqf',
            'commitment_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'cash_amount' => 'nullable|numeric|min:0',
            'goods_name' => 'array',
            'goods_name.*' => 'nullable|string|max:255',
            'goods_unit' => 'array',
            'goods_unit.*' => 'nullable|string|max:50',
            'goods_qty' => 'array',
            'goods_qty.*' => 'nullable|numeric|min:0',
            'goods_est_unit_value' => 'array',
            'goods_est_unit_value.*' => 'nullable|numeric|min:0',
        ]);

        try {
            $settings = $this->settings($businessId);
            $currency = (string) (request()->session()->get('business.currency_code') ?: $settings['currency_code'] ?? '');
            $pledgeCode = PledgeNumberService::nextCode($businessId);

            $pledgeId = null;

            $donorContactId = $request->input('donor_contact_id');
            $donorName = (string) $request->input('donor_name');
            $donorPhone = (string) $request->input('donor_phone');
            $donorEmail = (string) $request->input('donor_email');

            if (is_string($donorContactId) && preg_match('/^m-(\d+)$/', $donorContactId, $m)) {
                $memberId = (int) $m[1];
                $member = MosqueMember::query()
                    ->where('business_id', $businessId)
                    ->where('id', $memberId)
                    ->first();
                if (! empty($member)) {
                    $donorName = (string) ($member->name ?? $donorName);
                    $donorPhone = (string) ($member->phone ?? $donorPhone);
                    $donorEmail = (string) ($member->email ?? $donorEmail);
                }
                $donorContactId = null;
            } elseif ($donorContactId !== null && $donorContactId !== '') {
                $donorContactId = (int) $donorContactId;
            } else {
                $donorContactId = null;
            }

            DB::transaction(function () use ($businessId, $request, $currency, $pledgeCode, $settings, $donorContactId, $donorName, $donorPhone, $donorEmail, &$pledgeId) {
                $payload = [
                    'business_id' => $businessId,
                    'location_id' => $request->input('location_id') ? (int) $request->input('location_id') : null,
                    'pledge_code' => $pledgeCode,
                    'donor_contact_id' => $donorContactId,
                    'donor_name' => $donorName,
                    'donor_phone' => $donorPhone,
                    'donor_email' => $donorEmail,
                    'type' => $request->input('type'),
                    'due_date' => $request->input('due_date'),
                    'recurrence' => 'oneoff',
                    'est_total_value' => 0,
                    'currency' => $currency ?: null,
                    'status' => 'active',
                    'notes' => $request->input('notes'),
                    'created_by' => auth()->id(),
                ];

                if (Schema::hasColumn('mosq_pledges', 'commitment_date')) {
                    $payload['commitment_date'] = $request->input('commitment_date') ?: now()->toDateString();
                }

                $pledge = MosqPledge::query()->create($payload);

                $pledgeId = (int) $pledge->id;

                $total = 0.0;
                $cashAmount = (float) $request->input('cash_amount', 0);
                if ($cashAmount > 0) {
                    MosqPledgeItem::query()->create([
                        'pledge_id' => $pledge->id,
                        'kind' => 'cash',
                        'goods_id' => null,
                        'goods_name' => null,
                        'unit' => null,
                        'qty' => null,
                        'est_unit_value' => null,
                        'est_line_value' => $cashAmount,
                    ]);
                    $total += $cashAmount;
                }

                $goodsNames = (array) $request->input('goods_name', []);
                $goodsUnits = (array) $request->input('goods_unit', []);
                $goodsQtys = (array) $request->input('goods_qty', []);
                $goodsUnitValues = (array) $request->input('goods_est_unit_value', []);

                for ($i = 0; $i < count($goodsNames); $i++) {
                    $name = trim((string) ($goodsNames[$i] ?? ''));
                    $qty = (float) ($goodsQtys[$i] ?? 0);
                    if ($name === '' || $qty <= 0) {
                        continue;
                    }
                    $unit = trim((string) ($goodsUnits[$i] ?? ''));
                    $unitVal = (float) ($goodsUnitValues[$i] ?? 0);
                    $lineVal = $unitVal > 0 ? ($qty * $unitVal) : 0;

                    MosqPledgeItem::query()->create([
                        'pledge_id' => $pledge->id,
                        'kind' => 'goods',
                        'goods_id' => null,
                        'goods_name' => $name,
                        'unit' => $unit ?: null,
                        'qty' => $qty,
                        'est_unit_value' => $unitVal > 0 ? $unitVal : null,
                        'est_line_value' => $lineVal > 0 ? $lineVal : null,
                    ]);
                    $total += ($lineVal > 0 ? $lineVal : 0);
                }

                if ($total <= 0) {
                    abort(422, 'At least one item is required.');
                }

                $pledge->est_total_value = $total;
                $pledge->save();

                PledgeStatusService::recalcPledge($businessId, (int) $pledge->id, $settings);
            });

            MosqueAuditUtil::log($businessId, 'create', 'pledge', (int) $pledgeId, ['pledge_code' => $pledgeCode]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function edit($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.edit');

        $pledge = MosqPledge::query()->where('business_id', $businessId)->findOrFail($id);
        $items = MosqPledgeItem::query()->where('pledge_id', $pledge->id)->get();

        $goods = Schema::hasTable('mosq_donation_goods')
            ? MosqDonationGood::query()->where('business_id', $businessId)->where('is_active', true)->orderBy('name')->get()
            : collect();

        return view('mosque::pledges.edit', compact('pledge', 'items', 'goods'));
    }

    public function update(Request $request, $id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.edit');

        $request->validate([
            'donor_contact_id' => 'nullable|string|max:32',
            'donor_name' => 'nullable|string|max:255',
            'donor_phone' => 'nullable|string|max:50',
            'donor_email' => 'nullable|email|max:255',
            'type' => 'required|in:zakat,sadaqah,fitrah,general,event,waqf',
            'commitment_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'cash_amount' => 'nullable|numeric|min:0',
            'goods_name' => 'array',
            'goods_name.*' => 'nullable|string|max:255',
            'goods_unit' => 'array',
            'goods_unit.*' => 'nullable|string|max:50',
            'goods_qty' => 'array',
            'goods_qty.*' => 'nullable|numeric|min:0',
            'goods_est_unit_value' => 'array',
            'goods_est_unit_value.*' => 'nullable|numeric|min:0',
        ]);

        try {
            $settings = $this->settings($businessId);
            $pledge = MosqPledge::query()->where('business_id', $businessId)->findOrFail($id);

            $donorContactId = $request->input('donor_contact_id');
            $donorName = (string) $request->input('donor_name');
            $donorPhone = (string) $request->input('donor_phone');
            $donorEmail = (string) $request->input('donor_email');

            if (is_string($donorContactId) && preg_match('/^m-(\d+)$/', $donorContactId, $m)) {
                $memberId = (int) $m[1];
                $member = MosqueMember::query()
                    ->where('business_id', $businessId)
                    ->where('id', $memberId)
                    ->first();
                if (! empty($member)) {
                    $donorName = (string) ($member->name ?? $donorName);
                    $donorPhone = (string) ($member->phone ?? $donorPhone);
                    $donorEmail = (string) ($member->email ?? $donorEmail);
                }
                $donorContactId = null;
            } elseif ($donorContactId !== null && $donorContactId !== '') {
                $donorContactId = (int) $donorContactId;
            } else {
                $donorContactId = null;
            }

            DB::transaction(function () use ($businessId, $request, $pledge, $settings, $donorContactId, $donorName, $donorPhone, $donorEmail) {
                $pledge->donor_contact_id = $donorContactId;
                $pledge->donor_name = $donorName;
                $pledge->donor_phone = $donorPhone;
                $pledge->donor_email = $donorEmail;
                $pledge->type = $request->input('type');
                if (Schema::hasColumn('mosq_pledges', 'commitment_date')) {
                    $pledge->commitment_date = $request->input('commitment_date') ?: ($pledge->commitment_date ?: now()->toDateString());
                }
                $pledge->due_date = $request->input('due_date');
                $pledge->recurrence = 'oneoff';
                $pledge->notes = $request->input('notes');
                $pledge->save();

                MosqPledgeItem::query()->where('pledge_id', $pledge->id)->delete();

                $total = 0.0;
                $cashAmount = (float) $request->input('cash_amount', 0);
                if ($cashAmount > 0) {
                    MosqPledgeItem::query()->create([
                        'pledge_id' => $pledge->id,
                        'kind' => 'cash',
                        'est_line_value' => $cashAmount,
                    ]);
                    $total += $cashAmount;
                }

                $goodsNames = (array) $request->input('goods_name', []);
                $goodsUnits = (array) $request->input('goods_unit', []);
                $goodsQtys = (array) $request->input('goods_qty', []);
                $goodsUnitValues = (array) $request->input('goods_est_unit_value', []);

                for ($i = 0; $i < count($goodsNames); $i++) {
                    $name = trim((string) ($goodsNames[$i] ?? ''));
                    $qty = (float) ($goodsQtys[$i] ?? 0);
                    if ($name === '' || $qty <= 0) {
                        continue;
                    }
                    $unit = trim((string) ($goodsUnits[$i] ?? ''));
                    $unitVal = (float) ($goodsUnitValues[$i] ?? 0);
                    $lineVal = $unitVal > 0 ? ($qty * $unitVal) : 0;

                    MosqPledgeItem::query()->create([
                        'pledge_id' => $pledge->id,
                        'kind' => 'goods',
                        'goods_name' => $name,
                        'unit' => $unit ?: null,
                        'qty' => $qty,
                        'est_unit_value' => $unitVal > 0 ? $unitVal : null,
                        'est_line_value' => $lineVal > 0 ? $lineVal : null,
                    ]);
                    $total += ($lineVal > 0 ? $lineVal : 0);
                }

                if ($total <= 0) {
                    abort(422, 'At least one item is required.');
                }

                $pledge->est_total_value = $total;
                $pledge->save();

                PledgeStatusService::recalcPledge($businessId, (int) $pledge->id, $settings);
            });

            MosqueAuditUtil::log($businessId, 'update', 'pledge', (int) $pledge->id, ['pledge_code' => $pledge->pledge_code]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function destroy($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.delete');

        try {
            $pledge = MosqPledge::query()->where('business_id', $businessId)->findOrFail($id);

            $status = (string) ($pledge->status ?? '');
            if (! in_array($status, ['fulfilled', 'cancelled'], true)) {
                return ['success' => false, 'msg' => 'You can delete only Fulfilled or Cancelled pledges.'];
            }

            $meta = ['pledge_code' => (string) $pledge->pledge_code];

            DB::transaction(function () use ($businessId, $pledge) {
                $fulfillmentIds = MosqPledgeFulfillment::query()
                    ->where('pledge_id', $pledge->id)
                    ->pluck('id')
                    ->map(fn ($x) => (int) $x)
                    ->all();

                if (! empty($fulfillmentIds) && Schema::hasTable('mosque_finance_entries')) {
                    MosqueFinanceEntry::query()
                        ->where('business_id', $businessId)
                        ->where('ref_module', 'pledge')
                        ->whereIn('ref_id', $fulfillmentIds)
                        ->delete();
                }

                MosqPledgeFulfillment::query()->where('pledge_id', $pledge->id)->delete();
                MosqPledgeItem::query()->where('pledge_id', $pledge->id)->delete();
                $pledge->delete();
            });

            MosqueAuditUtil::log($businessId, 'delete', 'pledge', (int) $pledge->id, $meta);
            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'pledge', (int) $pledge->id, $meta);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function cancel($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.edit');

        try {
            $settings = $this->settings($businessId);
            $pledge = MosqPledge::query()->where('business_id', $businessId)->findOrFail($id);

            DB::transaction(function () use ($businessId, $pledge, $settings) {
                $fulfillmentIds = MosqPledgeFulfillment::query()
                    ->where('pledge_id', $pledge->id)
                    ->pluck('id')
                    ->map(fn ($x) => (int) $x)
                    ->all();

                if (! empty($fulfillmentIds) && Schema::hasTable('mosque_finance_entries')) {
                    MosqueFinanceEntry::query()
                        ->where('business_id', $businessId)
                        ->where('ref_module', 'pledge')
                        ->whereIn('ref_id', $fulfillmentIds)
                        ->delete();
                }

                MosqPledgeFulfillment::query()->where('pledge_id', $pledge->id)->delete();

                $pledge->status = 'cancelled';
                $pledge->save();
            });

            MosqueAuditUtil::log($businessId, 'cancel', 'pledge', (int) $pledge->id, ['pledge_code' => $pledge->pledge_code]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function receipt($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.view');

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = $this->settings($businessId);

        $pledge = MosqPledge::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $items = MosqPledgeItem::query()->where('pledge_id', $pledge->id)->get();
        $fulfillments = MosqPledgeFulfillment::query()->where('pledge_id', $pledge->id)->orderBy('date')->get();

        $calc = PledgeStatusService::recalcPledge($businessId, (int) $pledge->id, $settings);

        $status = (string) ($calc['status'] ?? $pledge->status ?? 'draft');
        $statusText = strtoupper(str_replace('_', ' ', $status));
        $statusClass = in_array($status, ['fulfilled'], true) ? 'success' : (in_array($status, ['partially_fulfilled', 'due_soon'], true) ? 'warning' : (in_array($status, ['overdue'], true) ? 'danger' : 'default'));

        $receiptTitle = in_array($status, ['fulfilled'], true) ? 'Pledge Payment Receipt' : 'Pledge Commitment Receipt';

        MosqueAuditUtil::log($businessId, 'print', 'pledge_receipt', (int) $pledge->id, [
            'pledge_code' => $pledge->pledge_code,
            'status' => $status,
        ]);

        return view('mosque::pledges.receipt', compact('profile', 'logoDataUri', 'settings', 'pledge', 'items', 'fulfillments', 'calc', 'statusText', 'statusClass', 'receiptTitle'));
    }

    public function contactsSearch(Request $request)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.view');

        $q = trim((string) $request->input('q', ''));
        $rows = MosqueMember::query()
            ->where('business_id', $businessId)
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($sub) use ($q) {
                    $sub->where('name', 'like', '%'.$q.'%')
                        ->orWhere('phone', 'like', '%'.$q.'%')
                        ->orWhere('email', 'like', '%'.$q.'%');
                });
            })
            ->when($q !== '', function ($query) use ($q) {
                return $query;
            })
            ->orderBy('name')
            ->limit(20)
            ->get(['id', 'name', 'phone', 'email']);

        $results = $rows->map(function ($c) {
            $text = (string) $c->name;
            if (! empty($c->phone)) {
                $text .= ' - '.$c->phone;
            }
            return [
                'id' => 'm-'.$c->id,
                'text' => $text,
                'name' => (string) $c->name,
                'phone' => (string) ($c->phone ?? ''),
                'email' => (string) ($c->email ?? ''),
            ];
        });

        return response()->json(['results' => $results]);
    }

    public function remind($id)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission('mosque.pledges.remind');

        try {
            $settings = $this->settings($businessId);
            $pledge = MosqPledge::query()->where('business_id', $businessId)->findOrFail($id);

            $calc = PledgeStatusService::recalcPledge($businessId, (int) $pledge->id, $settings);

            $payload = [
                'pledge_code' => (string) $pledge->pledge_code,
                'donor_name' => (string) ($pledge->donor_name ?? ''),
                'donor_phone' => (string) ($pledge->donor_phone ?? ''),
                'donor_email' => (string) ($pledge->donor_email ?? ''),
                'type' => (string) $pledge->type,
                'due_date' => (string) ($pledge->due_date ?? ''),
                'balance' => (string) number_format((float) ($calc['balance'] ?? 0), 2, '.', ''),
                'pay_link' => '',
            ];

            $sent = ReminderService::send($businessId, $payload, $settings);

            MosqueAuditUtil::log($businessId, 'remind', 'pledge', (int) $pledge->id, ['pledge_code' => $pledge->pledge_code]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_link' => $sent['whatsapp_link'] ?? ''];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
