<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Services\PledgeStatusService;
use Yajra\DataTables\Facades\DataTables;

class PledgesReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function settings(int $businessId): array
    {
        if (! Schema::hasTable('mosque_settings')) {
            return [];
        }
        $row = MosqueSetting::query()->where('business_id', $businessId)->first();
        return $row?->settings ?: [];
    }

    private function ensureEnabled(int $businessId): void
    {
        $settings = $this->settings($businessId);
        if (array_key_exists('pledges_enabled', $settings) && ! (bool) $settings['pledges_enabled']) {
            abort(403, 'Pledges is disabled.');
        }
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.pledges.reports')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        return view('mosque::pledges.reports.index');
    }

    public function agingData(Request $request)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        $settings = $this->settings($businessId);
        PledgeStatusService::recalcBusiness($businessId, $settings);

        $today = now()->toDateString();
        $rows = DB::table('mosq_pledges as p')
            ->where('p.business_id', $businessId)
            ->whereNotIn('p.status', ['fulfilled', 'cancelled'])
            ->whereNotNull('p.due_date')
            ->select([
                'p.id',
                'p.pledge_code',
                'p.type',
                'p.due_date',
                'p.est_total_value',
                DB::raw("(SELECT COALESCE(SUM(CASE WHEN f.channel IN ('cash','bank','card','online','mfs') THEN f.amount_cash ELSE f.est_value END),0) FROM mosq_pledge_fulfillments f WHERE f.pledge_id = p.id) as fulfilled_value"),
                DB::raw("DATEDIFF('{$today}', p.due_date) as days_overdue"),
            ]);

        return DataTables::of($rows)
            ->addColumn('balance', function ($row) {
                $est = (float) ($row->est_total_value ?? 0);
                $ful = (float) ($row->fulfilled_value ?? 0);
                return max($est - $ful, 0);
            })
            ->addColumn('bucket', function ($row) {
                $d = (int) ($row->days_overdue ?? 0);
                if ($d <= 0) {
                    return '0-30';
                }
                if ($d <= 30) {
                    return '0-30';
                }
                if ($d <= 60) {
                    return '31-60';
                }
                if ($d <= 90) {
                    return '61-90';
                }
                return '90+';
            })
            ->editColumn('est_total_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->est_total_value.'">'.$row->est_total_value.'</span>';
            })
            ->editColumn('fulfilled_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->fulfilled_value.'">'.$row->fulfilled_value.'</span>';
            })
            ->editColumn('due_date', fn ($row) => e((string) $row->due_date))
            ->rawColumns(['est_total_value', 'fulfilled_value'])
            ->make(true);
    }

    public function donorStatementData(Request $request)
    {
        $businessId = $this->businessId();
        $this->ensureEnabled($businessId);
        $this->ensurePermission();

        $q = trim((string) $request->input('q', ''));

        $rows = DB::table('mosq_pledges as p')
            ->where('p.business_id', $businessId)
            ->select([
                'p.id',
                'p.pledge_code',
                'p.donor_name',
                'p.donor_phone',
                'p.donor_email',
                'p.type',
                'p.due_date',
                'p.est_total_value',
                'p.status',
                DB::raw("(SELECT COALESCE(SUM(CASE WHEN f.channel IN ('cash','bank','card','online','mfs') THEN f.amount_cash ELSE f.est_value END),0) FROM mosq_pledge_fulfillments f WHERE f.pledge_id = p.id) as fulfilled_value"),
            ])
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($sub) use ($q) {
                    $sub->where('p.donor_name', 'like', '%'.$q.'%')
                        ->orWhere('p.donor_phone', 'like', '%'.$q.'%')
                        ->orWhere('p.pledge_code', 'like', '%'.$q.'%');
                });
            });

        return DataTables::of($rows)
            ->addColumn('balance', function ($row) {
                $est = (float) ($row->est_total_value ?? 0);
                $ful = (float) ($row->fulfilled_value ?? 0);
                return max($est - $ful, 0);
            })
            ->editColumn('est_total_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->est_total_value.'">'.$row->est_total_value.'</span>';
            })
            ->editColumn('fulfilled_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->fulfilled_value.'">'.$row->fulfilled_value.'</span>';
            })
            ->rawColumns(['est_total_value', 'fulfilled_value'])
            ->make(true);
    }
}

