<?php

namespace Modules\Mosque\Http\Controllers;

use App\Currency;
use App\Utils\BusinessUtil;
use App\Utils\Util;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueLicense;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;

class SettingsController extends Controller
{
    protected Util $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    private function ensureCanManage(): void
    {
        if (! auth()->user()->hasAnyPermission(['mosque.manage', 'mosque.license.manage'])) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function index()
    {
        $this->ensureCanManage();

        $businessId = request()->session()->get('user.business_id');
        $businessDetails = request()->session()->get('business') ?: [];

        /** @var BusinessUtil $businessUtil */
        $businessUtil = app(BusinessUtil::class);

        $license = MosqueLicense::query()
            ->where('business_id', $businessId)
            ->first();

        $masked = null;
        if (! empty($license)) {
            try {
                $plain = Crypt::decryptString($license->license_encrypted);
                $masked = str_repeat('*', max(strlen($plain) - 4, 0)).substr($plain, -4);
            } catch (\Exception $e) {
                $masked = '********';
            }
        }

        $settingsRow = Schema::hasTable('mosque_settings')
            ? MosqueSetting::query()->firstOrCreate(['business_id' => $businessId], ['settings' => []])
            : null;

        $settings = ! empty($settingsRow) ? ($settingsRow->settings ?: []) : [];

        $profile = Schema::hasTable('mosque_profiles')
            ? MosqueProfile::query()->firstOrCreate(['business_id' => $businessId], ['created_by' => auth()->id()])
            : null;

        $timezones = $businessUtil->allTimeZones();

        $currencies = Currency::query()
            ->select('id', 'country', 'currency', 'code', 'symbol')
            ->orderBy('country')
            ->get();

        $countries = $this->loadCountryList();

        return view('mosque::settings.index', [
            'license' => $license,
            'license_masked' => $masked,
            'settings_row' => $settingsRow,
            'profile' => $profile,
            'currencies' => $currencies,
            'timezones' => $timezones,
            'countries' => $countries,
            'business_timezone' => (string) ($businessDetails['time_zone'] ?? ''),
            'selected_currency_id' => (int) ($settings['currency_id'] ?? 0),
        ]);
    }

    private function loadCountryList(): array
    {
        $path = base_path('resources/plugins/AdminLTE/plugins/input-mask/phone-codes/phone-codes.json');
        $raw = is_file($path) ? (string) file_get_contents($path) : '';

        // Prefer regex extraction so it works even if the JSON contains invalid UTF-8 in other fields.
        try {
            $countries = [];
            if ($raw !== '' && preg_match_all('/"cc"\\s*:\\s*"([A-Za-z]{2})".*?"name_en"\\s*:\\s*"([^"]+)"/s', $raw, $m, PREG_SET_ORDER)) {
                foreach ($m as $match) {
                    $cc = strtoupper((string) ($match[1] ?? ''));
                    $name = trim((string) ($match[2] ?? ''));
                    if ($cc === '' || strlen($cc) !== 2 || $name === '') {
                        continue;
                    }
                    $key = strtolower($name);
                    if (! array_key_exists($key, $countries)) {
                        $countries[$key] = ['cc' => $cc, 'name' => $name];
                    }
                }
            }

            if (! empty($countries)) {
                $list = array_values($countries);
                usort($list, fn ($a, $b) => strcasecmp((string) ($a['name'] ?? ''), (string) ($b['name'] ?? '')));
                return $list;
            }
        } catch (\Exception $e) {
            // Continue to fallback.
        }

        // Fallback: try module-bundled list (best effort).
        $modulePath = base_path('Modules/Mosque/Resources/data_countries.json');
        if (is_file($modulePath)) {
            try {
                $rawModule = (string) file_get_contents($modulePath);
                $decoded = json_decode($rawModule, true);
                if (is_array($decoded) && ! empty($decoded)) {
                    $list = [];
                    foreach ($decoded as $cc => $name) {
                        $cc = strtoupper((string) $cc);
                        $name = is_string($name) ? trim($name) : '';
                        if ($cc !== '' && strlen($cc) === 2 && $name !== '') {
                            $list[] = ['cc' => $cc, 'name' => $name];
                        }
                    }
                    usort($list, fn ($a, $b) => strcasecmp((string) ($a['name'] ?? ''), (string) ($b['name'] ?? '')));
                    return $list;
                }
            } catch (\Exception $e) {
                return [];
            }
        }

        return [];
    }

    public function update()
    {
        $this->ensureCanManage();

        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        request()->validate([
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'qr_attendance_enabled' => 'nullable|boolean',
            'hijri_display_enabled' => 'nullable|boolean',
            'reminders_donations_enabled' => 'nullable|boolean',
            'reminders_membership_enabled' => 'nullable|boolean',
            'pledges_enabled' => 'nullable|boolean',
            'pledges_reminder_windows' => 'nullable|string|max:50',
            'prayer_calc_preference' => 'nullable|string|max:50',
            'prayer_calc_manual_note' => 'nullable|string',
            'template_donation_receipt' => 'nullable|string',
            'template_membership_receipt' => 'nullable|string',
            'whatsapp_default_country_code' => 'nullable|string|max:6',
            'template_whatsapp_message' => 'nullable|string',
            'template_email_message' => 'nullable|string',
            'template_delete_whatsapp_message' => 'nullable|string',
            'template_delete_email_subject' => 'nullable|string|max:255',
            'template_delete_email_message' => 'nullable|string',
            'template_donation_reminder_subject' => 'nullable|string|max:255',
            'template_donation_reminder_email' => 'nullable|string',
            'template_membership_reminder_subject' => 'nullable|string|max:255',
            'template_membership_reminder_email' => 'nullable|string',
            'template_pledge_reminder_whatsapp' => 'nullable|string',
            'template_pledge_reminder_sms' => 'nullable|string',
            'template_pledge_reminder_email_subject' => 'nullable|string|max:255',
            'template_pledge_reminder_email' => 'nullable|string',
            'eu_receipt_wording' => 'nullable|string',
            'donation_pay_mobile_1_name' => 'nullable|string|max:100',
            'donation_pay_mobile_1_number' => 'nullable|string|max:50',
            'donation_pay_mobile_1_note' => 'nullable|string|max:255',
            'donation_pay_mobile_2_name' => 'nullable|string|max:100',
            'donation_pay_mobile_2_number' => 'nullable|string|max:50',
            'donation_pay_mobile_2_note' => 'nullable|string|max:255',
            'donation_pay_mobile_3_name' => 'nullable|string|max:100',
            'donation_pay_mobile_3_number' => 'nullable|string|max:50',
            'donation_pay_mobile_3_note' => 'nullable|string|max:255',
            'donation_pay_bank_1_bank_name' => 'nullable|string|max:120',
            'donation_pay_bank_1_account_name' => 'nullable|string|max:120',
            'donation_pay_bank_1_account_no' => 'nullable|string|max:80',
            'donation_pay_bank_1_branch' => 'nullable|string|max:120',
            'donation_pay_bank_1_routing' => 'nullable|string|max:80',
            'donation_pay_bank_1_note' => 'nullable|string|max:255',
            'donation_pay_bank_2_bank_name' => 'nullable|string|max:120',
            'donation_pay_bank_2_account_name' => 'nullable|string|max:120',
            'donation_pay_bank_2_account_no' => 'nullable|string|max:80',
            'donation_pay_bank_2_branch' => 'nullable|string|max:120',
            'donation_pay_bank_2_routing' => 'nullable|string|max:80',
            'donation_pay_bank_2_note' => 'nullable|string|max:255',
            'donation_pay_bank_3_bank_name' => 'nullable|string|max:120',
            'donation_pay_bank_3_account_name' => 'nullable|string|max:120',
            'donation_pay_bank_3_account_no' => 'nullable|string|max:80',
            'donation_pay_bank_3_branch' => 'nullable|string|max:120',
            'donation_pay_bank_3_routing' => 'nullable|string|max:80',
            'donation_pay_bank_3_note' => 'nullable|string|max:255',
            'donation_pay_cash_label' => 'nullable|string|max:120',
            'donation_pay_cash_note' => 'nullable|string|max:255',
        ]);

        if (! Schema::hasTable('mosque_settings')) {
            return redirect()
                ->back()
                ->with('status', ['success' => 0, 'msg' => __('messages.something_went_wrong')]);
        }

        $row = MosqueSetting::query()->firstOrCreate(['business_id' => $businessId], ['settings' => []]);
        $existing = $row->settings ?: [];

        $input = [
            'currency_id' => (int) request()->input('currency_id', 0),
            'qr_attendance_enabled' => (bool) request()->boolean('qr_attendance_enabled'),
            'hijri_display_enabled' => (bool) request()->boolean('hijri_display_enabled'),
            'reminders_donations_enabled' => (bool) request()->boolean('reminders_donations_enabled'),
            'reminders_membership_enabled' => (bool) request()->boolean('reminders_membership_enabled'),
            'pledges_enabled' => (bool) request()->boolean('pledges_enabled', true),
            'pledges_reminder_windows' => (string) request()->input('pledges_reminder_windows', '7,30'),
            'prayer_calc_preference' => (string) request()->input('prayer_calc_preference', ''),
            'prayer_calc_manual_note' => (string) request()->input('prayer_calc_manual_note', ''),
            'template_donation_receipt' => (string) request()->input('template_donation_receipt', ''),
            'template_membership_receipt' => (string) request()->input('template_membership_receipt', ''),
            'whatsapp_default_country_code' => preg_replace('/\\D+/', '', (string) request()->input('whatsapp_default_country_code', '')),
            'template_whatsapp_message' => (string) request()->input('template_whatsapp_message', ''),
            'template_email_message' => (string) request()->input('template_email_message', ''),
            'template_delete_whatsapp_message' => (string) request()->input('template_delete_whatsapp_message', ''),
            'template_delete_email_subject' => (string) request()->input('template_delete_email_subject', ''),
            'template_delete_email_message' => (string) request()->input('template_delete_email_message', ''),
            'template_donation_reminder_subject' => (string) request()->input('template_donation_reminder_subject', ''),
            'template_donation_reminder_email' => (string) request()->input('template_donation_reminder_email', ''),
            'template_membership_reminder_subject' => (string) request()->input('template_membership_reminder_subject', ''),
            'template_membership_reminder_email' => (string) request()->input('template_membership_reminder_email', ''),
            'template_pledge_reminder_whatsapp' => (string) request()->input('template_pledge_reminder_whatsapp', ''),
            'template_pledge_reminder_sms' => (string) request()->input('template_pledge_reminder_sms', ''),
            'template_pledge_reminder_email_subject' => (string) request()->input('template_pledge_reminder_email_subject', ''),
            'template_pledge_reminder_email' => (string) request()->input('template_pledge_reminder_email', ''),
            'eu_receipt_wording' => (string) request()->input('eu_receipt_wording', ''),
            'donation_payment_info' => [
                'mobile' => [
                    [
                        'name' => (string) request()->input('donation_pay_mobile_1_name', ''),
                        'number' => (string) request()->input('donation_pay_mobile_1_number', ''),
                        'note' => (string) request()->input('donation_pay_mobile_1_note', ''),
                    ],
                    [
                        'name' => (string) request()->input('donation_pay_mobile_2_name', ''),
                        'number' => (string) request()->input('donation_pay_mobile_2_number', ''),
                        'note' => (string) request()->input('donation_pay_mobile_2_note', ''),
                    ],
                    [
                        'name' => (string) request()->input('donation_pay_mobile_3_name', ''),
                        'number' => (string) request()->input('donation_pay_mobile_3_number', ''),
                        'note' => (string) request()->input('donation_pay_mobile_3_note', ''),
                    ],
                ],
                'bank' => [
                    [
                        'bank_name' => (string) request()->input('donation_pay_bank_1_bank_name', ''),
                        'account_name' => (string) request()->input('donation_pay_bank_1_account_name', ''),
                        'account_no' => (string) request()->input('donation_pay_bank_1_account_no', ''),
                        'branch' => (string) request()->input('donation_pay_bank_1_branch', ''),
                        'routing' => (string) request()->input('donation_pay_bank_1_routing', ''),
                        'note' => (string) request()->input('donation_pay_bank_1_note', ''),
                    ],
                    [
                        'bank_name' => (string) request()->input('donation_pay_bank_2_bank_name', ''),
                        'account_name' => (string) request()->input('donation_pay_bank_2_account_name', ''),
                        'account_no' => (string) request()->input('donation_pay_bank_2_account_no', ''),
                        'branch' => (string) request()->input('donation_pay_bank_2_branch', ''),
                        'routing' => (string) request()->input('donation_pay_bank_2_routing', ''),
                        'note' => (string) request()->input('donation_pay_bank_2_note', ''),
                    ],
                    [
                        'bank_name' => (string) request()->input('donation_pay_bank_3_bank_name', ''),
                        'account_name' => (string) request()->input('donation_pay_bank_3_account_name', ''),
                        'account_no' => (string) request()->input('donation_pay_bank_3_account_no', ''),
                        'branch' => (string) request()->input('donation_pay_bank_3_branch', ''),
                        'routing' => (string) request()->input('donation_pay_bank_3_routing', ''),
                        'note' => (string) request()->input('donation_pay_bank_3_note', ''),
                    ],
                ],
                'cash' => [
                    'label' => (string) request()->input('donation_pay_cash_label', ''),
                    'note' => (string) request()->input('donation_pay_cash_note', ''),
                ],
            ],
        ];

        $merged = array_merge($existing, $input);

        $row->update([
            'settings' => $merged,
            'updated_by' => auth()->id(),
        ]);

        MosqueAuditUtil::log($businessId, 'update', 'settings', (int) $row->id, [
            'keys' => array_keys($input),
        ]);

        return redirect()
            ->back()
            ->with('status', ['success' => 1, 'msg' => __('lang_v1.updated_success')]);
    }

    private function uploadProfileLogo(?MosqueProfile $profile): ?string
    {
        if (! request()->hasFile('logo')) {
            return null;
        }

        $file = request()->file('logo');
        if (empty($file) || ! $file->isValid()) {
            throw new \Exception('Invalid logo file');
        }

        $ext = strtolower((string) ($file->getClientOriginalExtension() ?: 'png'));
        if (! in_array($ext, ['jpg', 'jpeg', 'png', 'webp', 'gif'], true)) {
            throw new \Exception('Invalid logo file extension');
        }

        $filename = now()->format('YmdHis').'_'.Str::random(16).'.'.$ext;
        $relativePath = 'mosque_logos/'.$filename;

        $defaultDisk = (string) config('filesystems.default');
        $disks = (array) config('filesystems.disks', []);
        $targetDisk = array_key_exists('public', $disks) ? 'public' : $defaultDisk;

        $stored = $file->storeAs('mosque_logos', $filename, $targetDisk);
        if (empty($stored)) {
            throw new \Exception('Logo upload failed');
        }

        $old = ! empty($profile?->logo) ? basename((string) $profile->logo) : null;
        if (! empty($old) && $old !== $filename) {
            $oldRelative = 'mosque_logos/'.$old;
            foreach (array_values(array_unique([$targetDisk, 'public', $defaultDisk])) as $disk) {
                try {
                    Storage::disk($disk)->delete($oldRelative);
                } catch (\Throwable $e) {
                    // ignore
                }
            }
        }

        return basename($relativePath);
    }

    public function updateProfile()
    {
        $this->ensureCanManage();

        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        if (! Schema::hasTable('mosque_profiles')) {
            return redirect()
                ->back()
                ->with('status', ['success' => 0, 'msg' => __('messages.something_went_wrong')]);
        }

        request()->validate([
            'name' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'reg_no' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:255',
            'timezone' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'header_text' => 'nullable|string',
            'logo' => 'nullable|file|mimes:jpeg,jpg,png,webp,gif|max:4096',
        ]);

        try {
            $profile = MosqueProfile::query()->firstOrCreate(
                ['business_id' => $businessId],
                ['created_by' => auth()->id()]
            );

            $update = [
                'name' => request()->input('name'),
                'address' => request()->input('address'),
                'reg_no' => request()->input('reg_no'),
                'country' => (function (): string {
                    $value = trim((string) request()->input('country'));
                    if ($value !== '' && preg_match('/^[A-Za-z]{2}$/', $value)) {
                        return strtoupper($value);
                    }
                    return $value;
                })(),
                'timezone' => request()->input('timezone'),
                'phone' => request()->input('phone'),
                'email' => request()->input('email'),
                'header_text' => request()->input('header_text'),
                'updated_by' => auth()->id(),
            ];

            $logoName = $this->uploadProfileLogo($profile);
            if (! empty($logoName)) {
                $update['logo'] = $logoName;
            } elseif (request()->hasFile('logo')) {
                return redirect()
                    ->back()
                    ->with('status', ['success' => 0, 'msg' => __('messages.something_went_wrong')]);
            }

            $profile->update($update);

            MosqueAuditUtil::log($businessId, 'update', 'profile', (int) $profile->id, [
                'has_logo' => ! empty($update['logo']) ? 1 : 0,
            ]);

            return redirect()
                ->back()
                ->with('status', ['success' => 1, 'msg' => __('lang_v1.updated_success')]);
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return redirect()
                ->back()
                ->with('status', ['success' => 0, 'msg' => __('messages.something_went_wrong')]);
        }
    }

    public function deactivate()
    {
        $this->ensureCanManage();

        $businessId = request()->session()->get('user.business_id');

        MosqueLicense::query()
            ->where('business_id', $businessId)
            ->delete();

        MosqueAuditUtil::log((int) $businessId, 'deactivate', 'license', null, ['by' => auth()->id()]);

        return redirect()
            ->action([\App\Http\Controllers\HomeController::class, 'index'])
            ->with('status', ['success' => 1, 'msg' => __('mosque::mosque.deactivated')]);
    }
}
