<?php

namespace Modules\Mosque\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosquePayslip;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueStaff;
use Modules\Mosque\Entities\MosqueStaffAdvance;
use Modules\Mosque\Entities\MosqueStaffAdvanceAdjustment;
use Modules\Mosque\Entities\MosqueStaffAttendance;
use Modules\Mosque\Entities\MosqueLeaveType;
use Modules\Mosque\Entities\MosqueStaffLeave;
use Modules\Mosque\Entities\MosqueStaffRole;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDeleteNotificationUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;

class StaffController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId) && auth()->check()) {
            $businessId = (int) (auth()->user()->business_id ?? 0);
        }
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.staff.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function normalizeDate(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }
        $value = trim((string) $value);
        if ($value === '') {
            return null;
        }

        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return $value;
        }

        // Mosque leaves use D-M-Y inputs (dd-mm-yyyy) regardless of business setting.
        try {
            return Carbon::createFromFormat('d-m-Y', $value)->format('Y-m-d');
        } catch (\Exception $e) {
        }

        $format = (string) request()->session()->get('business.date_format', 'm/d/Y');
        try {
            return Carbon::createFromFormat($format, $value)->format('Y-m-d');
        } catch (\Exception $e) {
        }

        try {
            return Carbon::parse($value)->format('Y-m-d');
        } catch (\Exception $e) {
        }

        return null;
    }

    public function index()
    {
        $this->ensurePermission();

        $businessId = $this->businessId();
        $leaveTypes = Schema::hasTable('mosque_leave_types')
            ? MosqueLeaveType::query()
                ->where('business_id', $businessId)
                ->where('active', true)
                ->orderBy('name')
                ->pluck('name', 'id')
            : collect();

        return view('mosque::staff.index', compact('leaveTypes'));
    }

    // Staff CRUD
    public function staffData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        try {
            $select = ['id', 'name', 'designation', 'phone', 'email', 'salary', 'status'];
            if (Schema::hasColumn('mosque_staff', 'whatsapp_number')) {
                $select[] = 'whatsapp_number';
            }
            if (Schema::hasColumn('mosque_staff', 'whatsapp_same_as_phone')) {
                $select[] = 'whatsapp_same_as_phone';
            }
            if (Schema::hasColumn('mosque_staff', 'photo_path')) {
                $select[] = 'photo_path';
            }

            $query = MosqueStaff::query()
                ->where('business_id', $businessId)
                ->select($select);

            return DataTables::of($query)
                ->addColumn('photo', function ($row) {
                    $url = $this->resolveStaffPhotoUrl($row->photo_path ?? null);
                    return '<img src="'.$url.'" alt="Photo" style="width:38px;height:38px;border-radius:8px;object-fit:cover;border:1px solid #e6eef7;background:#fff;">';
                })
                ->addColumn('name_with_id', function ($row) {
                    $name = e((string) ($row->name ?? ''));
                    $id = (int) ($row->id ?? 0);
                    $badge = $id > 0 ? '<span class="label label-default" style="margin-left:6px;">#'.$id.'</span>' : '';
                    return '<span>'.$name.$badge.'</span>';
                })
                ->addColumn('name_plain', function ($row) {
                    return (string) ($row->name ?? '');
                })
                ->editColumn('whatsapp_number', function ($row) {
                    $number = '';
                    if (! empty($row->whatsapp_same_as_phone)) {
                        $number = (string) ($row->phone ?? '');
                    } else {
                        $number = (string) ($row->whatsapp_number ?? '');
                    }

                    $number = trim($number);
                    if ($number === '') {
                        return '';
                    }

                    $digits = preg_replace('/\D+/', '', $number);
                    if ($digits === '') {
                        return e($number);
                    }

                    $url = 'https://wa.me/'.$digits;
                    return '<a href="'.$url.'" target="_blank" rel="noopener" class="text-success" title="WhatsApp">'
                        .'<i class="fa fa-whatsapp"></i> '.e($number)
                        .'</a>';
                })
                ->editColumn('salary', function ($row) {
                    return $this->formatCurrencySmart($row->salary);
                })
                ->addColumn('action', function ($row) {
                    $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'staffEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_staff_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                    $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'staffDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_staff"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                    return $edit.' '.$delete;
                })
                ->rawColumns(['action', 'photo', 'name_with_id', 'whatsapp_number'])
                ->make(true);
        } catch (\Throwable $e) {
            try {
                $dir = storage_path('logs');
                if (! File::isDirectory($dir)) {
                    File::makeDirectory($dir, 0755, true);
                }
                File::append(
                    $dir.DIRECTORY_SEPARATOR.'mosque_staff_table.log',
                    '['.now()->toDateTimeString().'] '.$e->getMessage()."\n".$e->getTraceAsString()."\n\n"
                );
            } catch (\Throwable $ignore) {
                // ignore
            }

            return response()->json([
                'success' => false,
                'msg' => __('messages.something_went_wrong'),
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function staffSearch(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $term = trim((string) $request->input('q', $request->input('term', '')));
        $limit = (int) $request->input('limit', 20);
        $limit = max(1, min($limit, 50));

        $query = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'phone', 'email'])
            ->orderBy('name');

        if ($term !== '') {
            $query->where(function ($q) use ($term) {
                $q->where('name', 'like', '%'.$term.'%')
                    ->orWhere('phone', 'like', '%'.$term.'%')
                    ->orWhere('email', 'like', '%'.$term.'%');
            });
        }

        $rows = $query->limit($limit)->get();

        $results = [];
        foreach ($rows as $row) {
            $label = trim((string) $row->name);
            $phone = trim((string) ($row->phone ?? ''));
            if ($phone !== '') {
                $label .= ' ('.$phone.')';
            }
            $results[] = ['id' => (int) $row->id, 'text' => $label];
        }

        return response()->json(['results' => $results]);
    }

    private function formatCurrencySmart($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }

        $amount = (float) $value;
        $rounded = round($amount, 2);
        $precision = (abs($rounded - round($rounded)) < 0.00001) ? 0 : 2;

        $decimal = session('currency.decimal_separator', '.');
        $thousand = session('currency.thousand_separator', ',');
        $symbol = session('currency.symbol', '');
        $placement = session('business.currency_symbol_placement', 'before');

        $formatted = number_format($rounded, $precision, $decimal, $thousand);
        if ($symbol === '') {
            return $formatted;
        }

        return $placement === 'after' ? ($formatted.' '.$symbol) : ($symbol.' '.$formatted);
    }

    public function staffCreate()
    {
        $this->ensurePermission();

        $businessId = $this->businessId();
        $roles = Schema::hasTable('mosque_staff_roles')
            ? MosqueStaffRole::query()
                ->where('business_id', $businessId)
                ->where('active', true)
                ->orderBy('sort_order')
                ->orderBy('name')
                ->pluck('name', 'id')
            : collect();

        $phoneCountries = $this->loadPhoneCountries();

        $defaultCountry = null;
        if (Schema::hasTable('mosque_profiles')) {
            $defaultCountry = \Modules\Mosque\Entities\MosqueProfile::query()
                ->where('business_id', $businessId)
                ->value('country');
        }

        return view('mosque::staff.staff.create', compact('roles', 'phoneCountries', 'defaultCountry'));
    }

    public function staffStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'name' => 'required|string|max:255',
            'designation_role_id' => 'required|integer',
            'designation' => 'nullable|string|max:255',
            'country_iso2' => 'required|string|size:2',
            'dial_code' => 'required|string|max:10',
            'phone_national' => 'required|string|max:30',
            'whatsapp_same_as_phone' => 'nullable|boolean',
            'whatsapp_national' => 'nullable|string|max:30',
            'whatsapp_number' => 'nullable|string|max:30', // backward-compatible
            'photo' => 'nullable|file|mimes:jpeg,jpg,png,webp,gif|max:2048',
            'photo_capture' => 'nullable|string',
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'salary' => 'nullable|numeric|min:0',
            'status' => 'nullable|string|max:50',
        ]);

        try {
            $businessId = $this->businessId();

            $roleId = (int) $request->input('designation_role_id');
            if (! Schema::hasTable('mosque_staff_roles') ||
                ! MosqueStaffRole::query()->where('business_id', $businessId)->where('active', true)->where('id', $roleId)->exists()
            ) {
                return ['success' => false, 'msg' => 'Invalid staff role.'];
            }

            $designation = $this->resolveStaffDesignation($businessId, $roleId, null);

            $countryIso2 = strtoupper((string) $request->input('country_iso2'));
            $dialCode = (string) $request->input('dial_code');
            $phoneNational = (string) $request->input('phone_national');
            $phoneFull = $this->formatE164Like($dialCode, $phoneNational);

            $whatsappSame = (bool) $request->boolean('whatsapp_same_as_phone', true);
            $whatsappNational = (string) $request->input('whatsapp_national', '');
            $whatsappFallback = (string) $request->input('whatsapp_number', ''); // backward-compatible
            $whatsappFull = '';
            if ($whatsappSame) {
                $whatsappFull = $phoneFull;
            } else {
                $val = $whatsappNational !== '' ? $whatsappNational : $whatsappFallback;
                $whatsappFull = $val !== '' ? $this->formatE164Like($dialCode, $val) : '';
            }

            $photoPath = $this->saveStaffPhoto($request);

            MosqueStaff::query()->create(array_merge(
                $request->only(['name', 'email', 'salary', 'status']),
                [
                    'designation' => $designation,
                    'photo_path' => $photoPath,
                    'country_iso2' => $countryIso2,
                    'dial_code' => $dialCode,
                    'phone_national' => $phoneNational,
                    'phone' => $phoneFull,
                    'whatsapp_same_as_phone' => $whatsappSame ? 1 : 0,
                    'whatsapp_number' => $whatsappFull,
                ],
                ['business_id' => $businessId]
            ));
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function staffEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $staff = MosqueStaff::query()->where('business_id', $businessId)->findOrFail($id);

        $roles = Schema::hasTable('mosque_staff_roles')
            ? MosqueStaffRole::query()
                ->where('business_id', $businessId)
                ->where('active', true)
                ->orderBy('sort_order')
                ->orderBy('name')
                ->pluck('name', 'id')
            : collect();

        $phoneCountries = $this->loadPhoneCountries();

        $defaultCountry = null;
        if (Schema::hasTable('mosque_profiles')) {
            $defaultCountry = \Modules\Mosque\Entities\MosqueProfile::query()
                ->where('business_id', $businessId)
                ->value('country');
        }

        $staffCountry = (string) ($staff->country_iso2 ?? '');
        $staffDial = (string) ($staff->dial_code ?? '');
        $staffNational = (string) ($staff->phone_national ?? '');

        if ($staffCountry === '' && is_string($staff->phone) && str_starts_with($staff->phone, '+')) {
            // Best-effort: infer dial code from phone number.
            $parsed = $this->inferDialFromE164($phoneCountries, (string) $staff->phone);
            $staffCountry = $parsed['country_iso2'] ?? '';
            $staffDial = $parsed['dial_code'] ?? '';
            $staffNational = $parsed['national'] ?? '';
        }

        $staffWhatsappSame = (bool) ($staff->whatsapp_same_as_phone ?? true);
        $staffWhatsappNational = '';
        if ($staffWhatsappSame) {
            $staffWhatsappNational = $staffNational;
        } else {
            $wh = (string) ($staff->whatsapp_number ?? '');
            $dialDigits = preg_replace('/\\D+/', '', $staffDial);
            $whDigits = preg_replace('/\\D+/', '', $wh);
            if ($dialDigits !== '' && str_starts_with($whDigits, $dialDigits)) {
                $staffWhatsappNational = substr($whDigits, strlen($dialDigits));
            } else {
                $staffWhatsappNational = $whDigits;
            }
        }

        $staffPhotoUrl = $this->resolveStaffPhotoUrl($staff->photo_path ?? null);

        return view('mosque::staff.staff.edit', compact('staff', 'roles', 'phoneCountries', 'defaultCountry', 'staffCountry', 'staffDial', 'staffNational', 'staffWhatsappSame', 'staffWhatsappNational', 'staffPhotoUrl'));
    }

    private function loadPhoneCountries(): array
    {
        $path = base_path('resources/plugins/AdminLTE/plugins/input-mask/phone-codes/phone-codes.json');
        if (! is_file($path)) {
            return [];
        }

        try {
            $raw = (string) file_get_contents($path);
            $decoded = json_decode($raw, true);
            if (! is_array($decoded)) {
                $clean = @iconv('UTF-8', 'UTF-8//IGNORE', $raw);
                $decoded = is_string($clean) ? json_decode($clean, true) : null;
            }

            if (! is_array($decoded)) {
                return [];
            }

            $countries = [];
            foreach ($decoded as $row) {
                if (! is_array($row)) {
                    continue;
                }
                // Some rows can contain non-string values; avoid warnings (Laravel converts them to exceptions).
                $ccRaw = $row['cc'] ?? '';
                $nameRaw = $row['name_en'] ?? '';
                $maskRaw = $row['mask'] ?? '';

                if (! is_string($ccRaw) || ! is_string($nameRaw) || ! is_string($maskRaw)) {
                    continue;
                }

                $cc = strtoupper(trim($ccRaw));
                $name = trim($nameRaw);
                $mask = trim($maskRaw);

                if ($cc === '' || strlen($cc) !== 2 || $name === '' || $mask === '') {
                    continue;
                }

                $dial = $this->extractDialCodeFromMask($mask);
                if ($dial === '') {
                    continue;
                }

                // Multiple rows can share the same `cc` (e.g. FR for France + territories).
                // Prefer the "main" country by selecting the shortest dial code (then smallest numeric).
                if (! array_key_exists($cc, $countries)) {
                    $countries[$cc] = [
                        'country_iso2' => $cc,
                        'name' => $name,
                        'dial_code' => $dial,
                    ];
                } else {
                    $existingDialDigits = preg_replace('/\\D+/', '', (string) ($countries[$cc]['dial_code'] ?? ''));
                    $newDialDigits = preg_replace('/\\D+/', '', $dial);

                    $existingLen = strlen($existingDialDigits);
                    $newLen = strlen($newDialDigits);

                    $existingNum = $existingDialDigits !== '' ? (int) $existingDialDigits : PHP_INT_MAX;
                    $newNum = $newDialDigits !== '' ? (int) $newDialDigits : PHP_INT_MAX;

                    $shouldReplace = false;
                    if ($newLen > 0 && ($existingLen === 0 || $newLen < $existingLen)) {
                        $shouldReplace = true;
                    } elseif ($newLen > 0 && $newLen === $existingLen && $newNum < $existingNum) {
                        $shouldReplace = true;
                    }

                    if ($shouldReplace) {
                        $countries[$cc] = [
                            'country_iso2' => $cc,
                            'name' => $name,
                            'dial_code' => $dial,
                        ];
                    }
                }
            }

            uasort($countries, function ($a, $b) {
                return strcasecmp((string) ($a['name'] ?? ''), (string) ($b['name'] ?? ''));
            });

            return array_values($countries);
        } catch (\Exception $e) {
            return [];
        }
    }

    private function extractDialCodeFromMask(string $mask): string
    {
        // Examples: "+880-#######", "+1(268)###-####", "+971-5#-###-####"
        if (! str_starts_with($mask, '+')) {
            return '';
        }
        $digits = '';
        for ($i = 1; $i < strlen($mask); $i++) {
            $ch = $mask[$i];
            if (ctype_digit($ch)) {
                $digits .= $ch;
                continue;
            }
            break;
        }
        return $digits !== '' ? '+'.$digits : '';
    }

    private function inferDialFromE164(array $phoneCountries, string $phone): array
    {
        $phone = trim($phone);
        if ($phone === '' || ! str_starts_with($phone, '+')) {
            return [];
        }

        $digits = preg_replace('/\\D+/', '', $phone);
        if ($digits === '') {
            return [];
        }

        $best = null;
        foreach ($phoneCountries as $c) {
            $dial = (string) ($c['dial_code'] ?? '');
            $dialDigits = preg_replace('/\\D+/', '', $dial);
            if ($dialDigits === '') {
                continue;
            }
            if (str_starts_with($digits, $dialDigits)) {
                if ($best === null || strlen($dialDigits) > strlen($best['dialDigits'])) {
                    $best = [
                        'country_iso2' => (string) ($c['country_iso2'] ?? ''),
                        'dial_code' => '+'.$dialDigits,
                        'dialDigits' => $dialDigits,
                    ];
                }
            }
        }

        if ($best === null) {
            return [];
        }

        $national = substr($digits, strlen($best['dialDigits']));

        return [
            'country_iso2' => $best['country_iso2'],
            'dial_code' => $best['dial_code'],
            'national' => $national,
        ];
    }

    public function staffUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'designation_role_id' => 'required|integer',
            'designation' => 'nullable|string|max:255',
            'country_iso2' => 'required|string|size:2',
            'dial_code' => 'required|string|max:10',
            'phone_national' => 'required|string|max:30',
            'whatsapp_same_as_phone' => 'nullable|boolean',
            'whatsapp_national' => 'nullable|string|max:30',
            'whatsapp_number' => 'nullable|string|max:30', // backward-compatible
            'photo' => 'nullable|file|mimes:jpeg,jpg,png,webp,gif|max:2048',
            'photo_capture' => 'nullable|string',
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'salary' => 'nullable|numeric|min:0',
            'status' => 'nullable|string|max:50',
        ]);

        try {
            $businessId = $this->businessId();
            $staff = MosqueStaff::query()->where('business_id', $businessId)->findOrFail($id);

            $roleId = (int) $request->input('designation_role_id');
            if (! Schema::hasTable('mosque_staff_roles') ||
                ! MosqueStaffRole::query()->where('business_id', $businessId)->where('active', true)->where('id', $roleId)->exists()
            ) {
                return ['success' => false, 'msg' => 'Invalid staff role.'];
            }

            $designation = $this->resolveStaffDesignation($businessId, $roleId, null);

            $countryIso2 = strtoupper((string) $request->input('country_iso2'));
            $dialCode = (string) $request->input('dial_code');
            $phoneNational = (string) $request->input('phone_national');
            $phoneFull = $this->formatE164Like($dialCode, $phoneNational);

            $whatsappSame = (bool) $request->boolean('whatsapp_same_as_phone', true);
            $whatsappNational = (string) $request->input('whatsapp_national', '');
            $whatsappFallback = (string) $request->input('whatsapp_number', ''); // backward-compatible
            $whatsappFull = '';
            if ($whatsappSame) {
                $whatsappFull = $phoneFull;
            } else {
                $val = $whatsappNational !== '' ? $whatsappNational : $whatsappFallback;
                $whatsappFull = $val !== '' ? $this->formatE164Like($dialCode, $val) : '';
            }

            $photoPath = $this->saveStaffPhoto($request);

            $update = array_merge(
                $request->only(['name', 'email', 'salary', 'status']),
                [
                    'designation' => $designation,
                    'country_iso2' => $countryIso2,
                    'dial_code' => $dialCode,
                    'phone_national' => $phoneNational,
                    'phone' => $phoneFull,
                    'whatsapp_same_as_phone' => $whatsappSame ? 1 : 0,
                    'whatsapp_number' => $whatsappFull,
                ]
            );

            if (! empty($photoPath)) {
                $update['photo_path'] = $photoPath;
            }

            $staff->update($update);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    private function resolveStaffDesignation(int $businessId, $roleId, $fallbackDesignation): ?string
    {
        $roleId = (int) $roleId;
        if ($roleId > 0 && Schema::hasTable('mosque_staff_roles')) {
            $role = MosqueStaffRole::query()
                ->where('business_id', $businessId)
                ->where('active', true)
                ->find($roleId);

            if (! empty($role)) {
                return trim((string) $role->name);
            }
        }

        $fallbackDesignation = trim((string) $fallbackDesignation);

        return $fallbackDesignation !== '' ? $fallbackDesignation : null;
    }

    private function formatE164Like(string $dialCode, string $national): string
    {
        $dialDigits = preg_replace('/\\D+/', '', $dialCode);
        $natDigits = preg_replace('/\\D+/', '', $national);

        if ($dialDigits === '' || $natDigits === '') {
            return '';
        }

        // Users usually type local numbers with a trunk prefix 0 (e.g. 017...).
        // For E.164-like storage, drop the leading zeros.
        if (strlen($natDigits) > 1 && $natDigits[0] === '0') {
            $natDigits = ltrim($natDigits, '0');
        }

        return '+'.$dialDigits.$natDigits;
    }

    private function saveStaffPhoto(Request $request): ?string
    {
        try {
            $targetDir = public_path('uploads/mosque_staff_photos');
            if (! File::isDirectory($targetDir)) {
                File::makeDirectory($targetDir, 0755, true);
            }

            if ($request->hasFile('photo')) {
                $file = $request->file('photo');
                $ext = strtolower((string) $file->getClientOriginalExtension());
                $name = 'staff_'.now()->format('YmdHis').'_'.Str::random(8).'.'.$ext;
                $file->move($targetDir, $name);
                return $name;
            }

            $capture = (string) $request->input('photo_capture', '');
            if ($capture !== '' && str_starts_with($capture, 'data:image')) {
                $parts = explode(',', $capture, 2);
                if (count($parts) !== 2) {
                    return null;
                }
                $meta = $parts[0];
                $data = base64_decode($parts[1], true);
                if ($data === false) {
                    return null;
                }

                $ext = 'jpg';
                if (str_contains($meta, 'png')) {
                    $ext = 'png';
                } elseif (str_contains($meta, 'webp')) {
                    $ext = 'webp';
                }

                $name = 'staff_'.now()->format('YmdHis').'_'.Str::random(8).'.'.$ext;
                file_put_contents($targetDir.DIRECTORY_SEPARATOR.$name, $data);
                return $name;
            }
        } catch (\Exception $e) {
            return null;
        }

        return null;
    }

    private function resolveStaffPhotoUrl(?string $photoPath): string
    {
        if (empty($photoPath)) {
            return asset('img/default.png');
        }

        $photoPath = trim((string) $photoPath);
        if ($photoPath === '') {
            return asset('img/default.png');
        }

        $publicDir = public_path('uploads/mosque_staff_photos');
        $publicFile = $publicDir.DIRECTORY_SEPARATOR.$photoPath;
        if (File::exists($publicFile)) {
            return asset('uploads/mosque_staff_photos/'.$photoPath);
        }

        // Backward compatibility: previously stored in `storage/app/mosque_staff_photos`.
        $legacyFile = storage_path('app/mosque_staff_photos/'.$photoPath);
        if (File::exists($legacyFile)) {
            if (! File::isDirectory($publicDir)) {
                File::makeDirectory($publicDir, 0755, true);
            }
            try {
                File::copy($legacyFile, $publicFile);
            } catch (\Exception $e) {
                // ignore
            }
            if (File::exists($publicFile)) {
                return asset('uploads/mosque_staff_photos/'.$photoPath);
            }
        }

        return asset('img/default.png');
    }

    // Staff Roles (Designation)
    public function rolesData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        if (! Schema::hasTable('mosque_staff_roles')) {
            abort(404);
        }

        $query = MosqueStaffRole::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'active', 'sort_order']);

        return DataTables::of($query)
            ->editColumn('active', function ($row) {
                return ! empty($row->active) ? __('lang_v1.yes') : __('lang_v1.no');
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.route('mosque.staff.roles.edit', [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_staff_role_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.route('mosque.staff.roles.destroy', [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_staff_role"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function rolesCreate()
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_staff_roles')) {
            abort(404);
        }

        return view('mosque::staff.roles.create');
    }

    public function rolesStore(Request $request)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_staff_roles')) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'active' => 'nullable|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        try {
            $businessId = $this->businessId();

            $role = MosqueStaffRole::query()->create([
                'business_id' => $businessId,
                'name' => trim((string) $request->input('name')),
                'active' => (bool) $request->boolean('active', true),
                'sort_order' => (int) $request->input('sort_order', 0),
            ]);

            MosqueAuditUtil::log($businessId, 'create', 'staff_role', (int) $role->id, [
                'name' => (string) $role->name,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function rolesEdit($id)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_staff_roles')) {
            abort(404);
        }

        $businessId = $this->businessId();

        $role = MosqueStaffRole::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        return view('mosque::staff.roles.edit', compact('role'));
    }

    public function rolesUpdate(Request $request, $id)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_staff_roles')) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'active' => 'nullable|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        try {
            $businessId = $this->businessId();
            $role = MosqueStaffRole::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $role->update([
                'name' => trim((string) $request->input('name')),
                'active' => (bool) $request->boolean('active', true),
                'sort_order' => (int) $request->input('sort_order', 0),
            ]);

            MosqueAuditUtil::log($businessId, 'update', 'staff_role', (int) $role->id, [
                'name' => (string) $role->name,
                'active' => (int) ($role->active ? 1 : 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function rolesDestroy($id)
    {
        $this->ensurePermission();

        if (! Schema::hasTable('mosque_staff_roles')) {
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }

        try {
            $businessId = $this->businessId();
            $role = MosqueStaffRole::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $role->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'staff_role', (int) $role->id, [
                'name' => (string) $role->name,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'staff role', (int) $role->id, [
                'name' => (string) $role->name,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function staffDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $staff = MosqueStaff::query()->where('business_id', $businessId)->findOrFail($id);
            $staff->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'staff', (int) $staff->id, [
                'name' => (string) $staff->name,
                'designation' => (string) ($staff->designation ?? ''),
                'phone' => (string) ($staff->phone ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'staff', (int) $staff->id, [
                'name' => (string) $staff->name,
                'designation' => (string) ($staff->designation ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Attendance
    public function attendanceData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_attendance as a')
            ->join('mosque_staff as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'a.staff_id')
                    ->where('s.business_id', '=', $businessId);
            })
            ->where('a.business_id', $businessId)
            ->select([
                'a.id',
                's.name as staff_name',
                's.designation as designation',
                'a.in_at',
                'a.out_at',
                'a.source',
                DB::raw("CASE WHEN a.out_at IS NULL THEN NULL ELSE TIMESTAMPDIFF(SECOND, a.in_at, a.out_at) END as worked_seconds"),
            ])
            ->orderByDesc('a.in_at');

        if (! empty($request->input('staff_id'))) {
            $query->where('a.staff_id', $request->input('staff_id'));
        }

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('a.in_at', '>=', $request->input('start_date'))
                ->whereDate('a.in_at', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->addColumn('worked_hours', function ($row) {
                if ($row->worked_seconds === null) {
                    return '<span class="label label-warning">Open</span>';
                }

                $seconds = max(0, (int) $row->worked_seconds);
                $hours = intdiv($seconds, 3600);
                $minutes = intdiv($seconds % 3600, 60);
                return sprintf('%02d:%02d', $hours, $minutes);
            })
            ->addColumn('action', function ($row) {
                $edit = '';
                if (empty($row->out_at)) {
                    $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'attendanceEdit'], [$row->id]).'" class="btn btn-xs btn-success btn-modal" data-container=".mosque_attendance_modal"><i class="glyphicon glyphicon-log-out"></i> Clock Out</button> ';
                }
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'attendanceDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_attendance"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.$delete;
            })
            ->rawColumns(['action', 'worked_hours'])
            ->make(true);
    }

    public function attendanceCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $staff = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::staff.attendance.create', compact('staff'));
    }

    public function attendanceStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'staff_id' => 'required|integer',
            'in_at' => 'required|date',
            'out_at' => 'nullable|date',
        ]);

        try {
            $businessId = $this->businessId();

            $inAt = \Carbon\Carbon::parse($request->input('in_at'));
            $outAt = null;
            if (! empty($request->input('out_at'))) {
                $outAt = \Carbon\Carbon::parse($request->input('out_at'));
                if ($outAt->lt($inAt)) {
                    return response()->json([
                        'success' => false,
                        'msg' => 'Out time must be after or equal to In time.',
                    ], 422);
                }
            }

            $staffId = (int) $request->input('staff_id');

            // Prevent duplicate "open" entries: if a manual attendance exists without out_at,
            // a later manual clock-out should update that row instead of inserting a new one.
            $open = MosqueStaffAttendance::query()
                ->where('business_id', $businessId)
                ->where('staff_id', $staffId)
                ->whereNull('out_at')
                ->orderByDesc('in_at')
                ->first();

            if (! empty($open)) {
                if ($outAt === null) {
                    return response()->json([
                        'success' => false,
                        'msg' => 'This staff already has an open attendance. Please clock out first.',
                    ], 422);
                }

                $openIn = \Carbon\Carbon::parse($open->in_at);
                if ($outAt->lt($openIn)) {
                    return response()->json([
                        'success' => false,
                        'msg' => 'Out time must be after or equal to In time.',
                    ], 422);
                }

                $open->out_at = $outAt->toDateTimeString();
                $open->save();

                return ['success' => true, 'msg' => 'Clock out saved successfully.'];
            }

            MosqueStaffAttendance::query()->create([
                'business_id' => $businessId,
                'staff_id' => $staffId,
                'in_at' => $inAt->toDateTimeString(),
                'out_at' => $outAt ? $outAt->toDateTimeString() : null,
                'source' => 'manual',
                'ip' => request()->ip(),
                'geo' => null,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function attendanceEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $attendance = MosqueStaffAttendance::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $staff = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->findOrFail((int) $attendance->staff_id);

        return view('mosque::staff.attendance.edit', compact('attendance', 'staff'));
    }

    public function attendanceUpdate(Request $request, $id)
    {
        $this->ensurePermission();

        $request->validate([
            'out_at' => 'required|date',
        ]);

        try {
            $businessId = $this->businessId();

            $attendance = MosqueStaffAttendance::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            $inAt = \Carbon\Carbon::parse($attendance->in_at);
            $outAt = \Carbon\Carbon::parse($request->input('out_at'));
            if ($outAt->lt($inAt)) {
                return response()->json([
                    'success' => false,
                    'msg' => 'Out time must be after or equal to In time.',
                ], 422);
            }

            $attendance->out_at = $outAt->toDateTimeString();
            $attendance->save();

            return ['success' => true, 'msg' => 'Clock out saved successfully.'];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function attendanceDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $attendance = MosqueStaffAttendance::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);
            $attendance->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'staff_attendance', (int) $attendance->id, [
                'staff_id' => (int) $attendance->staff_id,
                'in_at' => (string) $attendance->in_at,
                'out_at' => (string) ($attendance->out_at ?? ''),
                'source' => (string) ($attendance->source ?? ''),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'attendance', (int) $attendance->id, [
                'staff_id' => (int) $attendance->staff_id,
                'in_at' => (string) $attendance->in_at,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Staff Leaves
    public function leaveTypesData()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = MosqueLeaveType::query()
            ->where('business_id', $businessId)
            ->select(['id', 'name', 'is_paid', 'active']);

        return DataTables::of($query)
            ->editColumn('is_paid', function ($row) {
                return !empty($row->is_paid) ? '<span class="label label-success">Paid</span>' : '<span class="label label-warning">Unpaid</span>';
            })
            ->editColumn('active', function ($row) {
                return !empty($row->active) ? '<span class="label label-success">'.__('lang_v1.active').'</span>' : '<span class="label label-danger">'.__('lang_v1.inactive').'</span>';
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'leaveTypesEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_leave_type_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'leaveTypesDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_staff_role delete_mosque_leave_type"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $edit.' '.$delete;
            })
            ->rawColumns(['is_paid', 'active', 'action'])
            ->make(true);
    }

    public function leaveTypesCreate()
    {
        $this->ensurePermission();
        return view('mosque::staff.leaves.types.create');
    }

    public function leaveTypesStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'is_paid' => 'nullable|boolean',
            'active' => 'nullable|boolean',
        ]);

        try {
            $businessId = $this->businessId();
            MosqueLeaveType::query()->create([
                'business_id' => $businessId,
                'name' => trim((string) $request->input('name')),
                'is_paid' => (bool) $request->boolean('is_paid', true),
                'active' => (bool) $request->boolean('active', true),
                'created_by' => auth()->id(),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function leaveTypesEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $type = MosqueLeaveType::query()->where('business_id', $businessId)->findOrFail($id);
        return view('mosque::staff.leaves.types.edit', compact('type'));
    }

    public function leaveTypesUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'name' => 'required|string|max:255',
            'is_paid' => 'nullable|boolean',
            'active' => 'nullable|boolean',
        ]);

        try {
            $businessId = $this->businessId();
            $type = MosqueLeaveType::query()->where('business_id', $businessId)->findOrFail($id);
            $type->update([
                'name' => trim((string) $request->input('name')),
                'is_paid' => (bool) $request->boolean('is_paid', true),
                'active' => (bool) $request->boolean('active', true),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function leaveTypesDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $type = MosqueLeaveType::query()->where('business_id', $businessId)->findOrFail($id);
            $type->delete();
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function leavesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_staff_leaves as l')
            ->join('mosque_staff as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'l.staff_id')
                    ->where('s.business_id', '=', $businessId);
            })
            ->leftJoin('mosque_leave_types as t', function ($join) use ($businessId) {
                $join->on('t.id', '=', 'l.leave_type_id')
                    ->where('t.business_id', '=', $businessId);
            })
            ->where('l.business_id', $businessId)
            ->whereNull('l.deleted_at')
            ->select([
                'l.id',
                'l.staff_id',
                's.name as staff_name',
                's.designation',
                't.name as type_name',
                'l.start_date',
                'l.end_date',
                'l.status',
                'l.reason',
                DB::raw('(DATEDIFF(l.end_date, l.start_date) + 1) as days'),
            ])
            ->orderByDesc('l.start_date');

        if (!empty($request->input('staff_id'))) {
            $query->where('l.staff_id', (int) $request->input('staff_id'));
        }
        if (!empty($request->input('status')) && in_array($request->input('status'), ['pending', 'approved', 'rejected'], true)) {
            $query->where('l.status', $request->input('status'));
        }
        if (!empty($request->input('leave_type_id'))) {
            $query->where('l.leave_type_id', (int) $request->input('leave_type_id'));
        }
        $startDate = $this->normalizeDate($request->input('start_date'));
        $endDate = $this->normalizeDate($request->input('end_date'));
        if (!empty($startDate) && !empty($endDate)) {
            $query->whereDate('l.start_date', '>=', $startDate)
                ->whereDate('l.end_date', '<=', $endDate);
        }

        return DataTables::of($query)
            ->editColumn('start_date', function ($row) {
                try {
                    return Carbon::parse($row->start_date)->format('d-m-Y');
                } catch (\Exception $e) {
                    return (string) $row->start_date;
                }
            })
            ->editColumn('end_date', function ($row) {
                try {
                    return Carbon::parse($row->end_date)->format('d-m-Y');
                } catch (\Exception $e) {
                    return (string) $row->end_date;
                }
            })
            ->editColumn('status', function ($row) {
                $s = (string) ($row->status ?? 'pending');
                if ($s === 'approved') return '<span class="label label-success">Approved</span>';
                if ($s === 'rejected') return '<span class="label label-danger">Rejected</span>';
                return '<span class="label label-warning">Pending</span>';
            })
            ->addColumn('action', function ($row) {
                $edit = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'leavesEdit'], [$row->id]).'" class="btn btn-xs btn-primary btn-modal" data-container=".mosque_leave_modal"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</button>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'leavesDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_advance delete_mosque_leave"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';

                $approveReject = '';
                if (($row->status ?? 'pending') === 'pending') {
                    $approveReject .= ' <button data-href="'.route('mosque.staff.leaves.approve', [$row->id]).'" class="btn btn-xs btn-success mosque_leave_approve">Approve</button>';
                    $approveReject .= ' <button data-href="'.route('mosque.staff.leaves.reject', [$row->id]).'" class="btn btn-xs btn-warning mosque_leave_reject">Reject</button>';
                }

                return $edit.' '.$delete.$approveReject;
            })
            ->rawColumns(['status', 'action'])
            ->make(true);
    }

    public function leavesCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $staffOptions = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->limit(300)
            ->get(['id', 'name', 'phone'])
            ->mapWithKeys(function ($row) {
                $label = trim((string) $row->name);
                $phone = trim((string) ($row->phone ?? ''));
                if ($phone !== '') {
                    $label .= ' ('.$phone.')';
                }
                return [(int) $row->id => $label];
            });

        $types = Schema::hasTable('mosque_leave_types')
            ? MosqueLeaveType::query()
                ->where('business_id', $businessId)
                ->where('active', true)
                ->orderBy('name')
                ->pluck('name', 'id')
            : collect();

        return view('mosque::staff.leaves.create', compact('types', 'staffOptions'));
    }

    public function leavesStore(Request $request)
    {
        $this->ensurePermission();
        $request->validate([
            'staff_id' => 'required|integer',
            'leave_type_id' => 'nullable|integer',
            'start_date' => 'required',
            'end_date' => 'required',
            'reason' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();
            $staffId = (int) $request->input('staff_id');
            MosqueStaff::query()->where('business_id', $businessId)->findOrFail($staffId);

            $startDate = $this->normalizeDate($request->input('start_date'));
            $endDate = $this->normalizeDate($request->input('end_date'));
            if (empty($startDate) || empty($endDate) || Carbon::parse($endDate)->lt(Carbon::parse($startDate))) {
                return response()->json(['success' => false, 'msg' => __('messages.invalid_date')], 422);
            }

            $typeId = (int) $request->input('leave_type_id', 0);
            if ($typeId > 0) {
                MosqueLeaveType::query()->where('business_id', $businessId)->where('active', true)->findOrFail($typeId);
            } else {
                $typeId = null;
            }

            MosqueStaffLeave::query()->create([
                'business_id' => $businessId,
                'staff_id' => $staffId,
                'leave_type_id' => $typeId,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'status' => 'pending',
                'reason' => $request->input('reason'),
                'created_by' => auth()->id(),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function leavesEdit($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $leave = MosqueStaffLeave::query()->where('business_id', $businessId)->findOrFail($id);
        $staff = MosqueStaff::query()->where('business_id', $businessId)->findOrFail((int) $leave->staff_id);

        $types = Schema::hasTable('mosque_leave_types')
            ? MosqueLeaveType::query()
                ->where('business_id', $businessId)
                ->where('active', true)
                ->orderBy('name')
                ->pluck('name', 'id')
            : collect();

        return view('mosque::staff.leaves.edit', compact('leave', 'types', 'staff'));
    }

    public function leavesUpdate(Request $request, $id)
    {
        $this->ensurePermission();
        $request->validate([
            'leave_type_id' => 'nullable|integer',
            'start_date' => 'required',
            'end_date' => 'required',
            'reason' => 'nullable|string',
            'status' => 'required|string|in:pending,approved,rejected',
        ]);

        try {
            $businessId = $this->businessId();
            $leave = MosqueStaffLeave::query()->where('business_id', $businessId)->findOrFail($id);

            $startDate = $this->normalizeDate($request->input('start_date'));
            $endDate = $this->normalizeDate($request->input('end_date'));
            if (empty($startDate) || empty($endDate) || Carbon::parse($endDate)->lt(Carbon::parse($startDate))) {
                return response()->json(['success' => false, 'msg' => __('messages.invalid_date')], 422);
            }

            $typeId = (int) $request->input('leave_type_id', 0);
            if ($typeId > 0) {
                MosqueLeaveType::query()->where('business_id', $businessId)->where('active', true)->findOrFail($typeId);
            } else {
                $typeId = null;
            }

            $status = (string) $request->input('status');
            $update = [
                'leave_type_id' => $typeId,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'reason' => $request->input('reason'),
                'status' => $status,
            ];

            if (in_array($status, ['approved', 'rejected'], true)) {
                $update['approved_by'] = auth()->id();
                $update['approved_at'] = now();
            }

            $leave->update($update);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function leavesApprove($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $leave = MosqueStaffLeave::query()->where('business_id', $businessId)->findOrFail($id);
            $leave->update([
                'status' => 'approved',
                'approved_by' => auth()->id(),
                'approved_at' => now(),
            ]);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function leavesReject($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $leave = MosqueStaffLeave::query()->where('business_id', $businessId)->findOrFail($id);
            $leave->update([
                'status' => 'rejected',
                'approved_by' => auth()->id(),
                'approved_at' => now(),
            ]);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function leavesDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $leave = MosqueStaffLeave::query()->where('business_id', $businessId)->findOrFail($id);
            $leave->delete();
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Staff Advances (Advance salary payments)
    public function advancesData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_staff_advances as a')
            ->join('mosque_staff as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'a.staff_id')
                    ->where('s.business_id', '=', $businessId);
            })
            ->leftJoin('mosque_staff_advance_adjustments as aa', function ($join) use ($businessId) {
                $join->on('aa.advance_id', '=', 'a.id')
                    ->where('aa.business_id', '=', $businessId);
            })
            ->where('a.business_id', $businessId)
            ->whereNull('a.deleted_at')
            ->select([
                'a.id',
                'a.staff_id',
                's.name as staff_name',
                's.designation',
                'a.paid_on',
                'a.amount',
                'a.adjusted_amount',
                'a.payment_method',
                'a.payment_ref_no',
                DB::raw('COALESCE(SUM(aa.amount), 0) as adjusted_sum'),
                DB::raw("GROUP_CONCAT(DISTINCT aa.payslip_id ORDER BY aa.payslip_id SEPARATOR ',') as payslip_ids"),
            ])
            ->groupBy([
                'a.id',
                'a.staff_id',
                's.name',
                's.designation',
                'a.paid_on',
                'a.amount',
                'a.adjusted_amount',
                'a.payment_method',
                'a.payment_ref_no',
            ])
            ->orderByDesc('a.paid_on');

        if (! empty($request->input('staff_id'))) {
            $query->where('a.staff_id', $request->input('staff_id'));
        }

        if (! empty($request->input('start_date'))) {
            $query->whereDate('a.paid_on', '>=', $request->input('start_date'));
        }
        if (! empty($request->input('end_date'))) {
            $query->whereDate('a.paid_on', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->editColumn('paid_on', function ($row) {
                try {
                    return \Carbon\Carbon::parse($row->paid_on)->format('Y-m-d');
                } catch (\Exception $e) {
                    return (string) $row->paid_on;
                }
            })
            ->editColumn('amount', fn ($row) => $this->formatCurrencySmart($row->amount))
            ->editColumn('adjusted_amount', function ($row) {
                $val = isset($row->adjusted_sum) ? (float) $row->adjusted_sum : (float) $row->adjusted_amount;
                return $this->formatCurrencySmart($val);
            })
            ->addColumn('balance', function ($row) {
                $adjusted = isset($row->adjusted_sum) ? (float) $row->adjusted_sum : (float) $row->adjusted_amount;
                $bal = (float) $row->amount - $adjusted;
                return $this->formatCurrencySmart(max($bal, 0));
            })
            ->addColumn('adjusted_in', function ($row) {
                $ids = [];
                $raw = (string) ($row->payslip_ids ?? '');
                if ($raw !== '') {
                    $ids = array_values(array_filter(array_map('trim', explode(',', $raw))));
                }
                if (empty($ids)) {
                    return '<span style="color:#64748b;">-</span>';
                }

                $out = [];
                foreach ($ids as $pid) {
                    $pid = (int) $pid;
                    if ($pid <= 0) {
                        continue;
                    }
                    $printUrl = action([\Modules\Mosque\Http\Controllers\StaffController::class, 'payslipsPrint'], [$pid]);
                    $pdfUrl = action([\Modules\Mosque\Http\Controllers\StaffController::class, 'payslipsPdf'], [$pid]);
                    $out[] =
                        '<span class="btn-group" style="margin-right:4px;">'
                        .'<a target="_blank" href="'.$printUrl.'" class="btn btn-xs btn-default" title="Print payslip #'.$pid.'"><i class="fa fa-print"></i></a>'
                        .'<a href="'.$pdfUrl.'" class="btn btn-xs btn-default" title="Download PDF #'.$pid.'"><i class="fa fa-file-pdf"></i></a>'
                        .'</span>';
                }

                return ! empty($out) ? implode('', $out) : '<span style="color:#64748b;">-</span>';
            })
            ->addColumn('action', function ($row) {
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'advancesDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_advance"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                return $delete;
            })
            ->rawColumns(['action', 'adjusted_in'])
            ->make(true);
    }

    public function advancesCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $staff = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('mosque::staff.advances.create', compact('staff'));
    }

    public function advancesStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'staff_id' => 'required|integer',
            'paid_on' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'nullable|string|max:50',
            'payment_ref_no' => 'nullable|string|max:255',
            'note' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();

            MosqueStaffAdvance::query()->create([
                'business_id' => $businessId,
                'staff_id' => (int) $request->input('staff_id'),
                'paid_on' => $request->input('paid_on'),
                'amount' => (float) $request->input('amount'),
                'adjusted_amount' => 0,
                'payment_method' => $request->input('payment_method'),
                'payment_ref_no' => $request->input('payment_ref_no'),
                'note' => $request->input('note'),
                'created_by' => auth()->id(),
                'updated_by' => auth()->id(),
            ]);

            MosqueAuditUtil::log($businessId, 'create', 'staff_advance', null, [
                'staff_id' => (int) $request->input('staff_id'),
                'amount' => (float) $request->input('amount'),
                'paid_on' => (string) $request->input('paid_on'),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function advancesDestroy($id)
    {
        $this->ensurePermission();

        try {
            $businessId = $this->businessId();

            $advance = MosqueStaffAdvance::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            if ((float) ($advance->adjusted_amount ?? 0) > 0) {
                return ['success' => false, 'msg' => 'This advance is already adjusted in payroll. Delete is not allowed.'];
            }

            $advance->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'staff_advance', (int) $id, [
                'staff_id' => (int) $advance->staff_id,
                'amount' => (float) ($advance->amount ?? 0),
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'staff_advance', (int) $id, [
                'staff_id' => (int) $advance->staff_id,
                'amount' => (float) ($advance->amount ?? 0),
                'paid_on' => (string) ($advance->paid_on ?? ''),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    // Payslips
    public function payslipsData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_payslips as p')
            ->join('mosque_staff as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'p.staff_id')
                    ->where('s.business_id', '=', $businessId);
            })
            ->where('p.business_id', $businessId)
            ->whereNull('p.deleted_at')
            ->select([
                'p.id',
                's.name as staff_name',
                'p.period_ym',
                'p.gross',
                'p.net',
                'p.paid_on',
            ])
            ->orderByDesc('p.period_ym');

        if (! empty($request->input('staff_id'))) {
            $query->where('p.staff_id', $request->input('staff_id'));
        }
        if (! empty($request->input('period_ym'))) {
            $query->where('p.period_ym', $request->input('period_ym'));
        }

        return DataTables::of($query)
            ->editColumn('gross', function ($row) {
                return $this->formatCurrencySmart($row->gross);
            })
            ->editColumn('net', function ($row) {
                return $this->formatCurrencySmart($row->net);
            })
            ->addColumn('status', function ($row) {
                $paid = ! empty($row->paid_on);
                $label = $paid ? 'PAID' : 'DUE';
                $bg = $paid ? '#16a34a' : '#dc2626';
                return '<span style="display:inline-block;min-width:54px;text-align:center;padding:3px 10px;border-radius:999px;background:'.$bg.';color:#fff;font-weight:700;font-size:11px;">'.$label.'</span>';
            })
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'payslipsPrint'], [$row->id]).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $delete = '<button data-href="'.action([\Modules\Mosque\Http\Controllers\StaffController::class, 'payslipsDestroy'], [$row->id]).'" class="btn btn-xs btn-danger delete_mosque_payslip"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';

                $pay = '';
                if (empty($row->paid_on)) {
                    $pay = ' <button data-href="'.route('mosque.staff.payslips.pay', [$row->id]).'" class="btn btn-xs btn-success btn-modal" data-container=".mosque_payslip_modal"><i class="fa fa-check"></i> Pay</button>';
                }

                return $print.$pay.' '.$delete;
            })
            ->rawColumns(['action', 'status'])
            ->make(true);
    }

    public function payslipsPay($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $payslip = MosquePayslip::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        if (! empty($payslip->paid_on)) {
            abort(403, 'Payslip already paid.');
        }

        $staff = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->findOrFail($payslip->staff_id);

        $paymentMethods = [
            'cash' => 'Cash',
            'bank' => 'Bank',
            'card' => 'Card',
            'online' => 'Online',
            'other' => 'Other',
        ];

        return view('mosque::staff.payslips.pay', compact('payslip', 'staff', 'paymentMethods'));
    }

    public function payslipsPayStore(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $request->validate([
            'paid_on' => 'required|string',
            'payment_method' => 'required|string|in:cash,bank,card,online,other',
            'paid_amount' => 'nullable|numeric|min:0',
            'payment_ref_no' => 'nullable|string|max:255',
            'payment_note' => 'nullable|string|max:1000',
        ]);

        try {
            $payslip = MosquePayslip::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            if (! empty($payslip->paid_on)) {
                return ['success' => false, 'msg' => 'Payslip already paid.'];
            }

            $paidOn = $this->normalizeDate((string) $request->input('paid_on'));
            if (empty($paidOn)) {
                return response()->json(['success' => false, 'msg' => __('messages.invalid_date')], 422);
            }

            $paidAmount = (float) $request->input('paid_amount', 0);
            if ($paidAmount <= 0) {
                $paidAmount = (float) ($payslip->net ?? 0);
            }

            DB::transaction(function () use ($businessId, $payslip, $request, $paidOn, $paidAmount) {
                $payslip->update([
                    'paid_on' => $paidOn,
                    'payment_method' => (string) $request->input('payment_method'),
                    'paid_amount' => $paidAmount,
                    'payment_ref_no' => $request->input('payment_ref_no'),
                    'payment_note' => $request->input('payment_note'),
                ]);

                // Replace any existing finance entry for this payslip (idempotent).
                MosqueFinanceEntry::query()
                    ->where('business_id', $businessId)
                    ->where('ref_module', 'payslip')
                    ->where('ref_id', (int) $payslip->id)
                    ->delete();

                $category = MosqueFinanceCategory::query()->firstOrCreate(
                    ['business_id' => $businessId, 'type' => 'expense', 'name' => 'Salary'],
                    ['active' => true, 'sort_order' => 1]
                );

                if ($paidAmount > 0) {
                    MosqueFinanceEntry::query()->create([
                        'business_id' => $businessId,
                        'location_id' => null,
                        'type' => 'expense',
                        'category_id' => $category->id,
                        'amount' => (float) $paidAmount,
                        'entry_date' => $paidOn,
                        'ref_module' => 'payslip',
                        'ref_id' => (int) $payslip->id,
                        'fund_tag' => null,
                        'note' => 'Payslip: '.$payslip->period_ym.' ('.$payslip->payment_method.')',
                        'created_by' => auth()->id(),
                    ]);
                }

                MosqueAuditUtil::log($businessId, 'update', 'payslip', (int) $payslip->id, [
                    'staff_id' => (int) $payslip->staff_id,
                    'period_ym' => $payslip->period_ym,
                    'paid_on' => (string) $paidOn,
                    'paid_amount' => (float) $paidAmount,
                ]);
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function payslipsCreate()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $staffRows = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->orderBy('name')
            ->get(['id', 'name', 'designation', 'salary']);

            $advanceBalances = [];
            if (Schema::hasTable('mosque_staff_advances')) {
                $advanceBalances = DB::table('mosque_staff_advances')
                    ->where('business_id', $businessId)
                    ->whereNull('deleted_at')
                    ->whereRaw('amount > adjusted_amount')
                    ->groupBy('staff_id')
                    ->select(['staff_id'])
                    ->selectRaw('SUM(amount - adjusted_amount) as balance')
                    ->pluck('balance', 'staff_id')
                    ->toArray();
            }

        $staff = $staffRows->pluck('name', 'id');
        $staffDetails = $staffRows->keyBy('id')->map(function ($r) {
            return [
                'designation' => (string) ($r->designation ?? ''),
                'salary' => (float) ($r->salary ?? 0),
                'advance_balance' => 0,
            ];
        })->toArray();

        foreach ($staffDetails as $sid => $d) {
            $staffDetails[$sid]['advance_balance'] = (float) ($advanceBalances[$sid] ?? 0);
        }

        return view('mosque::staff.payslips.create', compact('staff', 'staffDetails'));
    }

    public function payslipsStore(Request $request)
    {
        $this->ensurePermission();

        $request->validate([
            'staff_id' => 'required|integer',
            'period_ym' => 'required|date_format:Y-m',
            'gross' => 'required|numeric|min:0',
            'bonus_amount' => 'nullable|numeric|min:0',
            'bonus_reason' => 'nullable|string|max:255',
            'deduction_amount' => 'nullable|numeric|min:0',
            'net' => 'nullable|numeric|min:0',
            'apply_advances' => 'nullable|boolean',
            'paid_on' => 'nullable|date',
            'paid_amount' => 'nullable|numeric|min:0',
            'payment_method' => 'nullable|string|max:50',
            'payment_ref_no' => 'nullable|string|max:255',
            'payment_note' => 'nullable|string',
            'note' => 'nullable|string',
        ]);

        try {
            $businessId = $this->businessId();

            $gross = (float) $request->input('gross', 0);
            $bonus = (float) $request->input('bonus_amount', 0);
            $deduction = (float) $request->input('deduction_amount', 0);

            $bonusLines = [];
            $deductionLines = [];

            // Parse dynamic line items from request (bonus_label[]/bonus_amount_line[]).
            $bonusLabels = $request->input('bonus_label', []);
            $bonusAmounts = $request->input('bonus_amount_line', []);
            if (is_array($bonusLabels) && is_array($bonusAmounts)) {
                foreach ($bonusLabels as $idx => $label) {
                    $label = trim((string) $label);
                    $amt = isset($bonusAmounts[$idx]) ? (float) $bonusAmounts[$idx] : 0;
                    if ($label === '' && $amt <= 0) {
                        continue;
                    }
                    $bonusLines[] = ['label' => $label, 'amount' => max($amt, 0)];
                }
            }
            if (! empty($bonusLines)) {
                $bonus = array_sum(array_map(fn ($x) => (float) ($x['amount'] ?? 0), $bonusLines));
            }

            $dedLabels = $request->input('deduction_label', []);
            $dedAmounts = $request->input('deduction_amount_line', []);
            if (is_array($dedLabels) && is_array($dedAmounts)) {
                foreach ($dedLabels as $idx => $label) {
                    $label = trim((string) $label);
                    $amt = isset($dedAmounts[$idx]) ? (float) $dedAmounts[$idx] : 0;
                    if ($label === '' && $amt <= 0) {
                        continue;
                    }
                    $deductionLines[] = ['label' => $label, 'amount' => max($amt, 0)];
                }
            }
            if (! empty($deductionLines)) {
                $deduction = array_sum(array_map(fn ($x) => (float) ($x['amount'] ?? 0), $deductionLines));
            }

            $net = max($gross + $bonus - $deduction, 0);

            $paidOn = $request->input('paid_on');
            $paymentMethod = (string) $request->input('payment_method', '');
            if (! empty($paidOn) && $paymentMethod === '') {
                return ['success' => false, 'msg' => 'Payment method is required when Paid On is set.'];
            }

            $paidAmount = (float) $request->input('paid_amount', 0);
            if (! empty($paidOn) && $paidAmount <= 0 && $net > 0) {
                $paidAmount = $net;
            }

            $applyAdvances = $request->boolean('apply_advances', true);

            $result = DB::transaction(function () use (
                $businessId,
                $request,
                $gross,
                $bonus,
                $bonusLines,
                $paidOn,
                $paymentMethod,
                $paidAmount,
                $applyAdvances,
                &$deduction,
                &$deductionLines,
                &$net
            ) {
                $staffId = (int) $request->input('staff_id');
                $periodYm = (string) $request->input('period_ym');

                $advanceAdjustments = [];

                if ($applyAdvances && $net > 0 && Schema::hasTable('mosque_staff_advances') && Schema::hasTable('mosque_staff_advance_adjustments')) {
                    try {
                        $periodEnd = \Carbon\Carbon::createFromFormat('Y-m', $periodYm)->endOfMonth()->toDateString();
                        $remaining = $net;

                        $advances = MosqueStaffAdvance::query()
                            ->where('business_id', $businessId)
                            ->where('staff_id', $staffId)
                            ->whereNull('deleted_at')
                            ->whereRaw('amount > adjusted_amount')
                            ->whereDate('paid_on', '<=', $periodEnd)
                            ->orderBy('paid_on')
                            ->lockForUpdate()
                            ->get();

                        foreach ($advances as $adv) {
                            $bal = max(((float) $adv->amount - (float) $adv->adjusted_amount), 0);
                            if ($bal <= 0) {
                                continue;
                            }
                            if ($remaining <= 0) {
                                break;
                            }

                            $use = min($bal, $remaining);
                            if ($use <= 0) {
                                continue;
                            }

                            $remaining -= $use;
                            $label = 'Advance adjustment';
                            try {
                                $label .= ' ('.$adv->paid_on->format('Y-m-d').')';
                            } catch (\Exception $e) {
                                $label .= ' ('.$adv->paid_on.')';
                            }

                            $deductionLines[] = ['label' => $label, 'amount' => (float) $use];
                            $advanceAdjustments[] = [
                                'advance_id' => (int) $adv->id,
                                'amount' => (float) $use,
                                'adjusted_on' => $periodEnd,
                            ];

                            $adv->adjusted_amount = (float) $adv->adjusted_amount + (float) $use;
                            $adv->updated_by = auth()->id();
                            $adv->save();
                        }

                        if (! empty($advanceAdjustments)) {
                            $deduction = array_sum(array_map(fn ($x) => (float) ($x['amount'] ?? 0), $deductionLines));
                            $net = max($gross + $bonus - $deduction, 0);
                        }
                    } catch (\Exception $e) {
                        // If advance adjustment fails, continue without blocking payroll creation.
                    }
                }

                $payslip = MosquePayslip::query()->create(array_merge(
                    $request->only(['staff_id', 'period_ym', 'payment_method', 'payment_ref_no', 'payment_note', 'note']),
                    [
                        'gross' => $gross,
                        'bonus_amount' => $bonus,
                        'bonus_reason' => (string) $request->input('bonus_reason', ''),
                        'bonus_lines' => $bonusLines,
                        'deduction_amount' => $deduction,
                        'deduction_lines' => $deductionLines,
                        'net' => $net,
                        'paid_on' => $paidOn,
                        'paid_amount' => $paidAmount,
                    ],
                    ['business_id' => $businessId]
                ));

                if (! empty($advanceAdjustments)) {
                    foreach ($advanceAdjustments as $adj) {
                        MosqueStaffAdvanceAdjustment::query()->create([
                            'business_id' => $businessId,
                            'advance_id' => (int) $adj['advance_id'],
                            'payslip_id' => (int) $payslip->id,
                            'adjusted_on' => (string) $adj['adjusted_on'],
                            'amount' => (float) $adj['amount'],
                            'created_by' => auth()->id(),
                        ]);
                    }
                }

                if (! empty($payslip->paid_on) && (float) $payslip->paid_amount > 0) {
                    $category = MosqueFinanceCategory::query()->firstOrCreate(
                        ['business_id' => $businessId, 'type' => 'expense', 'name' => 'Salary'],
                        ['active' => true, 'sort_order' => 1]
                    );

                    MosqueFinanceEntry::query()->create([
                        'business_id' => $businessId,
                        'location_id' => null,
                        'type' => 'expense',
                        'category_id' => $category->id,
                        'amount' => (float) $payslip->paid_amount,
                        'entry_date' => $payslip->paid_on,
                        'ref_module' => 'payslip',
                        'ref_id' => $payslip->id,
                        'fund_tag' => null,
                        'note' => 'Payslip: '.$payslip->period_ym.' ('.$payslip->payment_method.')',
                        'created_by' => auth()->id(),
                    ]);
                }

                MosqueAuditUtil::log($businessId, 'create', 'payslip', (int) $payslip->id, [
                    'staff_id' => (int) $payslip->staff_id,
                    'period_ym' => $payslip->period_ym,
                    'net' => (float) $payslip->net,
                    'paid_on' => (string) ($payslip->paid_on ?? ''),
                ]);

                return $payslip;
            });

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function payslipsDestroy($id)
    {
        $this->ensurePermission();
        try {
            $businessId = $this->businessId();
            $payslip = MosquePayslip::query()
                ->where('business_id', $businessId)
                ->findOrFail($id);

            DB::transaction(function () use ($businessId, $payslip) {
                if (Schema::hasTable('mosque_staff_advance_adjustments') && Schema::hasTable('mosque_staff_advances')) {
                    $adjustments = MosqueStaffAdvanceAdjustment::query()
                        ->where('business_id', $businessId)
                        ->where('payslip_id', (int) $payslip->id)
                        ->get();

                    foreach ($adjustments as $adj) {
                        $advance = MosqueStaffAdvance::query()
                            ->where('business_id', $businessId)
                            ->where('id', (int) $adj->advance_id)
                            ->lockForUpdate()
                            ->first();

                        if (! empty($advance)) {
                            $advance->adjusted_amount = max(((float) $advance->adjusted_amount - (float) $adj->amount), 0);
                            $advance->updated_by = auth()->id();
                            $advance->save();
                        }
                    }

                    MosqueStaffAdvanceAdjustment::query()
                        ->where('business_id', $businessId)
                        ->where('payslip_id', (int) $payslip->id)
                        ->delete();
                }

                $payslip->delete();
            });

            MosqueFinanceEntry::query()
                ->where('business_id', $businessId)
                ->where('ref_module', 'payslip')
                ->where('ref_id', $id)
                ->delete();

            MosqueAuditUtil::log($businessId, 'delete', 'payslip', (int) $id, [
                'staff_id' => (int) $payslip->staff_id,
                'period_ym' => $payslip->period_ym,
            ]);

            $notify = MosqueDeleteNotificationUtil::notify($businessId, 'payslip', (int) $id, [
                'staff_id' => (int) $payslip->staff_id,
                'period_ym' => (string) $payslip->period_ym,
                'net' => (float) ($payslip->net ?? 0),
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function payslipsPrint($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $payslip = MosquePayslip::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $staff = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->findOrFail($payslip->staff_id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'print', 'payslip', (int) $payslip->id, [
            'staff_id' => $staff->id,
            'period_ym' => $payslip->period_ym,
        ]);

        $is_pdf = false;
        return view('mosque::staff.payslips.print', compact('payslip', 'staff', 'profile', 'logoDataUri', 'settings', 'is_pdf'));
    }

    public function payslipsPdf($id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $payslip = MosquePayslip::query()
            ->where('business_id', $businessId)
            ->findOrFail($id);

        $staff = MosqueStaff::query()
            ->where('business_id', $businessId)
            ->findOrFail($payslip->staff_id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'pdf', 'payslip', (int) $payslip->id, [
            'staff_id' => $staff->id,
            'period_ym' => $payslip->period_ym,
        ]);

        $is_pdf = true;
        $pdf = Pdf::loadView('mosque::staff.payslips.print', compact('payslip', 'staff', 'profile', 'logoDataUri', 'settings', 'is_pdf'))
            ->setPaper('a4')
            ->setOption('defaultFont', 'DejaVu Sans');

        $filename = 'payslip_'.$staff->id.'_'.$payslip->period_ym.'.pdf';

        return $pdf->download($filename);
    }
}
