<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Yajra\DataTables\Facades\DataTables;

class StaffReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if (empty($businessId)) {
            abort(403, 'Unauthorized action.');
        }

        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (! auth()->user()->can('mosque.staff.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function attendanceSummaryData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_attendance as a')
            ->join('mosque_staff as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'a.staff_id')
                    ->where('s.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_staff', 'deleted_at')) {
                    $join->whereNull('s.deleted_at');
                }
            })
            ->where('a.business_id', $businessId)
            ->select([
                's.id as staff_id',
                's.name as staff_name',
                's.designation',
                DB::raw('COUNT(a.id) as punches_count'),
                DB::raw("COUNT(DISTINCT DATE(a.in_at)) as days_count"),
                DB::raw("SUM(CASE WHEN a.out_at IS NULL THEN 0 ELSE TIMESTAMPDIFF(SECOND, a.in_at, a.out_at) END) as worked_seconds"),
                DB::raw('MAX(a.in_at) as last_in_at'),
            ])
            ->groupBy('s.id', 's.name', 's.designation');

        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('a.in_at', '>=', $request->input('start_date'))
                ->whereDate('a.in_at', '<=', $request->input('end_date'));
        }
        if (! empty($request->input('source')) && in_array($request->input('source'), ['manual', 'qr'], true)) {
            $query->where('a.source', $request->input('source'));
        }

        return DataTables::of($query)
            ->addColumn('worked_hours', function ($row) {
                $seconds = max(0, (int) ($row->worked_seconds ?? 0));
                $hours = intdiv($seconds, 3600);
                $minutes = intdiv($seconds % 3600, 60);
                return sprintf('%02d:%02d', $hours, $minutes);
            })
            ->editColumn('last_in_at', function ($row) {
                return ! empty($row->last_in_at) ? \Carbon\Carbon::parse($row->last_in_at)->format('Y-m-d H:i') : '';
            })
            ->rawColumns(['worked_hours'])
            ->make(true);
    }

    public function salaryRegisterData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        $query = DB::table('mosque_payslips as p')
            ->join('mosque_staff as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'p.staff_id')
                    ->where('s.business_id', '=', $businessId);

                if (Schema::hasColumn('mosque_staff', 'deleted_at')) {
                    $join->whereNull('s.deleted_at');
                }
            })
            ->where('p.business_id', $businessId)
            ->whereNull('p.deleted_at')
            ->select([
                'p.period_ym',
                's.name as staff_name',
                's.designation',
                DB::raw('SUM(p.gross) as gross_total'),
                DB::raw('SUM(p.net) as net_total'),
                DB::raw("SUM(CASE WHEN p.paid_on IS NULL THEN 0 ELSE p.net END) as paid_total"),
                DB::raw("SUM(CASE WHEN p.paid_on IS NULL THEN 0 ELSE 1 END) as paid_count"),
                DB::raw('COUNT(p.id) as payslips_count'),
            ])
            ->groupBy('p.period_ym', 's.name', 's.designation');

        if (! empty($request->input('period_ym'))) {
            $query->where('p.period_ym', $request->input('period_ym'));
        }
        if (! empty($request->input('start_date')) && ! empty($request->input('end_date'))) {
            $query->whereDate('p.created_at', '>=', $request->input('start_date'))
                ->whereDate('p.created_at', '<=', $request->input('end_date'));
        }

        return DataTables::of($query)
            ->editColumn('gross_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->gross_total.'">'.$row->gross_total.'</span>';
            })
            ->editColumn('net_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->net_total.'">'.$row->net_total.'</span>';
            })
            ->editColumn('paid_total', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$row->paid_total.'">'.$row->paid_total.'</span>';
            })
            ->rawColumns(['gross_total', 'net_total', 'paid_total'])
            ->make(true);
    }
}
