<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenAllocation;
use Modules\Mosque\Entities\TenSeat;
use Modules\Mosque\Entities\TenTenant;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\TenancyNoticeUtil;

class TenancyAllocationsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensureTenancyEnabled(int $businessId): void
    {
        if (! Schema::hasTable('mosque_settings')) {
            return;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can('mosque.tenancy.allocate')) {
            abort(403, 'Unauthorized action.');
        }
    }

    public function store(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'seat_id' => 'required|integer',
            'tenant_id' => 'required|integer',
            'start_date' => 'required|date',
            'rent_amount' => 'required|numeric|min:0',
            'billing_cycle' => 'required|in:monthly,yearly',
            'security_deposit' => 'nullable|numeric|min:0',
            'note' => 'nullable|string|max:2000',
        ]);

        try {
            $seatId = (int) $request->input('seat_id');
            $tenantId = (int) $request->input('tenant_id');

            $seat = TenSeat::query()->where('business_id', $businessId)->findOrFail($seatId);
            $tenant = TenTenant::query()->where('business_id', $businessId)->findOrFail($tenantId);

            if ($seat->activeAllocation()->exists()) {
                return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_seat_has_allocation')];
            }

            $startDate = (string) $request->input('start_date');

            $allocation = null;
            DB::transaction(function () use ($businessId, $seat, $tenant, $request, $startDate, &$allocation) {
                $allocation = TenAllocation::query()->create([
                    'business_id' => $businessId,
                    'seat_id' => (int) $seat->id,
                    'tenant_id' => (int) $tenant->id,
                    'start_date' => $startDate,
                    'end_date' => null,
                    'status' => 'active',
                    'rent_amount' => (float) $request->input('rent_amount'),
                    'billing_cycle' => (string) $request->input('billing_cycle', 'monthly'),
                    'next_bill_on' => $startDate,
                    'security_deposit' => $request->input('security_deposit'),
                    'note' => $request->input('note'),
                ]);

                $seat->status = 'occupied';
                $seat->leaving_on = null;
                $seat->save();
            });

            MosqueAuditUtil::log($businessId, 'create', 'ten_allocation', (int) $allocation->id, [
                'seat_id' => (int) $seat->id,
                'tenant_id' => (int) $tenant->id,
                'start_date' => $startDate,
            ]);

            $notify = TenancyNoticeUtil::allocationApproved($businessId, $tenant, $allocation);

            return [
                'success' => true,
                'msg' => __('lang_v1.success'),
                'whatsapp_links' => $notify['whatsapp_links'] ?? [],
            ];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function move(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'new_seat_id' => 'required|integer',
            'move_date' => 'required|date',
            'note' => 'nullable|string|max:2000',
        ]);

        try {
            $allocation = TenAllocation::query()->where('business_id', $businessId)->findOrFail((int) $id);
            if ($allocation->status !== 'active') {
                return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_allocation_not_active')];
            }

            $newSeat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $request->input('new_seat_id'));
            if ($newSeat->activeAllocation()->exists()) {
                return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_seat_has_allocation')];
            }

            $oldSeat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $allocation->seat_id);
            $tenant = TenTenant::query()->where('business_id', $businessId)->findOrFail((int) $allocation->tenant_id);

            $moveDate = (string) $request->input('move_date');

            $newAllocation = null;
            DB::transaction(function () use ($businessId, $allocation, $oldSeat, $newSeat, $moveDate, $request, &$newAllocation) {
                $allocation->status = 'ended';
                $allocation->end_date = $moveDate;
                $allocation->save();

                $oldSeat->status = 'available';
                $oldSeat->leaving_on = null;
                $oldSeat->save();

                $newAllocation = TenAllocation::query()->create([
                    'business_id' => $businessId,
                    'seat_id' => (int) $newSeat->id,
                    'tenant_id' => (int) $allocation->tenant_id,
                    'start_date' => $moveDate,
                    'end_date' => null,
                    'status' => 'active',
                    'rent_amount' => (float) $allocation->rent_amount,
                    'billing_cycle' => (string) $allocation->billing_cycle,
                    'next_bill_on' => $moveDate,
                    'security_deposit' => $allocation->security_deposit,
                    'note' => (string) $request->input('note', ''),
                ]);

                $newSeat->status = 'occupied';
                $newSeat->leaving_on = null;
                $newSeat->save();
            });

            MosqueAuditUtil::log($businessId, 'move', 'ten_allocation', (int) $allocation->id, [
                'old_seat_id' => (int) $oldSeat->id,
                'new_seat_id' => (int) $newSeat->id,
                'move_date' => $moveDate,
            ]);

            $notify = TenancyNoticeUtil::allocationApproved($businessId, $tenant, $newAllocation);

            return [
                'success' => true,
                'msg' => __('lang_v1.success'),
                'whatsapp_links' => $notify['whatsapp_links'] ?? [],
            ];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function end(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'end_date' => 'required|date',
            'note' => 'nullable|string|max:2000',
        ]);

        try {
            $allocation = TenAllocation::query()->where('business_id', $businessId)->findOrFail((int) $id);
            $seat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $allocation->seat_id);

            $endDate = (string) $request->input('end_date');
            $today = now()->toDateString();

            DB::transaction(function () use ($allocation, $seat, $endDate, $today, $request) {
                $allocation->status = 'ended';
                $allocation->end_date = $endDate;
                $allocation->note = (string) $request->input('note', $allocation->note);
                $allocation->save();

                if ($endDate <= $today) {
                    $seat->status = 'available';
                    $seat->leaving_on = null;
                } else {
                    $seat->leaving_on = $endDate;
                }
                $seat->save();
            });

            MosqueAuditUtil::log($businessId, 'end', 'ten_allocation', (int) $allocation->id, [
                'end_date' => $endDate,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function markLeaving(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'leaving_on' => 'required|date',
        ]);

        try {
            $seat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $id);
            $seat->leaving_on = (string) $request->input('leaving_on');
            $seat->save();

            MosqueAuditUtil::log($businessId, 'update', 'ten_seat', (int) $seat->id, [
                'leaving_on' => (string) $seat->leaving_on,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
