<?php

namespace Modules\Mosque\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueFinanceCategory;
use Modules\Mosque\Entities\MosqueFinanceEntry;
use Modules\Mosque\Entities\MosqueProfile;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenBill;
use Modules\Mosque\Services\TenancyBillingService;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\MosqueDocumentUtil;
use Yajra\DataTables\Facades\DataTables;

class TenancyBillingController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(string $perm): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can($perm)) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureTenancyEnabled(int $businessId): void
    {
        if (! Schema::hasTable('mosque_settings')) {
            return;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    public function index()
    {
        $this->ensurePermission('mosque.tenancy.view');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        return view('mosque::tenancy.billing.index');
    }

    public function data(Request $request)
    {
        $this->ensurePermission('mosque.tenancy.view');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $query = DB::table('ten_bills as b')
            ->join('ten_allocations as a', function ($join) use ($businessId) {
                $join->on('a.id', '=', 'b.allocation_id')
                    ->where('a.business_id', '=', $businessId)
                    ->whereNull('a.deleted_at');
            })
            ->join('ten_tenants as t', function ($join) use ($businessId) {
                $join->on('t.id', '=', 'a.tenant_id')
                    ->where('t.business_id', '=', $businessId)
                    ->whereNull('t.deleted_at');
            })
            ->join('ten_seats as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'a.seat_id')
                    ->where('s.business_id', '=', $businessId)
                    ->whereNull('s.deleted_at');
            })
            ->join('ten_rooms as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 's.room_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            })
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as bd', function ($join) use ($businessId) {
                $join->on('bd.id', '=', 'f.building_id')
                    ->where('bd.business_id', '=', $businessId)
                    ->whereNull('bd.deleted_at');
            })
            ->where('b.business_id', $businessId)
            ->whereNull('b.deleted_at')
            ->select([
                'b.id',
                'b.period_from',
                'b.period_to',
                'b.amount',
                'b.status',
                'b.pos_invoice_id',
                'b.generated_on',
                't.full_name as tenant_name',
                't.phone as tenant_phone',
                DB::raw("CONCAT(bd.name,' / ',f.name_or_number,' / ',r.code,' / ',s.code) as seat_label"),
            ]);

        $status = (string) $request->input('status', '');
        if ($status !== '') {
            $query->where('b.status', $status);
        }

        $cycle = (string) $request->input('cycle', '');
        if ($cycle !== '') {
            $query->where('a.billing_cycle', $cycle);
        }

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId > 0) {
            $query->where('bd.id', $buildingId);
        }

        $dateRange = (string) $request->input('date_range', '');
        if ($dateRange !== '') {
            // expects "MM/DD/YYYY - MM/DD/YYYY"
            $parts = preg_split('/\s*-\s*/', $dateRange);
            if (is_array($parts) && count($parts) === 2) {
                try {
                    $start = \Carbon\Carbon::createFromFormat('m/d/Y', trim($parts[0]))->toDateString();
                    $end = \Carbon\Carbon::createFromFormat('m/d/Y', trim($parts[1]))->toDateString();
                    $query->whereDate('b.period_from', '>=', $start)->whereDate('b.period_to', '<=', $end);
                } catch (\Throwable $e) {
                }
            }
        }

        $keyword = trim((string) $request->input('keyword', ''));
        if ($keyword !== '') {
            $query->where(function ($q) use ($keyword) {
                $q->where('t.full_name', 'like', '%'.$keyword.'%')
                    ->orWhere('t.phone', 'like', '%'.$keyword.'%')
                    ->orWhere('s.code', 'like', '%'.$keyword.'%')
                    ->orWhere('r.code', 'like', '%'.$keyword.'%')
                    ->orWhere('bd.name', 'like', '%'.$keyword.'%');
            });
        }

        return DataTables::of($query)
            ->addColumn('amount_display', function ($row) {
                $v = (float) ($row->amount ?? 0);
                return '<span class="display_currency" data-currency_symbol="true" data-orig-value="'.$v.'">'.$v.'</span>';
            })
            ->addColumn('action', function ($row) {
                $print = '<a target="_blank" href="'.route('mosque.tenancy.billing.receipt.print', [(int) $row->id], false).'" class="btn btn-xs btn-default"><i class="fa fa-print"></i> '.__('messages.print').'</a>';
                $pdf = '<a href="'.route('mosque.tenancy.billing.receipt.pdf', [(int) $row->id], false).'" class="btn btn-xs btn-default"><i class="fa fa-file-pdf"></i> PDF</a>';

                $invoice = '';
                if (auth()->user()->can('mosque.tenancy.billing.run')) {
                    $url = route('mosque.tenancy.billing.invoice', [(int) $row->id], false);
                    $invoice = '<button data-href="'.$url.'" class="btn btn-xs btn-primary mosque_ajax_post"><i class="fa fa-receipt"></i> '.__('mosque::mosque.tenancy_create_invoice').'</button>';
                }

                return trim($print.' '.$pdf.' '.$invoice);
            })
            ->rawColumns(['amount_display', 'action'])
            ->make(true);
    }

    public function generate(Request $request)
    {
        $this->ensurePermission('mosque.tenancy.billing.run');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        try {
            $result = TenancyBillingService::generateDueBills($businessId, [
                'max_allocations' => 500,
                'send_notices' => true,
            ]);

            return [
                'success' => true,
                'msg' => __('mosque::mosque.tenancy_bills_generated', ['count' => (int) ($result['generated'] ?? 0)]),
                'whatsapp_links' => $result['whatsapp_links'] ?? [],
            ];
        } catch (\Throwable $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function createInvoice(Request $request, $id)
    {
        $this->ensurePermission('mosque.tenancy.billing.run');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        try {
            if (! Schema::hasTable('mosque_finance_entries') || ! Schema::hasTable('mosque_finance_categories')) {
                return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_finance_missing')];
            }

            /** @var TenBill $bill */
            $bill = TenBill::query()->where('business_id', $businessId)->findOrFail((int) $id);
            if (! empty($bill->pos_invoice_id)) {
                return ['success' => true, 'msg' => __('lang_v1.success')];
            }

            $category = MosqueFinanceCategory::query()
                ->where('business_id', $businessId)
                ->where('type', 'income')
                ->where('name', 'Hall/property rent')
                ->first();

            if (! $category) {
                $category = MosqueFinanceCategory::query()->create([
                    'business_id' => $businessId,
                    'type' => 'income',
                    'name' => 'Hall/property rent',
                    'active' => true,
                    'sort_order' => 10,
                ]);
            }

            $entry = MosqueFinanceEntry::query()->create([
                'business_id' => $businessId,
                'location_id' => null,
                'type' => 'income',
                'category_id' => (int) $category->id,
                'amount' => (float) $bill->amount,
                'entry_date' => now()->toDateString(),
                'ref_module' => 'tenancy_rent',
                'ref_id' => (int) $bill->id,
                'fund_tag' => null,
                'note' => 'Tenancy rent bill #'.$bill->id,
                'created_by' => auth()->id(),
            ]);

            $bill->pos_invoice_id = (int) $entry->id;
            $bill->status = 'paid';
            $bill->save();

            MosqueAuditUtil::log($businessId, 'create', 'ten_bill_invoice', (int) $bill->id, [
                'finance_entry_id' => (int) $entry->id,
                'amount' => (float) $bill->amount,
            ]);

            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Throwable $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function receiptPrint($id)
    {
        $this->ensurePermission('mosque.tenancy.view');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $bill = TenBill::query()->where('business_id', $businessId)->with(['allocation.tenant', 'allocation.seat.room.floor.building'])->findOrFail((int) $id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'print', 'ten_bill_receipt', (int) $bill->id, [
            'amount' => (float) $bill->amount,
        ]);

        return view('mosque::tenancy.billing.receipt_print', compact('bill', 'profile', 'logoDataUri', 'settings'));
    }

    public function receiptPdf($id)
    {
        $this->ensurePermission('mosque.tenancy.view');
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $bill = TenBill::query()->where('business_id', $businessId)->with(['allocation.tenant', 'allocation.seat.room.floor.building'])->findOrFail((int) $id);

        $profile = null;
        if (Schema::hasTable('mosque_profiles')) {
            $profile = MosqueProfile::query()->where('business_id', $businessId)->first();
        }
        $logoDataUri = MosqueDocumentUtil::logoDataUri($profile);

        $settings = [];
        if (Schema::hasTable('mosque_settings')) {
            $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        }

        MosqueAuditUtil::log($businessId, 'pdf', 'ten_bill_receipt', (int) $bill->id, [
            'amount' => (float) $bill->amount,
        ]);

        $pdf = Pdf::loadView('mosque::tenancy.billing.receipt_print', compact('bill', 'profile', 'logoDataUri', 'settings'))
            ->setPaper('a4');

        return $pdf->download('tenancy_bill_'.$bill->id.'.pdf');
    }
}
