<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenCase;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\TenancyKeyUtil;
use Yajra\DataTables\Facades\DataTables;

class TenancyCasesController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can('mosque.tenancy.cases.view')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureTenancyEnabled(int $businessId): void
    {
        if (! Schema::hasTable('mosque_settings')) {
            return;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    private function ensureUploadDir(): void
    {
        try {
            Storage::disk('public')->makeDirectory('mosque_tenancy/cases');
        } catch (\Throwable $e) {
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        return view('mosque::tenancy.cases.index');
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $query = TenCase::query()
            ->where('business_id', $businessId)
            ->select(['id', 'tenant_key', 'title', 'severity', 'created_at']);

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $edit = '';
                if (auth()->user()->can('mosque.tenancy.edit')) {
                    $edit = '<a href="'.route('mosque.tenancy.cases.edit', [(int) $row->id], false).'" class="btn btn-xs btn-primary"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</a>';
                }
                $delete = '';
                if (auth()->user()->can('mosque.tenancy.edit')) {
                    $deleteUrl = route('mosque.tenancy.cases.destroy', [(int) $row->id], false);
                    $delete = '<button data-href="'.$deleteUrl.'" onclick="(function(){var e=(arguments&&arguments[0])?arguments[0]:(window.event||null);if(e){if(e.preventDefault)e.preventDefault();if(e.stopImmediatePropagation)e.stopImmediatePropagation();else if(e.stopPropagation)e.stopPropagation();}var url=\''.$deleteUrl.'\';var ent=\'case\';if(typeof window.mosqueOpenDeleteConfirm===\'function\'){window.mosqueOpenDeleteConfirm(url,\'DELETE\',ent);return;}window.location.reload();})(); return false;" class="btn btn-xs btn-danger delete_mosque_case"><i class="glyphicon glyphicon-trash"></i> '.__('messages.delete').'</button>';
                }
                return trim($edit.' '.$delete);
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function create()
    {
        $this->ensurePermission();
        if (! auth()->user()->can('mosque.tenancy.edit')) {
            abort(403);
        }
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        return view('mosque::tenancy.cases.form', ['case' => null]);
    }

    public function edit($id)
    {
        $this->ensurePermission();
        if (! auth()->user()->can('mosque.tenancy.edit')) {
            abort(403);
        }
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $case = TenCase::query()->where('business_id', $businessId)->findOrFail((int) $id);
        return view('mosque::tenancy.cases.form', ['case' => $case]);
    }

    public function store(Request $request)
    {
        $this->ensurePermission();
        if (! auth()->user()->can('mosque.tenancy.edit')) {
            abort(403);
        }
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $request->validate([
            'phone' => 'nullable|string|max:50',
            'nid_no' => 'nullable|string|max:80',
            'passport_no' => 'nullable|string|max:80',
            'title' => 'required|string|max:255',
            'details' => 'required|string|max:10000',
            'severity' => 'required|in:low,medium,high,critical',
            'attachments.*' => 'nullable|file|max:5120',
        ]);

        $tenantKey = TenancyKeyUtil::tenantKey((string) $request->input('phone', ''), (string) $request->input('nid_no', ''), (string) $request->input('passport_no', ''));
        if ($tenantKey === '') {
            return response()->json(['success' => false, 'msg' => __('mosque::mosque.tenancy_err_case_key_required')], 422);
        }

        $this->ensureUploadDir();
        $attachments = [];
        if ($request->hasFile('attachments')) {
            foreach ((array) $request->file('attachments') as $file) {
                if (! $file || ! $file->isValid()) {
                    continue;
                }
                $name = 'case_'.$businessId.'_'.now()->format('YmdHis').'_'.Str::random(6).'.'.$file->getClientOriginalExtension();
                $path = $file->storeAs('mosque_tenancy/cases', $name, 'public');
                $attachments[] = ['name' => $file->getClientOriginalName(), 'path' => $path];
            }
        }

        $case = TenCase::query()->create([
            'business_id' => $businessId,
            'tenant_key' => $tenantKey,
            'title' => (string) $request->input('title'),
            'details' => (string) $request->input('details'),
            'severity' => (string) $request->input('severity'),
            'created_by' => auth()->id(),
            'attachments' => $attachments,
        ]);

        MosqueAuditUtil::log($businessId, 'create', 'ten_case', (int) $case->id, [
            'tenant_key' => $tenantKey,
            'severity' => (string) $case->severity,
        ]);

        return ['success' => true, 'msg' => __('lang_v1.success'), 'redirect' => route('mosque.tenancy.cases', [], false)];
    }

    public function update(Request $request, $id)
    {
        $this->ensurePermission();
        if (! auth()->user()->can('mosque.tenancy.edit')) {
            abort(403);
        }
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $case = TenCase::query()->where('business_id', $businessId)->findOrFail((int) $id);

        $request->validate([
            'title' => 'required|string|max:255',
            'details' => 'required|string|max:10000',
            'severity' => 'required|in:low,medium,high,critical',
            'attachments.*' => 'nullable|file|max:5120',
        ]);

        $old = $case->toArray();

        $this->ensureUploadDir();
        $attachments = is_array($case->attachments) ? $case->attachments : [];
        if ($request->hasFile('attachments')) {
            foreach ((array) $request->file('attachments') as $file) {
                if (! $file || ! $file->isValid()) {
                    continue;
                }
                $name = 'case_'.$businessId.'_'.now()->format('YmdHis').'_'.Str::random(6).'.'.$file->getClientOriginalExtension();
                $path = $file->storeAs('mosque_tenancy/cases', $name, 'public');
                $attachments[] = ['name' => $file->getClientOriginalName(), 'path' => $path];
            }
        }

        $case->update([
            'title' => (string) $request->input('title'),
            'details' => (string) $request->input('details'),
            'severity' => (string) $request->input('severity'),
            'attachments' => $attachments,
        ]);

        MosqueAuditUtil::log($businessId, 'update', 'ten_case', (int) $case->id, [
            'old' => $old,
            'new' => $case->toArray(),
        ]);

        return ['success' => true, 'msg' => __('lang_v1.success'), 'redirect' => route('mosque.tenancy.cases', [], false)];
    }

    public function destroy($id)
    {
        $this->ensurePermission();
        if (! auth()->user()->can('mosque.tenancy.edit')) {
            abort(403);
        }
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $case = TenCase::query()->where('business_id', $businessId)->findOrFail((int) $id);
        $old = $case->toArray();
        $case->delete();

        MosqueAuditUtil::log($businessId, 'delete', 'ten_case', (int) $case->id, $old);
        return ['success' => true, 'msg' => __('lang_v1.success')];
    }
}
