<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenAllocation;
use Modules\Mosque\Entities\TenBill;
use Modules\Mosque\Entities\TenBuilding;
use Modules\Mosque\Entities\TenCase;
use Modules\Mosque\Entities\TenFloor;
use Modules\Mosque\Entities\TenRequest;
use Modules\Mosque\Entities\TenRoom;
use Modules\Mosque\Entities\TenSeat;
use Modules\Mosque\Services\TenancyBillingService;

class TenancyDashboardController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->hasAnyPermission([
            'mosque.tenancy.view',
            'mosque.tenancy.edit',
            'mosque.tenancy.allocate',
            'mosque.tenancy.billing.run',
            'mosque.tenancy.requests.manage',
            'mosque.tenancy.cases.view',
            'mosque.tenancy.settings.manage',
        ])) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function tenancyEnabled(int $businessId): bool
    {
        if (! Schema::hasTable('mosque_settings')) {
            return true;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        return (bool) ($settings['tenancy_enabled'] ?? true);
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();

        if (! $this->tenancyEnabled($businessId)) {
            return view('mosque::tenancy.dashboard', [
                'tenancy_disabled' => true,
                'stats' => [],
            ]);
        }

        if (! Schema::hasTable('ten_buildings')) {
            return view('mosque::tenancy.dashboard', [
                'tenancy_disabled' => false,
                'stats' => [],
                'missing_tables' => true,
            ]);
        }

        $today = now()->toDateString();
        $soon = now()->addMonth()->toDateString();

        // Room type statistics
        $roomsQuery = TenRoom::query()->where('business_id', $businessId);
        $roomsCount = (clone $roomsQuery)->where('type', 'room')->count();
        $apartmentsCount = (clone $roomsQuery)->where('type', 'apartment')->count();
        $wardsCount = (clone $roomsQuery)->where('type', 'ward')->count();
        
        // Seat counts by room type
        $roomIds = (clone $roomsQuery)->where('type', 'room')->pluck('id')->toArray();
        $apartmentIds = (clone $roomsQuery)->where('type', 'apartment')->pluck('id')->toArray();
        $wardIds = (clone $roomsQuery)->where('type', 'ward')->pluck('id')->toArray();
        
        $roomsSeatsCount = empty($roomIds) ? 0 : TenSeat::query()
            ->where('business_id', $businessId)
            ->whereIn('room_id', $roomIds)
            ->count();
        $apartmentsSeatsCount = empty($apartmentIds) ? 0 : TenSeat::query()
            ->where('business_id', $businessId)
            ->whereIn('room_id', $apartmentIds)
            ->count();
        $wardsSeatsCount = empty($wardIds) ? 0 : TenSeat::query()
            ->where('business_id', $businessId)
            ->whereIn('room_id', $wardIds)
            ->count();

        $stats = [
            'buildings' => TenBuilding::query()->where('business_id', $businessId)->count(),
            'floors' => TenFloor::query()->where('business_id', $businessId)->count(),
            'rooms' => TenRoom::query()->where('business_id', $businessId)->count(),
            'rooms_count' => $roomsCount,
            'apartments_count' => $apartmentsCount,
            'wards_count' => $wardsCount,
            'rooms_seats_count' => $roomsSeatsCount,
            'apartments_seats_count' => $apartmentsSeatsCount,
            'wards_seats_count' => $wardsSeatsCount,
            'seats_total' => TenSeat::query()->where('business_id', $businessId)->count(),
            'seats_available' => TenSeat::query()->where('business_id', $businessId)->where('status', 'available')->count(),
            'seats_occupied' => TenSeat::query()->where('business_id', $businessId)->where('status', 'occupied')->count(),
            'seats_maintenance' => TenSeat::query()->where('business_id', $businessId)->whereIn('status', ['maintenance', 'held'])->count(),
            'seats_leaving_soon' => TenSeat::query()
                ->where('business_id', $businessId)
                ->whereNotNull('leaving_on')
                ->whereDate('leaving_on', '>=', $today)
                ->whereDate('leaving_on', '<=', $soon)
                ->count(),
            'requests_pending' => TenRequest::query()->where('business_id', $businessId)->where('status', 'pending')->count(),
            'bills_due' => TenBill::query()->where('business_id', $businessId)->whereIn('status', ['unpaid', 'partially_paid'])->whereDate('period_to', '<=', $today)->count(),
            'allocations_active' => TenAllocation::query()->where('business_id', $businessId)->where('status', 'active')->count(),
            'cases_total' => TenCase::query()->where('business_id', $businessId)->count(),
        ];

        $generator = TenancyBillingService::previewDueBills($businessId, 10);

        return view('mosque::tenancy.dashboard', [
            'tenancy_disabled' => false,
            'missing_tables' => false,
            'stats' => $stats,
            'preview_due' => $generator['due'] ?? [],
        ]);
    }
}
