<?php

namespace Modules\Mosque\Http\Controllers;

use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenRequest;
use Modules\Mosque\Utils\TenancyNoticeUtil;

class TenancyPublicController extends Controller
{
    private function ensureTokenNotExpired(array $settings): void
    {
        $ttlMinutes = (int) ($settings['tenancy_qr_ttl_minutes'] ?? (60 * 24 * 30));
        if ($ttlMinutes <= 0) {
            return;
        }

        $createdAt = (string) ($settings['tenancy_public_token_created_at'] ?? '');
        if (trim($createdAt) === '') {
            return;
        }

        try {
            $created = Carbon::parse($createdAt);
            if (now()->greaterThan($created->addMinutes($ttlMinutes))) {
                abort(403, __('mosque::mosque.tenancy_public_link_expired'));
            }
        } catch (\Throwable $e) {
        }
    }

    private function settingsRowByToken(string $token): ?MosqueSetting
    {
        if (! Schema::hasTable('mosque_settings')) {
            return null;
        }

        try {
            return MosqueSetting::query()
                ->where('settings->tenancy_public_token', $token)
                ->first();
        } catch (\Throwable $e) {
            // Fallback: scan settings (slow, best effort) if JSON operator unsupported.
            return MosqueSetting::query()->get()->first(function ($row) use ($token) {
                $settings = $row->settings ?: [];
                return (string) ($settings['tenancy_public_token'] ?? '') === $token;
            });
        }
    }

    private function ensureSubscriptionEnabled(int $businessId): void
    {
        $moduleUtil = new ModuleUtil();
        $enabled = (bool) $moduleUtil->hasThePermissionInSubscription($businessId, 'mosque_module', 'superadmin_package');
        if (! $enabled) {
            abort(403, 'Mosque module is not available for this business.');
        }
    }

    private function ensureTenancyEnabled(int $businessId, array $settings): void
    {
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    public function show(Request $request, $token)
    {
        $token = (string) $token;
        $row = $this->settingsRowByToken($token);
        if (! $row) {
            abort(404);
        }

        $businessId = (int) $row->business_id;
        $settings = $row->settings ?: [];
        $this->ensureSubscriptionEnabled($businessId);
        $this->ensureTenancyEnabled($businessId, $settings);

        $qrTitle = (string) ($settings['tenancy_qr_title'] ?? __('mosque::mosque.tenancy_requests_qr_title_default'));
        $primaryColor = (string) ($settings['tenancy_qr_primary_color'] ?? '#000000');

        $today = now()->toDateString();
        $soon = now()->addMonth()->toDateString();

        $seats = DB::table('ten_seats as s')
            ->join('ten_rooms as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 's.room_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            })
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->where('s.business_id', $businessId)
            ->whereNull('s.deleted_at')
            ->leftJoin('ten_allocations as a', function ($join) use ($businessId) {
                $join->on('a.seat_id', '=', 's.id')
                    ->where('a.business_id', '=', $businessId)
                    ->whereNull('a.deleted_at')
                    ->where('a.status', '=', 'active');
            })
            ->where(function ($q) use ($today, $soon) {
                $q->where('s.status', 'available')
                    ->orWhere(function ($qq) use ($today, $soon) {
                        $qq->whereNotNull('s.leaving_on')
                            ->whereDate('s.leaving_on', '>=', $today)
                            ->whereDate('s.leaving_on', '<=', $soon);
                    })
                    ->orWhere(function ($qq) use ($today, $soon) {
                        $qq->whereNotNull('a.end_date')
                            ->whereDate('a.end_date', '>=', $today)
                            ->whereDate('a.end_date', '<=', $soon);
                    });
            })
            ->orderBy('b.name')
            ->orderBy('f.sort_order')
            ->orderBy('r.code')
            ->orderBy('s.code')
            ->select([
                's.id',
                's.code as seat_code',
                's.status',
                's.leaving_on',
                DB::raw("CONCAT(b.name,' / ',f.name_or_number,' / ',r.code,' / ',s.code) as label"),
            ])
            ->get();

        $seats = $seats->map(function ($s) use ($businessId) {
            $seatId = (int) $s->id;
            $s->seat_token = Crypt::encryptString($businessId.'|'.$seatId);
            return $s;
        });

        return view('mosque::tenancy.public.request', [
            'token' => $token,
            'qrTitle' => $qrTitle,
            'primaryColor' => $primaryColor,
            'seats' => $seats,
            'signedUrl' => $request->fullUrl(),
        ]);
    }

    public function showShort(Request $request, $token)
    {
        $token = (string) $token;
        $row = $this->settingsRowByToken($token);
        if (! $row) {
            abort(404);
        }

        $businessId = (int) $row->business_id;
        $settings = $row->settings ?: [];
        $this->ensureSubscriptionEnabled($businessId);
        $this->ensureTenancyEnabled($businessId, $settings);
        $this->ensureTokenNotExpired($settings);

        $qrTitle = (string) ($settings['tenancy_qr_title'] ?? __('mosque::mosque.tenancy_requests_qr_title_default'));
        $primaryColor = (string) ($settings['tenancy_qr_primary_color'] ?? '#000000');

        $today = now()->toDateString();
        $soon = now()->addMonth()->toDateString();

        $seats = DB::table('ten_seats as s')
            ->join('ten_rooms as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 's.room_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            })
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->where('s.business_id', $businessId)
            ->whereNull('s.deleted_at')
            ->leftJoin('ten_allocations as a', function ($join) use ($businessId) {
                $join->on('a.seat_id', '=', 's.id')
                    ->where('a.business_id', '=', $businessId)
                    ->whereNull('a.deleted_at')
                    ->where('a.status', '=', 'active');
            })
            ->where(function ($q) use ($today, $soon) {
                $q->where('s.status', 'available')
                    ->orWhere(function ($qq) use ($today, $soon) {
                        $qq->whereNotNull('s.leaving_on')
                            ->whereDate('s.leaving_on', '>=', $today)
                            ->whereDate('s.leaving_on', '<=', $soon);
                    })
                    ->orWhere(function ($qq) use ($today, $soon) {
                        $qq->whereNotNull('a.end_date')
                            ->whereDate('a.end_date', '>=', $today)
                            ->whereDate('a.end_date', '<=', $soon);
                    });
            })
            ->orderBy('b.name')
            ->orderBy('f.sort_order')
            ->orderBy('r.code')
            ->orderBy('s.code')
            ->select([
                's.id',
                's.code as seat_code',
                's.status',
                's.leaving_on',
                DB::raw("CONCAT(b.name,' / ',f.name_or_number,' / ',r.code,' / ',s.code) as label"),
            ])
            ->get();

        $seats = $seats->map(function ($s) use ($businessId) {
            $seatId = (int) $s->id;
            $s->seat_token = Crypt::encryptString($businessId.'|'.$seatId);
            return $s;
        });

        return view('mosque::tenancy.public.request', [
            'token' => $token,
            'qrTitle' => $qrTitle,
            'primaryColor' => $primaryColor,
            'seats' => $seats,
            'signedUrl' => route('mosque.tenancy.public.short.submit', ['token' => $token], true),
        ]);
    }

    public function submit(Request $request, $token)
    {
        $token = (string) $token;
        $row = $this->settingsRowByToken($token);
        if (! $row) {
            abort(404);
        }

        $businessId = (int) $row->business_id;
        $settings = $row->settings ?: [];
        $this->ensureSubscriptionEnabled($businessId);
        $this->ensureTenancyEnabled($businessId, $settings);

        $request->validate([
            'requested_by_name' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'seat_id' => 'nullable|string|max:500',
            'prefer_period' => 'nullable|string|max:120',
            'notes' => 'nullable|string|max:1000',
        ]);

        if (! Schema::hasTable('ten_requests')) {
            abort(500, 'Tenancy requests table missing.');
        }

        $tracking = strtoupper(Str::random(10));
        $seatId = 0;
        $seatToken = (string) $request->input('seat_id', '');
        if (trim($seatToken) !== '') {
            try {
                $decoded = (string) Crypt::decryptString($seatToken);
                [$bid, $sid] = array_pad(explode('|', $decoded, 2), 2, '');
                if ((int) $bid === $businessId) {
                    $seatId = (int) $sid;
                }
            } catch (\Throwable $e) {
                $seatId = 0;
            }
        }

        // ensure token uniqueness per business
        $tries = 0;
        while (TenRequest::query()->where('business_id', $businessId)->where('token', $tracking)->exists() && $tries < 5) {
            $tracking = strtoupper(Str::random(10));
            $tries++;
        }

        TenRequest::query()->create([
            'business_id' => $businessId,
            'seat_id' => $seatId > 0 ? $seatId : null,
            'token' => $tracking,
            'requested_by_name' => (string) $request->input('requested_by_name'),
            'phone' => (string) $request->input('phone'),
            'prefer_period' => (string) $request->input('prefer_period', ''),
            'notes' => (string) $request->input('notes', ''),
            'status' => 'pending',
            'audit' => [
                'ip' => (string) $request->ip(),
                'ua' => (string) $request->userAgent(),
            ],
        ]);

        try {
            TenancyNoticeUtil::requestReceived($businessId, (string) $request->input('requested_by_name'), (string) $request->input('phone'), $tracking);
        } catch (\Throwable $e) {
        }

        return redirect()->back()->with('status', [
            'success' => 1,
            'msg' => __('mosque::mosque.tenancy_request_received', ['code' => $tracking]),
        ]);
    }

    public function submitShort(Request $request, $token)
    {
        $token = (string) $token;
        $row = $this->settingsRowByToken($token);
        if (! $row) {
            abort(404);
        }

        $businessId = (int) $row->business_id;
        $settings = $row->settings ?: [];
        $this->ensureSubscriptionEnabled($businessId);
        $this->ensureTenancyEnabled($businessId, $settings);
        $this->ensureTokenNotExpired($settings);

        return $this->submit($request, $token);
    }
}
