<?php

namespace Modules\Mosque\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenBuilding;
use Yajra\DataTables\Facades\DataTables;

class TenancyReportsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can('mosque.tenancy.view')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureTenancyEnabled(int $businessId): void
    {
        if (! Schema::hasTable('mosque_settings')) {
            return;
        }
        $settings = (MosqueSetting::query()->where('business_id', $businessId)->value('settings')) ?: [];
        if (! (bool) ($settings['tenancy_enabled'] ?? true)) {
            abort(403, 'Tenancy is disabled.');
        }
    }

    public function index()
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $buildings = Schema::hasTable('ten_buildings')
            ? TenBuilding::query()->where('business_id', $businessId)->orderBy('name')->pluck('name', 'id')->toArray()
            : [];

        return view('mosque::tenancy.reports', compact('buildings'));
    }

    public function occupancyData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $today = now()->toDateString();
        $soon = now()->addMonth()->toDateString();

        $query = DB::table('ten_rooms as r')
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->leftJoin('ten_seats as s', function ($join) use ($businessId) {
                $join->on('s.room_id', '=', 'r.id')
                    ->where('s.business_id', '=', $businessId)
                    ->whereNull('s.deleted_at');
            })
            ->leftJoin('ten_allocations as a', function ($join) use ($businessId) {
                $join->on('a.seat_id', '=', 's.id')
                    ->where('a.business_id', '=', $businessId)
                    ->whereNull('a.deleted_at')
                    ->where('a.status', '=', 'active');
            })
            ->where('r.business_id', $businessId)
            ->whereNull('r.deleted_at')
            ->groupBy('b.id', 'f.id', 'r.id', 'b.name', 'f.name_or_number', 'r.code')
            ->select([
                'b.name as building',
                'f.name_or_number as floor',
                'r.code as room',
                DB::raw('COUNT(s.id) as total_seats'),
                DB::raw("SUM(CASE WHEN (a.id IS NOT NULL OR s.status='occupied') THEN 1 ELSE 0 END) as occupied_seats"),
                DB::raw("SUM(CASE WHEN (s.status IN ('maintenance','held')) THEN 1 ELSE 0 END) as maintenance_seats"),
                DB::raw("SUM(CASE WHEN ((s.leaving_on IS NOT NULL AND DATE(s.leaving_on) BETWEEN '{$today}' AND '{$soon}') OR (a.end_date IS NOT NULL AND DATE(a.end_date) BETWEEN '{$today}' AND '{$soon}')) THEN 1 ELSE 0 END) as leaving_soon_seats"),
                DB::raw("SUM(CASE WHEN (a.id IS NULL AND (s.status IS NULL OR s.status='available') AND (s.leaving_on IS NULL OR DATE(s.leaving_on) > '{$soon}')) THEN 1 ELSE 0 END) as available_seats"),
            ]);

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId > 0) {
            $query->where('b.id', $buildingId);
        }
        $floorId = (int) $request->input('floor_id', 0);
        if ($floorId > 0) {
            $query->where('f.id', $floorId);
        }

        return DataTables::of($query)->make(true);
    }

    public function leavingSoonData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $start = (string) ($request->input('start_date') ?: now()->toDateString());
        $end = (string) ($request->input('end_date') ?: now()->addMonth()->toDateString());

        $query = DB::table('ten_seats as s')
            ->join('ten_rooms as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 's.room_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            })
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->leftJoin('ten_allocations as a', function ($join) use ($businessId) {
                $join->on('a.seat_id', '=', 's.id')
                    ->where('a.business_id', '=', $businessId)
                    ->whereNull('a.deleted_at')
                    ->where('a.status', '=', 'active');
            })
            ->leftJoin('ten_tenants as t', function ($join) use ($businessId) {
                $join->on('t.id', '=', 'a.tenant_id')
                    ->where('t.business_id', '=', $businessId)
                    ->whereNull('t.deleted_at');
            })
            ->where('s.business_id', $businessId)
            ->whereNull('s.deleted_at')
            ->where(function ($q) use ($start, $end) {
                $q->where(function ($qq) use ($start, $end) {
                    $qq->whereNotNull('s.leaving_on')
                        ->whereDate('s.leaving_on', '>=', $start)
                        ->whereDate('s.leaving_on', '<=', $end);
                })->orWhere(function ($qq) use ($start, $end) {
                    $qq->whereNotNull('a.end_date')
                        ->whereDate('a.end_date', '>=', $start)
                        ->whereDate('a.end_date', '<=', $end);
                });
            })
            ->select([
                'b.name as building',
                'f.name_or_number as floor',
                'r.code as room',
                's.code as seat',
                't.full_name as tenant',
                't.phone as phone',
                DB::raw("COALESCE(DATE(a.end_date), DATE(s.leaving_on)) as leaving_on"),
            ])
            ->orderBy('b.name')
            ->orderBy('f.sort_order')
            ->orderBy('r.code')
            ->orderBy('s.code');

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId > 0) {
            $query->where('b.id', $buildingId);
        }

        return DataTables::of($query)->make(true);
    }

    public function billingSummaryData(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureTenancyEnabled($businessId);

        $start = (string) ($request->input('start_date') ?: now()->startOfYear()->toDateString());
        $end = (string) ($request->input('end_date') ?: now()->endOfYear()->toDateString());

        $query = DB::table('ten_bills as bl')
            ->join('ten_allocations as a', function ($join) use ($businessId) {
                $join->on('a.id', '=', 'bl.allocation_id')
                    ->where('a.business_id', '=', $businessId)
                    ->whereNull('a.deleted_at');
            })
            ->join('ten_seats as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'a.seat_id')
                    ->where('s.business_id', '=', $businessId)
                    ->whereNull('s.deleted_at');
            })
            ->join('ten_rooms as r', function ($join) use ($businessId) {
                $join->on('r.id', '=', 's.room_id')
                    ->where('r.business_id', '=', $businessId)
                    ->whereNull('r.deleted_at');
            })
            ->join('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'r.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->join('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->where('bl.business_id', $businessId)
            ->whereNull('bl.deleted_at')
            ->whereDate('bl.period_from', '>=', $start)
            ->whereDate('bl.period_from', '<=', $end)
            ->groupBy('b.id', 'b.name', DB::raw("DATE_FORMAT(bl.period_from, '%Y-%m')"))
            ->select([
                'b.name as building',
                DB::raw("DATE_FORMAT(bl.period_from, '%Y-%m') as period"),
                DB::raw('COUNT(bl.id) as bills'),
                DB::raw("SUM(CASE WHEN bl.status='paid' THEN 1 ELSE 0 END) as paid_bills"),
                DB::raw("SUM(CASE WHEN bl.status<>'paid' THEN 1 ELSE 0 END) as unpaid_bills"),
                DB::raw('SUM(bl.amount) as total_amount'),
                DB::raw("SUM(CASE WHEN bl.status='paid' THEN bl.amount ELSE 0 END) as paid_amount"),
                DB::raw("SUM(CASE WHEN bl.status<>'paid' THEN bl.amount ELSE 0 END) as unpaid_amount"),
            ]);

        $buildingId = (int) $request->input('building_id', 0);
        if ($buildingId > 0) {
            $query->where('b.id', $buildingId);
        }

        return DataTables::of($query)->make(true);
    }
}
