<?php

namespace Modules\Mosque\Http\Controllers;

use App\Utils\ModuleUtil;
use DNS2D;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Modules\Mosque\Entities\MosqueSetting;
use Modules\Mosque\Entities\TenAllocation;
use Modules\Mosque\Entities\TenRequest;
use Modules\Mosque\Entities\TenSeat;
use Modules\Mosque\Entities\TenTenant;
use Modules\Mosque\Utils\MosqueAuditUtil;
use Modules\Mosque\Utils\QrPngUtil;
use Modules\Mosque\Utils\TenancyNoticeUtil;
use Yajra\DataTables\Facades\DataTables;

class TenancyRequestsController extends Controller
{
    private function businessId(): int
    {
        $businessId = (int) request()->session()->get('user.business_id');
        if ($businessId <= 0) {
            abort(403, 'Unauthorized action.');
        }
        return $businessId;
    }

    private function ensurePermission(): void
    {
        if (auth()->user()->can('superadmin')) {
            return;
        }
        if (! auth()->user()->can('mosque.tenancy.requests.manage')) {
            abort(403, 'Unauthorized action.');
        }
    }

    private function ensureSubscriptionEnabled(int $businessId): void
    {
        $moduleUtil = new ModuleUtil();
        $enabled = (bool) $moduleUtil->hasThePermissionInSubscription($businessId, 'mosque_module', 'superadmin_package');
        if (! $enabled) {
            abort(403, 'Mosque module is not available for this business.');
        }
    }

    private function ensureSettingsRow(int $businessId): MosqueSetting
    {
        if (! Schema::hasTable('mosque_settings')) {
            abort(500, 'Mosque settings table missing.');
        }

        return MosqueSetting::query()->firstOrCreate(['business_id' => $businessId], ['settings' => []]);
    }

    private function normalizeHexColor(string $hexColor): string
    {
        $hex = ltrim(trim($hexColor), '#');
        if (strlen($hex) === 3) {
            $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        }
        if (! preg_match('/^[0-9a-fA-F]{6}$/', $hex)) {
            $hex = '000000';
        }
        return strtoupper($hex);
    }

    private function buildQrPng(string $qrUrl, string $primaryColor = '#000000'): string
    {
        $hex = $this->normalizeHexColor($primaryColor);
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        return QrPngUtil::barcodeOutputToPng((string) DNS2D::getBarcodePNG($qrUrl, 'QRCODE', 10, 10, [$r, $g, $b]));
    }

    private function renderQrPngWithTitle(string $qrUrl, string $title, string $primaryColor = '#000000'): string
    {
        $png = $this->buildQrPng($qrUrl, $primaryColor);
        $title = trim((string) $title);
        if ($title === '' || ! function_exists('imagecreatefromstring')) {
            return $png;
        }

        $qrImg = @imagecreatefromstring($png);
        if (! $qrImg) {
            return $png;
        }

        $qrW = imagesx($qrImg);
        $qrH = imagesy($qrImg);

        $font = 5;
        $charW = imagefontwidth($font);
        $charH = imagefontheight($font);
        $strlen = function (string $s): int {
            return function_exists('mb_strlen') ? (int) mb_strlen($s) : (int) strlen($s);
        };

        $padding = 12;
        $maxChars = max(10, (int) floor(($qrW - ($padding * 2)) / max(1, $charW)));
        $words = preg_split('/\s+/', $title) ?: [];
        $lines = [];
        $line = '';

        foreach ($words as $word) {
            $test = $line === '' ? $word : ($line.' '.$word);
            if ($strlen($test) <= $maxChars) {
                $line = $test;
                continue;
            }
            if ($line !== '') {
                $lines[] = $line;
            }
            $line = $word;
        }
        if ($line !== '') {
            $lines[] = $line;
        }
        $lines = array_slice($lines, 0, 2);

        $titleH = ($charH * count($lines)) + $padding + 6;
        $outH = $qrH + $titleH + $padding;
        $outW = $qrW;

        $out = imagecreatetruecolor($outW, $outH);
        if (! $out) {
            imagedestroy($qrImg);
            return $png;
        }

        $white = imagecolorallocate($out, 255, 255, 255);
        $black = imagecolorallocate($out, 0, 0, 0);
        imagefilledrectangle($out, 0, 0, $outW, $outH, $white);

        $y = (int) floor($padding / 2);
        foreach ($lines as $l) {
            $l = trim((string) $l);
            $lW = $charW * $strlen($l);
            $x = max(0, (int) floor(($outW - $lW) / 2));
            imagestring($out, $font, $x, $y, $l, $black);
            $y += $charH + 4;
        }

        imagecopy($out, $qrImg, 0, $titleH, 0, 0, $qrW, $qrH);
        imagedestroy($qrImg);

        ob_start();
        imagepng($out);
        $outPng = (string) ob_get_clean();
        imagedestroy($out);

        return $outPng !== '' ? $outPng : $png;
    }

    private function getOrCreatePublicToken(int $businessId): string
    {
        $row = $this->ensureSettingsRow($businessId);
        $settings = $row->settings ?: [];

        $token = (string) ($settings['tenancy_public_token'] ?? '');
        if (trim($token) === '') {
            $token = Str::random(48);
            $settings['tenancy_public_token'] = $token;
            $settings['tenancy_public_token_created_at'] = now()->toDateTimeString();
            $settings['tenancy_public_token_regenerate_count'] = (int) ($settings['tenancy_public_token_regenerate_count'] ?? 0);
            $row->settings = $settings;
            $row->save();
        }

        if (empty($settings['tenancy_public_token_created_at'])) {
            $settings['tenancy_public_token_created_at'] = now()->toDateTimeString();
            $settings['tenancy_public_token_regenerate_count'] = (int) ($settings['tenancy_public_token_regenerate_count'] ?? 0);
            $row->settings = $settings;
            $row->save();
        }

        return $token;
    }

    private function shortPublicUrl(string $token): string
    {
        return route('mosque.tenancy.public.short', ['token' => $token], true);
    }

    public function index(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $settingsRow = $this->ensureSettingsRow($businessId);
        $settings = $settingsRow->settings ?: [];

        $token = $this->getOrCreatePublicToken($businessId);
        $ttl = (int) ($settings['tenancy_qr_ttl_minutes'] ?? 60 * 24 * 30);
        $title = (string) ($settings['tenancy_qr_title'] ?? __('mosque::mosque.tenancy_requests_qr_title_default'));
        $color = (string) ($settings['tenancy_qr_primary_color'] ?? '#000000');

        $publicUrl = $this->shortPublicUrl($token);
        $regenCount = (int) ($settings['tenancy_public_token_regenerate_count'] ?? 0);
        $canRegenerate = $regenCount < 1;
        $qrImageUrl = route('mosque.tenancy.requests.qr.image', [], false);
        $qrDownloadUrl = route('mosque.tenancy.requests.qr.download', [], false);

        return view('mosque::tenancy.requests.index', compact('publicUrl', 'title', 'color', 'qrImageUrl', 'qrDownloadUrl', 'canRegenerate', 'regenCount'));
    }

    public function regenerateLink(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $row = $this->ensureSettingsRow($businessId);
        $settings = $row->settings ?: [];
        $regenCount = (int) ($settings['tenancy_public_token_regenerate_count'] ?? 0);
        if ($regenCount >= 1) {
            return ['success' => false, 'msg' => __('mosque::mosque.tenancy_regenerate_once_only')];
        }

        $settings['tenancy_public_token'] = Str::random(48);
        $settings['tenancy_public_token_created_at'] = now()->toDateTimeString();
        $settings['tenancy_public_token_regenerate_count'] = $regenCount + 1;
        $row->settings = $settings;
        $row->save();

        MosqueAuditUtil::log($businessId, 'regenerate', 'tenancy_public_qr', null, []);

        return ['success' => true, 'msg' => __('lang_v1.success')];
    }

    public function qrImage(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $row = $this->ensureSettingsRow($businessId);
        $settings = $row->settings ?: [];
        $token = $this->getOrCreatePublicToken($businessId);
        $publicUrl = $this->shortPublicUrl($token);

        \Log::info('Tenancy QR Image request - Business: ' . $businessId . ', URL: ' . $publicUrl);

        $png = $this->renderQrPngWithTitle($publicUrl, '', '#000000');
        
        if ($png === '') {
            \Log::error('Tenancy QR - renderQrPngWithTitle returned empty');
            abort(500, 'Failed to generate QR image.');
        }

        \Log::info('Tenancy QR - PNG data length: ' . strlen($png) . ' bytes');

        return response($png, 200, [
            'Content-Type' => 'image/png',
            'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        ]);
    }

    public function qrDownload(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $row = $this->ensureSettingsRow($businessId);
        $settings = $row->settings ?: [];
        $token = $this->getOrCreatePublicToken($businessId);
        $publicUrl = $this->shortPublicUrl($token);

        \Log::info('Tenancy QR Download request - Business: ' . $businessId . ', URL: ' . $publicUrl);

        $png = $this->renderQrPngWithTitle($publicUrl, '', '#000000');
        
        if ($png === '') {
            \Log::error('Tenancy QR Download - renderQrPngWithTitle returned empty');
            abort(500, 'Failed to generate QR image.');
        }

        \Log::info('Tenancy QR Download - PNG data length: ' . strlen($png) . ' bytes');

        $filename = 'tenancy_qr_'.now()->format('YmdHis').'.png';
        return response($png, 200, [
            'Content-Type' => 'image/png',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"',
            'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        ]);
    }

    public function data(Request $request)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        $query = DB::table('ten_requests as r')
            ->leftJoin('ten_seats as s', function ($join) use ($businessId) {
                $join->on('s.id', '=', 'r.seat_id')
                    ->where('s.business_id', '=', $businessId)
                    ->whereNull('s.deleted_at');
            })
            ->leftJoin('ten_rooms as rm', function ($join) use ($businessId) {
                $join->on('rm.id', '=', 's.room_id')
                    ->where('rm.business_id', '=', $businessId)
                    ->whereNull('rm.deleted_at');
            })
            ->leftJoin('ten_floors as f', function ($join) use ($businessId) {
                $join->on('f.id', '=', 'rm.floor_id')
                    ->where('f.business_id', '=', $businessId)
                    ->whereNull('f.deleted_at');
            })
            ->leftJoin('ten_buildings as b', function ($join) use ($businessId) {
                $join->on('b.id', '=', 'f.building_id')
                    ->where('b.business_id', '=', $businessId)
                    ->whereNull('b.deleted_at');
            })
            ->where('r.business_id', $businessId)
            ->whereNull('r.deleted_at')
            ->select([
                'r.id',
                'r.created_at',
                'r.requested_by_name',
                'r.phone',
                'r.prefer_period',
                'r.status',
                'r.token',
                DB::raw("CASE WHEN r.seat_id IS NULL THEN '-' ELSE CONCAT(b.name,' / ',f.name_or_number,' / ',rm.code,' / ',s.code) END as seat_label"),
            ]);

        $status = (string) $request->input('status', '');
        if ($status !== '') {
            $query->where('r.status', $status);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                if (! in_array((string) $row->status, ['pending'], true)) {
                    return '';
                }
                $approveUrl = route('mosque.tenancy.requests.approve', [(int) $row->id], false);
                $declineUrl = route('mosque.tenancy.requests.decline', [(int) $row->id], false);
                $approve = '<button data-href="'.$approveUrl.'" class="btn btn-xs btn-success mosque_ajax_post"><i class="fa fa-check"></i> '.__('mosque::mosque.tenancy_approve').'</button>';
                $decline = '<button data-href="'.$declineUrl.'" class="btn btn-xs btn-danger mosque_ajax_post"><i class="fa fa-times"></i> '.__('mosque::mosque.tenancy_decline').'</button>';
                return $approve.' '.$decline;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function approve(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        try {
            $req = TenRequest::query()->where('business_id', $businessId)->findOrFail((int) $id);
            if ($req->status !== 'pending') {
                return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_request_not_pending')];
            }

            $tenant = TenTenant::query()
                ->where('business_id', $businessId)
                ->where('phone', (string) $req->phone)
                ->first();

            if (! $tenant) {
                $tenant = TenTenant::query()->create([
                    'business_id' => $businessId,
                    'user_id' => null,
                    'full_name' => (string) $req->requested_by_name,
                    'phone' => (string) $req->phone,
                    'whatsapp' => null,
                    'email' => null,
                    'attachments' => [],
                    'form_data' => ['source' => 'tenancy_public_request', 'prefer_period' => (string) ($req->prefer_period ?? '')],
                ]);
                MosqueAuditUtil::log($businessId, 'create', 'ten_tenant', (int) $tenant->id, ['source' => 'ten_request']);
            }

            $allocation = null;
            if (! empty($req->seat_id)) {
                $seat = TenSeat::query()->where('business_id', $businessId)->findOrFail((int) $req->seat_id);
                if ($seat->activeAllocation()->exists()) {
                    return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_seat_has_allocation')];
                }

                $allocation = TenAllocation::query()->create([
                    'business_id' => $businessId,
                    'seat_id' => (int) $seat->id,
                    'tenant_id' => (int) $tenant->id,
                    'start_date' => now()->toDateString(),
                    'end_date' => null,
                    'status' => 'pending',
                    'rent_amount' => 0,
                    'billing_cycle' => 'monthly',
                    'next_bill_on' => now()->toDateString(),
                    'security_deposit' => null,
                    'note' => 'Created from public request #'.$req->id,
                ]);

                $seat->status = 'held';
                $seat->save();
            }

            $req->status = 'approved';
            $req->audit = array_merge((array) ($req->audit ?: []), [
                'approved_by' => auth()->id(),
                'approved_at' => now()->toDateTimeString(),
                'tenant_id' => (int) $tenant->id,
                'allocation_id' => $allocation ? (int) $allocation->id : null,
            ]);
            $req->save();

            MosqueAuditUtil::log($businessId, 'approve', 'ten_request', (int) $req->id, $req->audit);
            $notify = TenancyNoticeUtil::allocationApproved($businessId, $tenant, $allocation);

            return ['success' => true, 'msg' => __('lang_v1.success'), 'whatsapp_links' => $notify['whatsapp_links'] ?? []];
        } catch (\Throwable $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }

    public function decline(Request $request, $id)
    {
        $this->ensurePermission();
        $businessId = $this->businessId();
        $this->ensureSubscriptionEnabled($businessId);

        try {
            $req = TenRequest::query()->where('business_id', $businessId)->findOrFail((int) $id);
            if ($req->status !== 'pending') {
                return ['success' => false, 'msg' => __('mosque::mosque.tenancy_err_request_not_pending')];
            }

            $req->status = 'declined';
            $req->audit = array_merge((array) ($req->audit ?: []), [
                'declined_by' => auth()->id(),
                'declined_at' => now()->toDateTimeString(),
            ]);
            $req->save();

            MosqueAuditUtil::log($businessId, 'decline', 'ten_request', (int) $req->id, $req->audit);
            return ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Throwable $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            return ['success' => false, 'msg' => __('messages.something_went_wrong')];
        }
    }
}
